package io.embrace.android.embracesdk;

import com.fernandocejas.arrow.optional.Optional;
import com.google.gson.annotations.SerializedName;

import java.util.List;
import java.util.Map;

/**
 * Represents a particular user's session within the app.
 */
public class Session {
    /**
     * Enum to discriminate the different ways a session can start / end
     */
    enum SessionLifeEventType {
        @SerializedName("s")
        STATE,
        @SerializedName("m")
        MANUAL,
        @SerializedName("t")
        TIMED
    }

    /**
     * A unique ID which identifies the session.
     */
    @SerializedName("id")
    private final String sessionId;
    /**
     * The time that the session started.
     */
    @SerializedName("st")
    private final Long startTime;
    /**
     * The time that the session ended.
     */
    @SerializedName("et")
    private final Long endTime;
    /**
     * The ordinal of the session, starting from 1.
     */
    @SerializedName("sn")
    private final Integer number;

    @SerializedName("ht")
    private final Long lastHeartbeatTime;

    @SerializedName("tt")
    private final Long terminationTime;

    @SerializedName("ls")
    private final String lastState;

    @SerializedName("ba")
    private final Double startingBatteryLevel;

    @SerializedName("ce")
    private final Boolean endedCleanly;

    @SerializedName("tr")
    private final Boolean receivedTermination;

    @SerializedName("cs")
    private final Boolean coldStart;

    @SerializedName("ss")
    private final List<String> eventIds;

    @SerializedName("t")
    private final String sessionType;

    @SerializedName("il")
    private final List<String> infoLogIds;

    @SerializedName("wl")
    private final List<String> warningLogIds;

    @SerializedName("el")
    private final List<String> errorLogIds;

    @SerializedName("lic")
    private final Integer infoLogsAttemptedToSend;

    @SerializedName("lwc")
    private final Integer warnLogsAttemptedToSend;

    @SerializedName("lec")
    private final Integer errorLogsAttemptedToSend;

    @SerializedName("e")
    private final ExceptionError exceptionError;

    @SerializedName("ri")
    private final String crashReportId;

    @SerializedName("em")
    private final SessionLifeEventType endType;

    @SerializedName("sm")
    private final SessionLifeEventType startType;

    @SerializedName("oc")
    private final List<Orientation> orientations;

    @SerializedName("sp")
    private final Map<String, String> properties;

    @SerializedName("sd")
    private final Long startupDuration;

    @SerializedName("sdt")
    private final Long startupThreshold;

    @SerializedName("si")
    private final Long sdkStartupDuration;

    private transient final UserInfo user;

    public String getSessionId() {
        return sessionId;
    }

    public UserInfo getUser() {
        return user;
    }

    public Long getStartTime() {
        return startTime;
    }

    public Long getEndTime() {
        return endTime;
    }

    public Integer getNumber() {
        return number;
    }

    public Long getLastHeartbeatTime() {
        return lastHeartbeatTime;
    }

    public Long getTerminationTime() {
        return terminationTime;
    }

    public String getLastState() {
        return lastState;
    }

    public Double getStartingBatteryLevel() {
        return startingBatteryLevel;
    }

    public Boolean isEndedCleanly() {
        return endedCleanly;
    }

    public Boolean isReceivedTermination() {
        return receivedTermination;
    }

    public Boolean isColdStart() {
        return coldStart;
    }

    public List<String> getEventIds() {
        return eventIds;
    }

    public String getSessionType() {
        return sessionType;
    }

    public List<String> getInfoLogIds() {
        return infoLogIds;
    }

    public List<String> getWarningLogIds() {
        return warningLogIds;
    }

    public List<String> getErrorLogIds() {
        return errorLogIds;
    }

    public Integer getInfoLogsAttemptedToSend() {
        return infoLogsAttemptedToSend;
    }

    public Integer getWarnLogsAttemptedToSend() {
        return warnLogsAttemptedToSend;
    }

    public Integer getErrorLogsAttemptedToSend() {
        return errorLogsAttemptedToSend;
    }

    public ExceptionError getExceptionError() {
        return exceptionError;
    }

    public String getCrashReportId() {
        return crashReportId;
    }

    public SessionLifeEventType getEndType() {
        return endType;
    }

    public SessionLifeEventType getStartType() {
        return startType;
    }

    public List<Orientation> getOrientations() {
        return orientations;
    }

    public Map<String, String> getProperties() {
        return properties;
    }

    public Long getStartupDuration() {
        return startupDuration;
    }

    public Long getStartupThreshold() {
        return startupThreshold;
    }

    public Long getSdkStartupDuration() {
        return sdkStartupDuration;
    }

    private Session(Builder builder) {
        sessionId = builder.sessionId;
        user = builder.user;
        startTime = builder.startTime;
        endTime = builder.endTime;
        number = builder.number;
        lastHeartbeatTime = builder.lastHeartbeatTime;
        terminationTime = builder.terminationTime;
        lastState = builder.lastState;
        startingBatteryLevel = builder.startingBatteryLevel;
        endedCleanly = builder.endedCleanly;
        receivedTermination = builder.receivedTermination;
        coldStart = builder.coldStart;
        eventIds = builder.eventIds;
        sessionType = builder.sessionType;
        infoLogIds = builder.infoLogIds;
        warningLogIds = builder.warningLogIds;
        errorLogIds = builder.errorLogIds;
        exceptionError = builder.exceptionError;
        infoLogsAttemptedToSend = builder.infoLogsAttemptedToSend;
        warnLogsAttemptedToSend = builder.warnLogsAttemptedToSend;
        errorLogsAttemptedToSend = builder.errorLogsAttemptedToSend;
        crashReportId = builder.crashReportId;
        startType = builder.startType;
        endType = builder.endType;
        orientations = builder.orientations;
        properties = builder.properties;
        startupDuration = builder.startupDuration;
        startupThreshold = builder.startupThreshold;
        sdkStartupDuration = builder.sdkStartupDuration;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static Builder newBuilder(Session copy) {
        Builder builder = new Builder();
        builder.sessionId = copy.getSessionId();
        builder.user = copy.getUser();
        builder.startTime = copy.getStartTime();
        builder.endTime = copy.getEndTime();
        builder.number = copy.getNumber();
        builder.lastHeartbeatTime = copy.getLastHeartbeatTime();
        builder.terminationTime = copy.getTerminationTime();
        builder.lastState = copy.getLastState();
        builder.startingBatteryLevel = copy.getStartingBatteryLevel();
        builder.endedCleanly = copy.isEndedCleanly();
        builder.receivedTermination = copy.isReceivedTermination();
        builder.coldStart = copy.isColdStart();
        builder.eventIds = copy.getEventIds();
        builder.sessionType = copy.getSessionType();
        builder.infoLogIds = copy.getInfoLogIds();
        builder.warningLogIds = copy.getWarningLogIds();
        builder.errorLogIds = copy.getErrorLogIds();
        builder.infoLogsAttemptedToSend = copy.getInfoLogsAttemptedToSend();
        builder.warnLogsAttemptedToSend = copy.getWarnLogsAttemptedToSend();
        builder.errorLogsAttemptedToSend = copy.getErrorLogsAttemptedToSend();
        builder.exceptionError = copy.getExceptionError();
        builder.crashReportId = copy.getCrashReportId();
        builder.startType = copy.getStartType();
        builder.endType = copy.getEndType();
        builder.orientations = copy.getOrientations();
        builder.properties = copy.getProperties();
        builder.startupDuration = copy.getStartupDuration();
        builder.startupThreshold = copy.getStartupThreshold();
        builder.sdkStartupDuration = copy.getSdkStartupDuration();

        return builder;
    }

    public static final class Builder {

        private String sessionId;
        private UserInfo user;
        private Long startTime;
        private Long endTime;
        private Integer number;
        private Long lastHeartbeatTime;
        private Long terminationTime;
        private String lastState;
        private Double startingBatteryLevel;
        private Boolean endedCleanly;
        private Boolean receivedTermination;
        private Boolean coldStart;
        private List<String> eventIds;
        private String sessionType;
        private List<String> infoLogIds;
        private List<String> warningLogIds;
        private List<String> errorLogIds;
        private Integer infoLogsAttemptedToSend;
        private Integer warnLogsAttemptedToSend;
        private Integer errorLogsAttemptedToSend;
        private ExceptionError exceptionError;
        private String crashReportId;
        private SessionLifeEventType endType;
        private SessionLifeEventType startType;
        private List<Orientation> orientations;
        private Map<String, String> properties;
        private Long startupDuration;
        private Long startupThreshold;
        private Long sdkStartupDuration;

        private Builder() {
        }

        public Builder withSessionId(String sessionId) {
            this.sessionId = sessionId;
            return this;
        }

        public Builder withUserInfo(Optional<UserInfo> userInfo) {
            if (userInfo.isPresent()) {
                this.user = userInfo.get();
            }
            return this;
        }

        public Builder withStartTime(Long startTime) {
            this.startTime = startTime;
            return this;
        }

        public Builder withEndTime(Long endTime) {
            this.endTime = endTime;
            return this;
        }

        public Builder withNumber(Integer number) {
            this.number = number;
            return this;
        }

        public Builder withLastHeartbeatTime(Long lastHeartbeatTime) {
            this.lastHeartbeatTime = lastHeartbeatTime;
            return this;
        }

        public Builder withTerminationTime(Long terminationTime) {
            this.terminationTime = terminationTime;
            return this;
        }

        public Builder withLastState(String lastState) {
            this.lastState = lastState;
            return this;
        }

        public Builder withStartingBatteryLevel(Optional<Float> startingBatteryLevel) {
            if (startingBatteryLevel.isPresent()) {
                this.startingBatteryLevel = Double.valueOf(startingBatteryLevel.get());
            }
            return this;
        }

        public Builder withEndedCleanly(boolean endedCleanly) {
            this.endedCleanly = endedCleanly;
            return this;
        }

        public Builder withReceivedTermination(boolean receivedTermination) {
            this.receivedTermination = receivedTermination;
            return this;
        }

        public Builder withColdStart(boolean coldStart) {
            this.coldStart = coldStart;
            return this;
        }

        public Builder withEventIds(List<String> eventIds) {
            this.eventIds = eventIds;
            return this;
        }

        public Builder withSessionType(String sessionType) {
            this.sessionType = sessionType;
            return this;
        }

        public Builder withInfoLogIds(List<String> infoLogIds) {
            this.infoLogIds = infoLogIds;
            return this;
        }

        public Builder withWarningLogIds(List<String> warningLogIds) {
            this.warningLogIds = warningLogIds;
            return this;
        }

        public Builder withErrorLogIds(List<String> errorLogIds) {
            this.errorLogIds = errorLogIds;
            return this;
        }

        public Builder withInfoLogsAttemptedToSend(Integer infoLogsAttemptedToSend) {
            this.infoLogsAttemptedToSend = infoLogsAttemptedToSend;
            return this;
        }

        public Builder withWarnLogsAttemptedToSend(Integer warnLogsAttemptedToSend) {
            this.warnLogsAttemptedToSend = warnLogsAttemptedToSend;
            return this;
        }

        public Builder withErrorLogsAttemptedToSend(Integer errorLogsAttemptedToSend) {
            this.errorLogsAttemptedToSend = errorLogsAttemptedToSend;
            return this;
        }

        public Builder withExceptionErrors(ExceptionError exceptionError) {
            this.exceptionError = exceptionError;
            return this;
        }

        public Builder withCrashReportId(String crashReportId) {
            this.crashReportId = crashReportId;
            return this;
        }

        public Builder withEndType(SessionLifeEventType endType) {
            this.endType = endType;
            return this;
        }

        public Builder withStartType(SessionLifeEventType startType) {
            this.startType = startType;
            return this;
        }

        public Builder withProperties(EmbraceSessionProperties properties) {
            this.properties = properties.get();
            return this;
        }

        public Builder withOrientations(List<Orientation> orientations) {
            if (!orientations.isEmpty()) {
                this.orientations = orientations;
            }
            return this;
        }

        public Builder withStartupDuration(long startupDuration) {
            this.startupDuration = startupDuration;
            return this;
        }

        public Builder withStartupThreshold(long startupThreshold) {
            this.startupThreshold = startupThreshold;
            return this;
        }

        public Builder withSdkStartupDuration(long sdkStartupDuration) {
            this.sdkStartupDuration = sdkStartupDuration;
            return this;
        }

        public Session build() {
            return new Session(this);
        }
    }
}
