package io.embrace.android.embracesdk;

import com.fernandocejas.arrow.checks.Preconditions;
import com.fernandocejas.arrow.optional.Optional;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Logs messages remotely, so that they can be viewed as events during a user's session.
 */
class EmbraceRemoteLogger implements MemoryCleanerListener {

    /**
     * The default limit of logs that can be send.
     */
    private static final int DEFAULT_LOG_INFO_LIMIT = 100;
    private static final int DEFAULT_LOG_WARNING_LIMIT = 100;
    private static final int DEFAULT_LOG_ERROR_LIMIT = 250;
    private static final int LOG_MESSAGE_MAXIMUM_ALLOWED_LENGTH = 128;


    private final MetadataService metadataService;

    private final ScreenshotService screenshotService;

    private final ApiClient apiClient;

    private final UserService userService;

    private final ConfigService configService;

    private final EmbraceSessionProperties sessionProperties;

    private final BackgroundWorker worker = BackgroundWorker.ofSingleThread("Remote logging");

    private final Object lock = new Object();
    private final NavigableMap<Long, String> infoLogIds = new ConcurrentSkipListMap<>();
    private final NavigableMap<Long, String> warningLogIds = new ConcurrentSkipListMap<>();
    private final NavigableMap<Long, String> errorLogIds = new ConcurrentSkipListMap<>();

    private AtomicInteger logsInfoCount = new AtomicInteger(0);
    private AtomicInteger logsErrorCount = new AtomicInteger(0);
    private AtomicInteger logsWarnCount = new AtomicInteger(0);
    private final int maxLength;

    EmbraceRemoteLogger(
            MetadataService metadataService,
            ScreenshotService screenshotService,
            ApiClient apiClient,
            UserService userService,
            ConfigService configService,
            MemoryCleanerService memoryCleanerService,
            EmbraceSessionProperties sessionProperties) {

        this.metadataService = Preconditions.checkNotNull(metadataService);
        this.screenshotService = Preconditions.checkNotNull(screenshotService);
        this.apiClient = Preconditions.checkNotNull(apiClient);
        this.userService = Preconditions.checkNotNull(userService);
        this.configService = Preconditions.checkNotNull(configService);
        Preconditions.checkNotNull(memoryCleanerService).addListener(this);
        // Session properties
        this.sessionProperties = Preconditions.checkNotNull(sessionProperties);

        if (configService.getConfig().getLogMessageMaximumAllowedLength().isPresent()) {
            maxLength = configService.getConfig().getLogMessageMaximumAllowedLength().get();
        } else {
            maxLength = LOG_MESSAGE_MAXIMUM_ALLOWED_LENGTH;
        }
    }

    /**
     * Gets the current thread's stack trace.
     *
     * @return the stack trace for the current thread or a throwable
     */
    static List<String> getWrappedStackTrace() {
        return getWrappedStackTrace(Thread.currentThread().getStackTrace());
    }

    /**
     * Gets the stack trace of the throwable.
     *
     * @return the stack trace of a throwable
     */
    static List<String> getWrappedStackTrace(StackTraceElement[] stackTraceElements) {
        List<String> augmentedStackReturnAddresses = new ArrayList<>();
        for (StackTraceElement element : stackTraceElements) {
            augmentedStackReturnAddresses.add(element.toString());
        }
        return augmentedStackReturnAddresses;
    }

    /**
     * Creates a remote log.
     *
     * @param message        the message to log
     * @param type           the type of message to log, which must be INFO_LOG, WARNING_LOG, or ERROR_LOG
     * @param takeScreenshot whether to take a screenshot when logging the event
     * @param properties     custom properties to send as part of the event
     */
    void log(String message,
             EmbraceEvent.Type type,
             boolean takeScreenshot,
             Map<String, Object> properties) {
        log(message, type, takeScreenshot, false, properties, null, null);
    }

    /**
     * Creates a remote log.
     *
     * @param message            the message to log
     * @param type               the type of message to log, which must be INFO_LOG, WARNING_LOG, or ERROR_LOG
     * @param takeScreenshot     whether to take a screenshot when logging the event
     * @param isException        whether the log is an exception
     * @param properties         custom properties to send as part of the event
     * @param stackTraceElements the stacktrace elements of a throwable
     */
    void log(String message,
             EmbraceEvent.Type type,
             boolean takeScreenshot,
             boolean isException,
             Map<String, Object> properties,
             StackTraceElement[] stackTraceElements,
             String javascriptStacktrace) {

        final long timestamp = System.currentTimeMillis();

        final Stacktraces stacktraces = new Stacktraces(
                stackTraceElements != null ? getWrappedStackTrace(stackTraceElements) : getWrappedStackTrace(),
                javascriptStacktrace);

        worker.submit(() -> {
            synchronized (lock) {
                if (configService.isLogMessageDisabled(message)) {
                    EmbraceLogger.logWarning(String.format("Log message disabled. Ignoring log with message %s", message));
                    return null;
                }

                if (configService.isMessageTypeDisabled(MessageType.LOG)) {
                    EmbraceLogger.logWarning("Log message disabled. Ignoring all Logs.");
                    return null;
                }

                String id = Uuid.getEmbUuid();
                if (type.equals(EmbraceEvent.Type.INFO_LOG)) {
                    logsInfoCount.incrementAndGet();
                    if (infoLogIds.size() < configService.getConfig().getInfoLogLimit().or(DEFAULT_LOG_INFO_LIMIT)) {
                        infoLogIds.put(timestamp, id);
                    } else {
                        EmbraceLogger.logWarning("Info Log limit has been reached.");
                        return null;
                    }
                } else if (type.equals(EmbraceEvent.Type.WARNING_LOG)) {
                    logsWarnCount.incrementAndGet();
                    if (warningLogIds.size() < configService.getConfig().getWarnLogLimit().or(DEFAULT_LOG_WARNING_LIMIT)) {
                        warningLogIds.put(timestamp, id);
                    } else {
                        EmbraceLogger.logWarning("Warning Log limit has been reached.");
                        return null;
                    }
                } else if (type.equals(EmbraceEvent.Type.ERROR_LOG)) {
                    logsErrorCount.incrementAndGet();
                    if (errorLogIds.size() < configService.getConfig().getErrorLogLimit().or(DEFAULT_LOG_ERROR_LIMIT)) {
                        errorLogIds.put(timestamp, id);
                    } else {
                        EmbraceLogger.logWarning("Error Log limit has been reached.");
                        return null;
                    }
                } else {
                    EmbraceLogger.logWarning("Unknown log level " + type.toString());
                    return null;
                }

                // Build event
                Event.Builder builder = Event.newBuilder()
                        .withType(type)
                        .withName(processLogMessage(message))
                        .withIsException(isException)
                        .withTimestamp(System.currentTimeMillis())
                        .withAppState(metadataService.getAppState())
                        .withMessageId(id)
                        .withCustomProperties(properties)
                        .withSessionProperties(sessionProperties.get());
                Optional<String> optionalSessionId = metadataService.getActiveSessionId();
                if (optionalSessionId.isPresent()) {
                    builder.withSessionId(optionalSessionId.get());
                }
                if (takeScreenshot && !configService.isScreenshotDisabledForEvent(message)) {
                    boolean screenshotTaken = screenshotService.takeScreenshotLogEvent(id);
                    builder.withScreenshotTaken(screenshotTaken);
                }
                Event event = builder.build();

                // Build event message
                EventMessage.Builder eventMessageBuilder = EventMessage.newBuilder()
                        .withEvent(event)
                        .withDeviceInfo(metadataService.getDeviceInfo())
                        .withTelephonyInfo(metadataService.getTelephonyInfo())
                        .withAppInfo(metadataService.getAppInfo())
                        .withUserInfo(userService.getUserInfo())
                        .withStacktraces(stacktraces);
                apiClient.sendLogs(eventMessageBuilder.build());
            }
            return null;
        });
    }

    /**
     * Finds all IDs of log events at info level within the given time window.
     *
     * @param startTime the beginning of the time window
     * @param endTime   the end of the time window
     * @return the list of log IDs within the specified range
     */
    List<String> findInfoLogIds(long startTime, long endTime) {
        return new ArrayList<>(this.infoLogIds.subMap(startTime, endTime).values());
    }

    /**
     * Finds all IDs of log events at warning level within the given time window.
     *
     * @param startTime the beginning of the time window
     * @param endTime   the end of the time window
     * @return the list of log IDs within the specified range
     */
    List<String> findWarningLogIds(long startTime, long endTime) {
        return new ArrayList<>(this.warningLogIds.subMap(startTime, endTime).values());
    }

    /**
     * Finds all IDs of log events at error level within the given time window.
     *
     * @param startTime the beginning of the time window
     * @param endTime   the end of the time window
     * @return the list of log IDs within the specified range
     */
    List<String> findErrorLogIds(long startTime, long endTime) {
        return new ArrayList<>(this.errorLogIds.subMap(startTime, endTime).values());
    }

    /**
     * The total number of info logs that the app attempted to send.
     */
    int getInfoLogsAttemptedToSend() {
        return logsInfoCount.get();
    }

    /**
     * The total number of warning logs that the app attempted to send.
     */
    int getWarnLogsAttemptedToSend() {
        return logsWarnCount.get();
    }

    /**
     * The total number of error logs that the app attempted to send.
     */
    int getErrorLogsAttemptedToSend() {
        return logsErrorCount.get();
    }

    private String processLogMessage(String message) {
        if (message.length() > maxLength) {
            String endChars = "...";

            // ensure that we never end up with a negative offset when extracting substring, regardless of the config value set
            int allowedLength = maxLength >= endChars.length() ? maxLength - endChars.length() : LOG_MESSAGE_MAXIMUM_ALLOWED_LENGTH - endChars.length();
            EmbraceLogger.logWarning(String.format("Truncating message to %s characters", message));
            return String.format("%s%s", message.substring(0, allowedLength), endChars);
        } else {
            return message;
        }
    }

    @Override
    public void cleanCollections() {
        this.logsInfoCount.set(0);
        this.logsWarnCount.set(0);
        this.logsErrorCount.set(0);
        this.infoLogIds.clear();
        this.warningLogIds.clear();
        this.errorLogIds.clear();
    }
}
