package io.embrace.android.embracesdk;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.BatteryManager;

import com.fernandocejas.arrow.checks.Preconditions;
import com.fernandocejas.arrow.optional.Optional;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.TimeUnit;

/**
 * Polls for the charging state, level, and other battery information. This information is stored
 * in a timeseries, and can be queried.
 * <p>
 * Since the ACTION_BATTERY_CHANGED {@link Intent} is sticky querying with a null listener returns
 * the last value immediately. We poll for the battery state using this method, rather than
 * depending on the device pushing the state, since the rate at which the updates are published
 * varies from device to device.
 * <p>
 * This class is thread safe.
 */
final class EmbracePowerService extends BroadcastReceiver implements PowerService, MemoryCleanerListener {

    private final ScheduledWorker batteryWorker = ScheduledWorker.ofSingleThread("Power Service");

    private final BackgroundWorker actionBatteryReceiverWorker = BackgroundWorker.ofSingleThread("Action Battery Receiver Worker");

    private final IntentFilter intentFilter = new IntentFilter(Intent.ACTION_BATTERY_CHANGED);

    private final NavigableMap<Long, BatteryMeasurement> measurements = new ConcurrentSkipListMap<>();

    private final NavigableMap<Long, Boolean> batteryCharging = new TreeMap<>();

    private final Context context;

    EmbracePowerService(Context context, MemoryCleanerService memoryCleanerService) {
        this.context = Preconditions.checkNotNull(context, "context must not be null");
        this.batteryWorker.scheduleAtFixedRate(() -> queryBatteryLevel(context), 0, 60, TimeUnit.SECONDS);
        registerActionBatteryReceiver();
        Preconditions.checkNotNull(memoryCleanerService).addListener(this);
    }

    @Override
    public void onReceive(Context context, Intent intent) {
        try {
            long timestamp = System.currentTimeMillis();
            addBatteryMeasurement(intent, timestamp);
            updateBatteryChargingStatus(intent, timestamp);
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to handle ACTION_BATTERY_CHANGED broadcast", ex);
        }
    }

    @Override
    public void close() {
        EmbraceLogger.logDebug("Stopping EmbracePowerService");
        context.unregisterReceiver(this);
        batteryWorker.close();
    }

    private void registerActionBatteryReceiver(){
        actionBatteryReceiverWorker.submit(() -> {
            try {
                context.registerReceiver(this, intentFilter);
            } catch (Exception ex) {
                EmbraceLogger.logDebug("Failed to register EmbracePowerService broadcast receiver. Battery charging status will be unavailable.", ex);
            }

           return null;
        });
    }

    private void addBatteryMeasurement(Intent intent, long timestamp) {
        float batteryLevel = intent.getIntExtra(BatteryManager.EXTRA_LEVEL, -1);
        float batteryMax = intent.getIntExtra(BatteryManager.EXTRA_SCALE, -1);
        if (batteryMax > 0 && batteryLevel >= 0) {
            float batteryPercentage = batteryLevel / batteryMax;
            measurements.put(timestamp, new BatteryMeasurement(timestamp, batteryPercentage));
        }
    }

    private void updateBatteryChargingStatus(Intent intent, long timestamp) {
        synchronized (this) {
            int status = intent.getIntExtra(BatteryManager.EXTRA_STATUS, -1);
            if (status > -1) {
                boolean batteryCharging = (status == BatteryManager.BATTERY_STATUS_CHARGING ||
                        status == BatteryManager.BATTERY_STATUS_FULL);
                if (this.batteryCharging.isEmpty()
                        || batteryCharging != this.batteryCharging.lastEntry().getValue()) {
                    this.batteryCharging.put(timestamp, batteryCharging);
                }
            }
        }
    }

    @Override
    public List<Interval> getChargingIntervals(long startTime, long endTime) {
        synchronized (this) {
            List<Interval> results = new ArrayList<>();
            for (Map.Entry<Long, Boolean> entry : batteryCharging.subMap(startTime, endTime).entrySet()) {
                if (entry.getValue()) {
                    long currentTime = entry.getKey();
                    Long next = batteryCharging.higherKey(currentTime);
                    results.add(new Interval(currentTime, next != null ? next : currentTime));
                }
            }
            return results;
        }
    }


    @Override
    public List<BatteryMeasurement> getBatteryMeasurements(long startTime, long endTime) {
        return new ArrayList<>(this.measurements.subMap(startTime, endTime).values());
    }

    @Override
    public Optional<Float> getLatestBatteryLevel() {
        if (measurements.isEmpty()) {
            return Optional.absent();
        }
        BatteryMeasurement latest = this.measurements.lastEntry().getValue();
        return Optional.of(latest.getValue());
    }

    private void queryBatteryLevel(Context context) {
        try {
            // Passing a null receiver retrieves the latest battery level, without subscribing
            onReceive(context, context.registerReceiver(null, intentFilter));
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to query battery level", ex);
        }
    }

    @Override
    public void cleanCollections() {
        this.measurements.clear();
        this.batteryCharging.clear();
    }
}
