package io.embrace.android.embracesdk;

import com.fernandocejas.arrow.checks.Preconditions;

final class EmbracePerformanceInfoService implements PerformanceInfoService {

    private final AnrService anrService;

    private final NetworkConnectivityService networkConnectivityService;

    private final NetworkLoggingService networkLoggingService;

    private final PowerService powerService;

    private final MemoryService memoryService;

    private final SignalQualityService signalQualityService;

    private final ConnectionClassService connectionClassService;

    private final MetadataService metadataService;

    EmbracePerformanceInfoService(
            AnrService anrService,
            NetworkConnectivityService networkConnectivityService,
            NetworkLoggingService networkLoggingService,
            PowerService powerService,
            MemoryService memoryService,
            SignalQualityService signalQualityService,
            ConnectionClassService connectionClassService,
            MetadataService metadataService) {

        this.anrService = Preconditions.checkNotNull(anrService);
        this.networkConnectivityService = Preconditions.checkNotNull(networkConnectivityService);
        this.networkLoggingService = Preconditions.checkNotNull(networkLoggingService);
        this.powerService = Preconditions.checkNotNull(powerService);
        this.memoryService = Preconditions.checkNotNull(memoryService);
        this.signalQualityService = Preconditions.checkNotNull(signalQualityService);
        this.connectionClassService = Preconditions.checkNotNull(connectionClassService);
        this.metadataService = Preconditions.checkNotNull(metadataService);
    }


    @Override
    public SessionPerformanceInfo getSessionPerformanceInfo(long startTime, long lastKnownTime) {
        SessionPerformanceInfo.Builder builder = SessionPerformanceInfo.newBuilder()
                .withNetworkSessionV2(networkLoggingService.getNetworkSession(startTime, lastKnownTime));
        builder.withAnrIntervals(anrService.getAnrIntervals(startTime, lastKnownTime));
        addPerformanceInfo(builder, startTime, lastKnownTime);
        return builder.build();
    }

    @Override
    public EventPerformanceInfo getPerformanceInfo(long startTime, long endTime) {
        EventPerformanceInfo.Builder builder = EventPerformanceInfo.newBuilder()
                .withNetworkCalls(networkLoggingService.getNetworkCalls(startTime, endTime));
        addPerformanceInfo(builder, startTime, endTime);
        return builder.build();
    }

    private <T extends PerformanceInfo.Builder> void addPerformanceInfo(T builder, long startTime, long endTime) {
        builder.withBatteryMeasurements(powerService.getBatteryMeasurements(startTime, endTime))
                .withChargingIntervals(powerService.getChargingIntervals(startTime, endTime))
                .withDiskUsage(metadataService.getDiskUsage())
                .withMemorySamples(memoryService.getMemorySamples(startTime, endTime))
                .withMemoryWarnings(memoryService.getMemoryWarnings(startTime, endTime))
                .withNetworkInterfaceIntervals(networkConnectivityService.getNetworkInterfaceIntervals(startTime, endTime))
                .withSignalStrength(signalQualityService.getSignalMeasurements(startTime, endTime))
                .withConnectionQualityIntervals(connectionClassService.getQualityIntervals(startTime, endTime));
        if (powerService.getLatestBatteryLevel().isPresent()) {
            builder.withBatteryLevel(powerService.getLatestBatteryLevel().get());
        }
    }
}
