package io.embrace.android.embracesdk;

import com.fernandocejas.arrow.checks.Preconditions;
import com.fernandocejas.arrow.optional.Optional;

/**
 * Intercepts uncaught Java exceptions and forwards them to the Embrace API.
 */
final class EmbraceCrashService implements CrashService {

    private static final String CRASH_REPORT_EVENT_NAME = "_crash_report";

    private final LocalConfig localConfig;

    private final SessionService sessionService;

    private final MetadataService metadataService;

    private final ApiClient apiClient;

    private final UserService userService;

    private final EventService eventService;

    private final AnrService anrService;

    private final NdkService ndkService;

    private boolean mainCrashHandled = false;

    private JsException jsException;

    EmbraceCrashService(
            LocalConfig localConfig,
            SessionService sessionService,
            MetadataService metadataService,
            ApiClient apiClient,
            UserService userService,
            EventService eventService,
            AnrService anrService,
            NdkService ndkService) {

        this.localConfig = Preconditions.checkNotNull(localConfig);
        if (this.localConfig.getConfigurations().getCrashHandler().getEnabled()) {
            registerExceptionHandler();
        }

        this.sessionService = Preconditions.checkNotNull(sessionService);
        this.metadataService = Preconditions.checkNotNull(metadataService);
        this.apiClient = Preconditions.checkNotNull(apiClient);
        this.userService = Preconditions.checkNotNull(userService);
        this.eventService = Preconditions.checkNotNull(eventService);
        this.anrService = Preconditions.checkNotNull(anrService);
        this.ndkService = Preconditions.checkNotNull(ndkService);
    }

    /**
     * Handles a crash caught by the {@link EmbraceUncaughtExceptionHandler} by constructing a
     * JSON message containing a description of the crash, device, and context, and then sending
     * it to the Embrace API.
     *
     * @param thread    the crashing thread
     * @param exception the exception thrown by the thread
     */
    @Override
    public void handleCrash(Thread thread, Throwable exception) {
        if (!mainCrashHandled) {
            mainCrashHandled = true;

            // Stop ANR tracking first to avoid capture ANR when crash message is being sent
            anrService.forceANRTrackingStopOnCrash();

            Crash crash;

            // Check if the unity crash id exists. If so, means that the native crash capture
            // is enabled for an Unity build. When a native crash occurs and the NDK sends an
            // uncaught exception the SDK assign the unity crash id as the java crash id.
            String unityCrashId = this.ndkService.getUnityCrashId();
            if (unityCrashId != null) {
                crash = Crash.ofThrowable(exception, jsException, unityCrashId);
            } else {
                crash = Crash.ofThrowable(exception, jsException);
            }


            Event.Builder builder = Event.newBuilder()
                    .withName(CRASH_REPORT_EVENT_NAME)
                    .withType(EmbraceEvent.Type.CRASH)
                    .withTimestamp(System.currentTimeMillis())
                    .withAppState(metadataService.getAppState())
                    .withActiveEventIds(eventService.getActiveEventIds())
                    .withSessionProperties(sessionService.getProperties());

            Optional<String> optionalSessionId = metadataService.getActiveSessionId();
            if (optionalSessionId.isPresent()) {
                builder.withSessionId(optionalSessionId.get());
            }

            EventMessage.Builder versionedEventBuilder = EventMessage.newBuilder()
                    .withAppInfo(metadataService.getAppInfo())
                    .withDeviceInfo(metadataService.getDeviceInfo())
                    .withUserInfo(userService.getUserInfo())
                    .withCrash(crash)
                    .withEvent(builder.build());
            try {
                apiClient.sendEvent(versionedEventBuilder.build()).get();
            } catch (Exception ex) {
                EmbraceLogger.logDebug("Failed to report crash to the api", ex);
            }
            sessionService.handleCrash(crash.getCrashId());
        } else {
            // Disregarded for now.
        }
    }

    /**
     * Registers the Embrace {@link java.lang.Thread.UncaughtExceptionHandler} to intercept uncaught
     * exceptions and forward them to the Embrace API as crashes.
     */
    private void registerExceptionHandler() {
        Thread.UncaughtExceptionHandler defaultHandler = Thread.getDefaultUncaughtExceptionHandler();
        EmbraceUncaughtExceptionHandler embraceHandler =
                new EmbraceUncaughtExceptionHandler(defaultHandler, this);
        Thread.setDefaultUncaughtExceptionHandler(embraceHandler);
    }

    /**
     * Associates an unhandled JS exception with a crash
     *
     * @param jsException the unhandled JS exception
     */
    @Override
    public void logUnhandledJsException(JsException jsException) {
        this.jsException = jsException;
    }
}
