package io.embrace.android.embracesdk;

import android.app.Activity;
import android.text.TextUtils;
import android.util.Pair;

import com.fernandocejas.arrow.checks.Preconditions;
import com.fernandocejas.arrow.optional.Optional;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Deque;
import java.util.List;
import java.util.concurrent.LinkedBlockingDeque;

import io.embrace.android.embracesdk.TapBreadcrumb.TapBreadcrumbType;
import java9.util.stream.Collectors;
import java9.util.stream.StreamSupport;

/**
 * Handles the logging of breadcrumbs.
 * <p>
 * Breadcrumbs record a user's journey through the app and are split into:
 * <ul>
 * <li>View breadcrumbs: Each time the user changes view in the app</li>
 * <li>Tap breadcrumbs: Each time the user taps a UI element in the app</li>
 * <li>Custom breadcrumbs: User-defined interactions within the app</li>
 * </ul>
 * Breadcrumbs are limited at query-time by default to 100 per session, but this can be overridden
 * in server-side configuration. They are stored in an unbounded queue.
 */
class EmbraceBreadcrumbService implements BreadcrumbService, ActivityListener, MemoryCleanerListener {

    private static final String QUERY_PARAMETER_DELIMITER = "?";

    /**
     * Clock used by the service
     */
    private final Clock clock;

    /**
     * The default limit for how many open tracked fragments are allowed, which can be overridden
     * by {@link Config}.
     */
    private static final Integer DEFAULT_FRAGMENT_STACK_SIZE = 20;

    /**
     * The config service, for retrieving the breadcrumb limit.
     */
    private final ConfigService configService;

    /**
     * Local config for controlling whether tap coordinates are omitted
     */
    private final LocalConfig localConfig;

    /**
     * A deque of breadcrumbs.
     */
    private final LinkedBlockingDeque<ViewBreadcrumb> viewBreadcrumbs = new LinkedBlockingDeque<>();

    private final LinkedBlockingDeque<TapBreadcrumb> tapBreadcrumbs = new LinkedBlockingDeque<>();

    private final LinkedBlockingDeque<CustomBreadcrumb> customBreadcrumbs = new LinkedBlockingDeque<>();

    private final LinkedBlockingDeque<WebViewBreadcrumb> webViewBreadcrumbs = new LinkedBlockingDeque<>();

    final LinkedBlockingDeque<FragmentBreadcrumb> fragmentBreadcrumbs = new LinkedBlockingDeque<>();

    final List<FragmentBreadcrumb> fragmentStack = Collections.synchronizedList(new ArrayList<>());

    EmbraceBreadcrumbService(Clock clock,
                             ConfigService configService,
                             LocalConfig localConfig,
                             ActivityService activityService,
                             MemoryCleanerService memoryCleanerService) {

        this.clock = clock;
        this.configService = Preconditions.checkNotNull(
                configService,
                "configService must not be null");
        this.localConfig = Preconditions.checkNotNull(
                localConfig,
                "localConfig must not be null");
        Preconditions.checkNotNull(activityService, "activityService must not be null");
        activityService.addListener(this);
        Preconditions.checkNotNull(memoryCleanerService).addListener(this);
    }

    @Override
    public void logView(String screen, long timestamp) {
        addToViewLogsQueue(screen, timestamp, false);

    }

    @Override
    public void forceLogView(String screen, long timestamp) {
        addToViewLogsQueue(screen, timestamp, true);
    }

    @Override
    public boolean startFragment(String name) {
        synchronized (this) {
            if (fragmentStack.size() >= DEFAULT_FRAGMENT_STACK_SIZE) {
                return false;
            }
            return fragmentStack.add(new FragmentBreadcrumb(name, clock.now(), 0));
        }
    }

    @Override
    public boolean endFragment(String name) {
        FragmentBreadcrumb start;
        FragmentBreadcrumb end = new FragmentBreadcrumb(name, 0, clock.now());

        synchronized (this) {
            int index = fragmentStack.lastIndexOf(end);
            if (index < 0) {
                return false;
            }
            start = fragmentStack.remove(index);
        }
        end.setStartTime(start.getStartTime());

        tryAddBreadcrumb(this.fragmentBreadcrumbs, end);

        return true;
    }

    @Override
    public void logTap(Pair<Float, Float> point, String element, long timestamp, TapBreadcrumbType type) {
        try {
            if (!localConfig.getConfigurations().getTaps().getCaptureCoordinates()) {
                point = new Pair<>(0.0F, 0.0F);
            }

            tryAddBreadcrumb(this.tapBreadcrumbs, new TapBreadcrumb(point, element, timestamp, type));
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to log tap breadcrumb for element " + element);
        }
    }

    @Override
    public void logCustom(String message, long timestamp) {
        if (TextUtils.isEmpty(message)) {
            EmbraceLogger.logWarning("Breadcrumb message must not be blank");
            return;
        }
        try {

            tryAddBreadcrumb(this.customBreadcrumbs, new CustomBreadcrumb(message, timestamp));
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to log custom breadcrumb with message " + message, ex);
        }
    }

    @Override
    public void logWebView(String url, long startTime) {

        if (!localConfig.getConfigurations().getWebViewConfig().isWebViewsCaptureEnabled()) {
            return;
        }

        if (url == null) {
            return;
        }

        try {
            // Check if web view query params should be captured.
            String parsedUrl = url;
            if (!localConfig.getConfigurations().getWebViewConfig().isQueryParamsCaptureEnabled()) {
                int queryOffset = url.indexOf(QUERY_PARAMETER_DELIMITER);

                if (queryOffset > 0) {
                    parsedUrl = url.substring(0, queryOffset);
                }
            }

            tryAddBreadcrumb(this.webViewBreadcrumbs, new WebViewBreadcrumb(parsedUrl, startTime));
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to log WebView breadcrumb for url " + url);
        }
    }

    @Override
    public List<ViewBreadcrumb> getViewBreadcrumbsForSession(long startTime, long endTime) {
        return filterBreadcrumbsForTimeWindow(
                this.viewBreadcrumbs,
                startTime,
                endTime);
    }

    @Override
    public List<TapBreadcrumb> getTapBreadcrumbsForSession(long startTime, long endTime) {
        return filterBreadcrumbsForTimeWindow(
                this.tapBreadcrumbs,
                startTime,
                endTime);
    }

    @Override
    public List<CustomBreadcrumb> getCustomBreadcrumbsForSession(long startTime, long endTime) {
        return filterBreadcrumbsForTimeWindow(
                this.customBreadcrumbs,
                startTime,
                endTime);
    }

    @Override
    public List<WebViewBreadcrumb> getWebViewBreadcrumbsForSession(long startTime, long endTime) {
        return filterBreadcrumbsForTimeWindow(
                this.webViewBreadcrumbs,
                startTime,
                endTime);
    }

    @Override
    public List<FragmentBreadcrumb> getFragmentBreadcrumbsForSession(long startTime, long endTime) {
        return filterBreadcrumbsForTimeWindow(
                this.fragmentBreadcrumbs,
                startTime,
                endTime);
    }

    @Override
    public Breadcrumbs getBreadcrumbs(long start, long end) {
        return Breadcrumbs.newBuilder()
                .withCustomBreadcrumbs(getCustomBreadcrumbsForSession(start, end))
                .withTapBreadcrumbs(getTapBreadcrumbsForSession(start, end))
                .withViewBreadcrumbs(getViewBreadcrumbsForSession(start, end))
                .withWebViewBreadcrumbs(getWebViewBreadcrumbsForSession(start, end))
                .withFragmentBreadcrumbs(getFragmentBreadcrumbsForSession(start, end))
                .build();
    }

    @Override
    public Optional<String> getLastViewBreadcrumbScreenName() {
        if (this.viewBreadcrumbs.isEmpty()) {
            return Optional.absent();
        } else {
            return this.viewBreadcrumbs.peek().getScreen();
        }
    }

    @Override
    public void onView(Activity activity) {
        logView(activity.getClass().getName(), clock.now());
    }

    /**
     * Close all open fragments when the activity closes
     */
    @Override
    public void onViewClose(Activity activity) {
        try {
            ViewBreadcrumb lastViewBreadcrumb = this.viewBreadcrumbs.peek();
            if (lastViewBreadcrumb != null) {
                lastViewBreadcrumb.setEnd(clock.now());
            }
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to add set end time for breadcrumb", ex);
        }

        if (fragmentStack.size() == 0) {
            return;
        }

        long ts = clock.now();
        synchronized (fragmentStack) {
            for (FragmentBreadcrumb fragment : fragmentStack) {
                fragment.setEndTime(ts);
                tryAddBreadcrumb(fragmentBreadcrumbs, fragment);
            }
            fragmentStack.clear();
        }
    }

    @Override
    public void cleanCollections() {
        this.viewBreadcrumbs.clear();
        this.tapBreadcrumbs.clear();
        this.customBreadcrumbs.clear();
        this.webViewBreadcrumbs.clear();
        this.fragmentBreadcrumbs.clear();
        this.fragmentStack.clear();
    }

    /**
     * Adds the view breadcrumb to the queue.
     *
     * @param screen    name of the screen.
     * @param timestamp time of occurrence of the tap event.
     * @param force     will run no duplication checks on the previous view breadcrumb registry.
     */
    private synchronized void addToViewLogsQueue(String screen, long timestamp, boolean force) {
        try {
            ViewBreadcrumb lastViewBreadcrumb = this.viewBreadcrumbs.peek();
            if (force || lastViewBreadcrumb == null
                    || !lastViewBreadcrumb.getScreen().or("").equalsIgnoreCase(String.valueOf(screen))) {
                // TODO: is `lastViewBreadcrumb` a copy or the actual object in the queue?
                if (lastViewBreadcrumb != null) {
                    lastViewBreadcrumb.setEnd(timestamp);
                }

                tryAddBreadcrumb(this.viewBreadcrumbs, new ViewBreadcrumb(screen, timestamp));
            }
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to add view breadcrumb for " + screen, ex);
        }
    }

    /**
     * Returns the latest breadcrumbs within the specified interval, up to the maximum queue size or
     * configured limit in the app configuration.
     *
     * @param breadcrumbs breadcrumbs list to filter.
     * @param startTime   beginning of the time window.
     * @param endTime     end of the time window.
     * @return filtered breadcrumbs from the provided FixedSizeDeque.
     */
    private <T extends Breadcrumb> List<T> filterBreadcrumbsForTimeWindow(
            Deque<T> breadcrumbs,
            long startTime,
            long endTime) {

        return StreamSupport.stream(breadcrumbs)
                .filter(breadcrumb -> breadcrumb.getStartTime() >= startTime)
                .filter(breadcrumb -> endTime <= 0L || breadcrumb.getStartTime() <= endTime)
                .collect(Collectors.toList());
    }

    private <T> void tryAddBreadcrumb(LinkedBlockingDeque<T> breadcrumbs, T breadcrumb) {
        int limit = configService.getConfig().getCustomBreadcrumbLimit();

        if (!breadcrumbs.isEmpty() && breadcrumbs.size() == limit) {
            breadcrumbs.removeLast();
        }

        breadcrumbs.push(breadcrumb);
    }
}

