package io.embrace.android.embracesdk;

import java.io.Closeable;
import java.util.List;
import java.util.Map;

/**
 * Provides event lifecycle management for the SDK.
 * <p>
 * A start event is submitted, followed by an end event, and then the duration is timed. These
 * events appear on the session timeline.
 * <p>
 * A story ID refers to the UUID for a particular event. An event ID is the concatenation of the
 * user-supplied event name, and the event identifier.
 */
interface EventService extends Closeable {
    /**
     * Starts an event.
     * <p>
     * A screenshot is taken if the event is late, and screenshots are not disabled for this event
     * as part of the {@link Config}.
     *
     * @param name the name of the event
     */
    void startEvent(String name);

    /**
     * Starts an event.
     * <p>
     * A screenshot is taken if the event is late, and screenshots are not disabled for this event
     * as part of the {@link Config}.
     *
     * @param name       the name of the event
     * @param identifier the identifier, to uniquely distinguish between events with the same name
     */
    void startEvent(String name, String identifier);

    /**
     * Starts an event.
     * <p>
     * A screenshot is optionally taken if the event is late, and screenshots are not disabled
     * for this event as part of the {@link Config}, or as part of this method call.
     *
     * @param name            the name of the event
     * @param identifier      the identifier, to uniquely distinguish between events with the same name
     * @param allowScreenshot whether a screenshot should be taken for a late event
     */
    void startEvent(String name, String identifier, boolean allowScreenshot);

    /**
     * Starts an event.
     * <p>
     * A screenshot is optionally taken if the event is late, and screenshots are not disabled
     * for this event as part of the {@link Config}, or as part of this method call.
     *
     * @param name       the name of the event
     * @param identifier the identifier, to uniquely distinguish between events with the same name
     * @param properties custom properties which can be sent as part of the request
     */
    void startEvent(String name, String identifier, Map<String, Object> properties);

    /**
     * Starts an event.
     * <p>
     * A screenshot is optionally taken if the event is late, and screenshots are not disabled
     * for this event as part of the {@link Config}, or as part of this method call.
     *
     * @param name            the name of the event
     * @param identifier      the identifier, to uniquely distinguish between events with the same name
     * @param allowScreenshot whether a screenshot should be taken for a late event
     * @param properties      custom properties which can be sent as part of the request
     */
    void startEvent(String name, String identifier, boolean allowScreenshot, Map<String, Object> properties);

    /**
     * Starts an event.
     * <p>
     * A screenshot is optionally taken if the event is late, and screenshots are not disabled
     * for this event as part of the {@link Config}, or as part of this method call.
     *
     * @param name            the name of the event
     * @param identifier      the identifier, to uniquely distinguish between events with the same name
     * @param allowScreenshot whether a screenshot should be taken for a late event
     * @param properties      custom properties which can be sent as part of the request
     * @param startTime       a back-dated time at which the event occurred
     */
    void startEvent(
            String name,
            String identifier,
            boolean allowScreenshot,
            Map<String, Object> properties,
            Long startTime);

    /**
     * Ends an event which matches the given name.
     *
     * @param name the name of the event to terminate
     */
    void endEvent(String name);

    /**
     * Ends an event which matches the given name and identifier.
     *
     * @param name       the name of the event to terminate
     * @param identifier the unique identifier of the event to terminate
     */
    void endEvent(String name, String identifier);

    /**
     * Ends an event which matches the given name and identifier.
     *
     * @param name       the name of the event to terminate
     * @param properties custom properties which can be sent as part of the moment
     */
    void endEvent(String name, Map<String, Object> properties);

    /**
     * Ends an event which matches the given name and identifier.
     *
     * @param name       the name of the event to terminate
     * @param identifier the unique identifier of the event to terminate
     * @param properties custom properties which can be sent as part of the moment
     */
    void endEvent(String name, String identifier, Map<String, Object> properties);

    /**
     * Finds all event IDs (event UUIDs) within the given time window.
     *
     * @param startTime the start time of the window to search
     * @param endTime   the end time of the window to search
     * @return the list of story IDs within the specified range
     */
    List<String> findEventIdsForSession(long startTime, long endTime);

    /**
     * Gets all of the IDs of the currently active moments.
     *
     * @return list of IDs for the currently active moments
     */
    List<String> getActiveEventIds();

    /**
     * Get startup duration and startup threshold info of the cold start session.
     *
     * @return the startup moment info
     */
    StartupEventInfo getStartupMomentInfo();
}
