package io.embrace.android.embracesdk;

import android.content.Context;
import android.content.res.Resources;
import android.util.Base64;

import com.fernandocejas.arrow.optional.Optional;
import com.google.gson.annotations.SerializedName;
import com.jsoniter.JsonIterator;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import java9.util.stream.StreamSupport;

public final class LocalConfig {

    private static final String APP_CONFIG_KEY = "app";
    private static final String APP_DISK_USAGE_CONFIG_KEY = "report_disk_usage";
    private static final String BASE_URL_KEY = "base_urls";
    private static final String BASE_URL_DATA_CONFIG_KEY = "data";
    private static final String BASE_URL_DATA_DEV_CONFIG_KEY = "data_dev";
    private static final String BASE_URL_CONFIG_KEY = "config";
    private static final String BASE_URL_IMAGES_CONFIG_KEY = "images";
    private static final String CRASH_HANDLER_CONFIG_KEY = "crash_handler";
    private static final String CRASH_HANDLER_ENABLED_CONFIG_KEY = "enabled";
    private static final String STARTUP_MOMENT_CONFIG_KEY = "startup_moment";
    private static final String STARTUP_MOMENT_AUTOMATICALLY_END_CONFIG_KEY = "automatically_end";
    private static final String STARTUP_MOMENT_TAKE_SCREENSHOT_CONFIG_KEY = "take_screenshot";
    private static final String NETWORKING_CONFIG_KEY = "networking";
    private static final String NETWORKING_CAPTURE_REQUEST_LENGTH_CONFIG_KEY = "capture_request_content_length";
    private static final String NETWORKING_DEFAULT_CAPTURE_LIMIT_CONFIG_KEY = "default_capture_limit";
    private static final String NETWORKING_DOMAINS_CONFIG_KEY = "domains";
    private static final String NETWORKING_DOMAIN_NAME_CONFIG_KEY = "domain_name";
    private static final String NETWORKING_DOMAIN_LIMIT_CONFIG_KEY = "domain_limit";
    private static final String NETWORKING_TRACE_ID_CONFIG_KEY = "trace_id_header";
    private static final String NETWORKING_DISABLE_URL_PATTERNS_CONFIG_KEY = "disabled_url_patterns";
    private static final String NETWORKING_ENABLE_MONITORING_CONFIG_KEY = "enable_native_monitoring";
    private static final String SESSION_CONFIG_KEY = "session";
    private static final String SESSION_ASYNC_END_CONFIG_KEY = "async_end";
    private static final String SESSION_MAX_SESSION_SEC_CONFIG_KEY = "max_session_seconds";
    private static final String TAPS_CONFIG_KEY = "taps";
    private static final String TAPS_CAPTURE_COORDINATES_CONFIG_KEY = "capture_coordinates";
    private static final String WEBVIEW_CONFIG_KEY = "webview";
    private static final String WEBVIEW_CAPTURE_QUERY_PARAMS_CONFIG_KEY = "capture_query_params";
    private static final String WEBVIEW_ENABLE_CAPTURE_CONFIG_KEY = "enable";

    /**
     * Build info app id name.
     */
    static final String BUILD_INFO_APP_ID = "emb_app_id";

    /**
     * Build info sdk config id name.
     */
    static final String BUILD_INFO_SDK_CONFIG = "emb_sdk_config";

    /**
     * Build info ndk enabled.
     */
    private static final String BUILD_INFO_NDK_ENABLED = "emb_ndk_enabled";

    /**
     * The default value for native crash capture enabling
     */
    private static final Boolean NDK_ENABLED_DEFAULT = false;

    /**
     * The Embrace app ID. This is used to identify the app within the database.
     */
    private final String appId;

    /**
     * The Embrace sdk configurations. This is used to setup configurations.
     */
    private final SdkConfigs configs;

    /**
     * Control whether the Embrace SDK is able to capture native crashes.
     */
    @SerializedName("ndk_enabled")
    private Boolean ndkEnabled;

    LocalConfig(String appId, Boolean ndkEnabled, SdkConfigs configs) {
        this.appId = appId;
        this.ndkEnabled = ndkEnabled;
        this.configs = Optional.fromNullable(configs).orNull();
    }

    /**
     * Loads the build information from resources provided by the config file packaged within the application by Gradle at
     * build-time.
     *
     * @return the local configuration
     */

    static LocalConfig fromResources(Context context) {
        try {

            String appId = context.getResources().getString(getResourcesIdentifier(context, BUILD_INFO_APP_ID, "string"));

            boolean ndkEnabled = NDK_ENABLED_DEFAULT;
            int ndkEnabledJsonId = getResourcesIdentifier(context, BUILD_INFO_NDK_ENABLED, "string");
            if (ndkEnabledJsonId != 0) {
                ndkEnabled = Boolean.parseBoolean(context.getResources().getString(ndkEnabledJsonId));
            }

            String sdkConfigJson = null;
            int sdkConfigJsonId = getResourcesIdentifier(context, BUILD_INFO_SDK_CONFIG, "string");
            if (sdkConfigJsonId != 0) {
                String encodedConfig = context.getResources().getString(sdkConfigJsonId);
                //Decode sdkConfig
                sdkConfigJson = new String(Base64.decode(encodedConfig, Base64.DEFAULT));
            }

            return buildConfig(appId, ndkEnabled, sdkConfigJson);
        } catch (Exception ex) {
            throw new RuntimeException("Failed to load local config from resources.", ex);
        }
    }

    static LocalConfig buildConfig(String appId, boolean ndkEnabled, String sdkConfigs) {
        if (appId == null || appId.isEmpty()) {
            throw new IllegalArgumentException("Embrace AppId cannot be null or empty.");
        }

        EmbraceLogger.logInfo(String.format("Native crash capture is %s", ndkEnabled ? "enabled" : "disabled"));

        SdkConfigs configs = new SdkConfigs();
        if (sdkConfigs != null && !sdkConfigs.isEmpty()) {
            JsonIterator iter = JsonIterator.parse(sdkConfigs);

            try {
                for (String field = iter.readObject(); field != null; field = iter.readObject()) {
                    switch (field) {
                        case APP_CONFIG_KEY:
                            configs.app = new SdkConfigs.App();
                            for (String appField = iter.readObject(); appField != null; appField = iter.readObject()) {
                                if (APP_DISK_USAGE_CONFIG_KEY.equals(appField)) {
                                    configs.app.reportDiskUsage = iter.readBoolean();
                                } else {
                                    iter.skip();
                                }
                            }
                            break;
                        case BASE_URL_KEY:
                            configs.baseUrls = new SdkConfigs.BaseUrls();
                            for (String baseUrlField = iter.readObject(); baseUrlField != null; baseUrlField = iter.readObject()) {
                                switch (baseUrlField) {
                                    case BASE_URL_DATA_CONFIG_KEY:
                                        configs.baseUrls.data = iter.readString();
                                        break;
                                    case BASE_URL_DATA_DEV_CONFIG_KEY:
                                        configs.baseUrls.dataDev = iter.readString();
                                        break;
                                    case BASE_URL_CONFIG_KEY:
                                        configs.baseUrls.config = iter.readString();
                                        break;
                                    case BASE_URL_IMAGES_CONFIG_KEY:
                                        configs.baseUrls.images = iter.readString();
                                        break;
                                    default:
                                        iter.skip();
                                }
                            }
                            break;
                        case CRASH_HANDLER_CONFIG_KEY:
                            configs.crashHandler = new SdkConfigs.CrashHandler();
                            for (String crashHandlerField = iter.readObject(); crashHandlerField != null; crashHandlerField = iter.readObject()) {
                                if (CRASH_HANDLER_ENABLED_CONFIG_KEY.equals(crashHandlerField)) {
                                    configs.crashHandler.enabled = iter.readBoolean();
                                } else {
                                    iter.skip();
                                }
                            }
                            break;
                        case STARTUP_MOMENT_CONFIG_KEY:
                            configs.startupMoment = new SdkConfigs.StartupMoment();
                            for (String startupMomentField = iter.readObject(); startupMomentField != null; startupMomentField = iter.readObject()) {
                                switch (startupMomentField) {
                                    case STARTUP_MOMENT_AUTOMATICALLY_END_CONFIG_KEY:
                                        configs.startupMoment.automaticallyEnd = iter.readBoolean();
                                        break;
                                    case STARTUP_MOMENT_TAKE_SCREENSHOT_CONFIG_KEY:
                                        configs.startupMoment.takeScreenshot = iter.readBoolean();
                                        break;
                                    default:
                                        iter.skip();
                                }
                            }
                            break;
                        case NETWORKING_CONFIG_KEY:
                            configs.networking = new SdkConfigs.Networking();
                            for (String networkingField = iter.readObject(); networkingField != null; networkingField = iter.readObject()) {
                                switch (networkingField) {
                                    case NETWORKING_CAPTURE_REQUEST_LENGTH_CONFIG_KEY:
                                        configs.networking.captureRequestContentLength = iter.readBoolean();
                                        break;
                                    case NETWORKING_DEFAULT_CAPTURE_LIMIT_CONFIG_KEY:
                                        if (!iter.readNull()) {
                                            configs.networking.defaultCaptureLimit = iter.readInt();
                                        }
                                        break;
                                    case NETWORKING_DOMAINS_CONFIG_KEY:
                                        List<SdkConfigs.Networking.Domain> domains = new ArrayList<>();
                                        iter.readArrayCB((domainObject, attachment) -> {
                                            String obj = domainObject.readObject();
                                            for (String domainField = obj; domainField != null; domainField = domainObject.readObject()) {
                                                SdkConfigs.Networking.Domain domain = new SdkConfigs.Networking.Domain();
                                                switch (domainField) {
                                                    case NETWORKING_DOMAIN_NAME_CONFIG_KEY:
                                                        domain.domain = domainObject.readString();
                                                        break;
                                                    case NETWORKING_DOMAIN_LIMIT_CONFIG_KEY:
                                                        if (!iter.readNull()) {
                                                            domain.limit = domainObject.readInt();
                                                        }
                                                        break;
                                                    default:
                                                        domainObject.skip();
                                                }

                                                if (domain.getDomain() != null) {
                                                    domains.add(domain);
                                                }
                                            }

                                            return true;
                                        }, String.class);

                                        configs.networking.domains = domains;
                                        break;
                                    case NETWORKING_TRACE_ID_CONFIG_KEY:
                                        configs.networking.traceIdHeader = iter.readString();
                                        break;
                                    case NETWORKING_DISABLE_URL_PATTERNS_CONFIG_KEY:
                                        List<String> disabledUrls = new ArrayList<>();
                                        iter.readArrayCB((patter, attachment) -> {
                                            String urlPattern = patter.readString();

                                            if (urlPattern != null) {
                                                disabledUrls.add(urlPattern);
                                                return true;
                                            }

                                            return false;
                                        }, String.class);

                                        configs.networking.disabledUrlPatterns = disabledUrls;
                                        break;
                                    case NETWORKING_ENABLE_MONITORING_CONFIG_KEY:
                                        configs.networking.enableNativeMonitoring = iter.readBoolean();
                                        break;
                                    default:
                                        iter.skip();
                                }
                            }
                            break;
                        case SESSION_CONFIG_KEY:
                            configs.sessionConfig = new SdkConfigs.SessionConfig();
                            for (String sessionConfigField = iter.readObject(); sessionConfigField != null; sessionConfigField = iter.readObject()) {
                                switch (sessionConfigField) {
                                    case SESSION_ASYNC_END_CONFIG_KEY:
                                        configs.sessionConfig.asyncEnd = iter.readBoolean();
                                        break;
                                    case SESSION_MAX_SESSION_SEC_CONFIG_KEY:
                                        if (!iter.readNull()) {
                                            configs.sessionConfig.maxSessionSeconds = iter.readInt();
                                        }
                                        break;
                                    default:
                                        iter.skip();
                                }
                            }
                            break;
                        case TAPS_CONFIG_KEY: {
                            configs.taps = new SdkConfigs.Taps();
                            for (String tapsField = iter.readObject(); tapsField != null; tapsField = iter.readObject()) {
                                if (TAPS_CAPTURE_COORDINATES_CONFIG_KEY.equals(tapsField)) {
                                    configs.taps.captureCoordinates = iter.readBoolean();
                                } else {
                                    iter.skip();
                                }
                            }
                        }
                        break;
                        case WEBVIEW_CONFIG_KEY:
                            configs.webView = new SdkConfigs.WebView();
                            for (String webviewField = iter.readObject(); webviewField != null; webviewField = iter.readObject()) {
                                switch (webviewField) {
                                    case WEBVIEW_CAPTURE_QUERY_PARAMS_CONFIG_KEY:
                                        configs.webView.captureQueryParams = iter.readBoolean();
                                        break;
                                    case WEBVIEW_ENABLE_CAPTURE_CONFIG_KEY:
                                        configs.webView.captureWebViews = iter.readBoolean();
                                        break;
                                    default:
                                        iter.skip();
                                }
                            }
                            break;
                        default:
                            iter.skip();
                    }
                }
            } catch (IOException ex) {
                EmbraceLogger.logError("Failed to parse Embrace config from config json file.", ex);
            }

        } else {
            configs = new SdkConfigs();
        }

        List<String> nonDefaultValues = configs.getNonDefaultValues();
        for (String nonDefaultValue : nonDefaultValues) {
            EmbraceLogger.logInfo("Custom config value " + nonDefaultValue);
        }
        return new LocalConfig(appId, ndkEnabled, configs);
    }

    /**
     * Given a config property name and a config property type, retrieves the embrace config resource id.
     *
     * @param context
     * @param configProperty
     * @param type
     * @return
     */
    private static int getResourcesIdentifier(Context context, String configProperty, String type) {
        Resources resources = context.getResources();
        return resources.getIdentifier(configProperty, type, context.getPackageName());
    }

    String getAppId() {
        return appId;
    }

    Boolean isNdkEnabled() {
        return ndkEnabled;
    }

    public SdkConfigs getConfigurations() {
        return configs;
    }

    /**
     * Checks if the url is allowed to be reported based on the specified disabled pattern.
     *
     * @param url the url to test
     * @return true if the url is disabled for reporting, false otherwise
     */
    boolean isUrlDisabled(String url) {
        Set<Pattern> patterns = getConfigurations().getNetworking().getDisabledUrlRegexPatterns();
        return StreamSupport.stream(patterns)
                .anyMatch(p -> p.matcher(url).find());
    }

    static class SdkConfigs {

        /**
         * App settings
         */
        @SerializedName(APP_CONFIG_KEY)
        private App app;

        /**
         * Base URL settings
         */
        @SerializedName(BASE_URL_KEY)
        private BaseUrls baseUrls;

        /**
         * Crash handler settings
         */
        @SerializedName(CRASH_HANDLER_CONFIG_KEY)
        private CrashHandler crashHandler;

        /**
         * Startup moment settings
         */
        @SerializedName(STARTUP_MOMENT_CONFIG_KEY)
        private StartupMoment startupMoment;

        /**
         * Networking moment settings
         */
        @SerializedName(NETWORKING_CONFIG_KEY)
        private Networking networking;

        /**
         * Session config settings
         */
        @SerializedName(SESSION_CONFIG_KEY)
        private SessionConfig sessionConfig;

        /**
         * Taps
         */
        @SerializedName(TAPS_CONFIG_KEY)
        private Taps taps;

        /**
         * Webview settings
         */
        @SerializedName(WEBVIEW_CONFIG_KEY)
        private WebView webView;

        SdkConfigs() {
            this(null, null, null, null, null, null, null, null);
        }

        SdkConfigs(App app, BaseUrls baseUrls, CrashHandler crashHandler, StartupMoment startupMoment, Networking networking, SessionConfig sessionConfig, Taps taps, WebView webView) {

            this.app = Optional.fromNullable(app).or(new App());
            this.baseUrls = Optional.fromNullable(baseUrls).or(new BaseUrls());
            this.crashHandler = Optional.fromNullable(crashHandler).or(new CrashHandler());
            this.startupMoment = Optional.fromNullable(startupMoment).or(new StartupMoment());
            this.networking = Optional.fromNullable(networking).or(new Networking());
            this.sessionConfig = Optional.fromNullable(sessionConfig).or(new SessionConfig());
            this.taps = Optional.fromNullable(taps).or(new Taps());
            this.webView = Optional.fromNullable(webView).or(new WebView());
        }

        App getApp() {
            return app;
        }

        BaseUrls getBaseUrls() {
            return baseUrls;
        }

        CrashHandler getCrashHandler() {
            return crashHandler;
        }

        StartupMoment getStartupMoment() {
            return startupMoment;
        }

        Networking getNetworking() {
            return networking;
        }

        SessionConfig getSessionConfig() {
            return sessionConfig;
        }

        Taps getTaps() {
            return taps;
        }

        public WebView getWebViewConfig() {
            return webView;
        }

        List<String> getNonDefaultValues() {
            List<String> nonDefaultValues = new ArrayList<>();
            List<ConfigElement> elements = Arrays.asList(app, baseUrls, crashHandler, networking, sessionConfig, startupMoment, taps, webView);
            for (ConfigElement element : elements) {
                if (element.getNonDefaultValues().isEmpty()) {
                    continue;
                }
                nonDefaultValues.addAll(element.getNonDefaultValues());
            }
            return nonDefaultValues;
        }

        interface ConfigElement {
            List<String> getNonDefaultValues();
        }

        /**
         * Represents the base URLs element specified in the Embrace config file.
         */
        static class App implements ConfigElement {
            static final boolean REPORT_DISK_USAGE_DEFAULT = true;

            /**
             * Control whether we scan for and report app disk usage. This can be a costly operation
             * for apps with a lot of local files.
             */
            @SerializedName(APP_DISK_USAGE_CONFIG_KEY)
            private boolean reportDiskUsage;

            App() {
                this(null);
            }

            App(Boolean reportDiskUsage) {
                this.reportDiskUsage = Optional.fromNullable(reportDiskUsage).or(REPORT_DISK_USAGE_DEFAULT);
            }

            boolean getReportDiskUsage() {
                return reportDiskUsage;
            }

            @Override
            public List<String> getNonDefaultValues() {
                List<String> nonDefaultValues = new ArrayList<>();
                if (REPORT_DISK_USAGE_DEFAULT != reportDiskUsage) {
                    nonDefaultValues.add("app.report_disk_usage: " + this.reportDiskUsage);
                }
                return nonDefaultValues;
            }
        }

        /**
         * Represents the base URLs element specified in the Embrace config file.
         */
        static class BaseUrls implements ConfigElement {
            static final String CONFIG_DEFAULT = "https://config.emb-api.com";
            static final String DATA_DEFAULT = "https://data.emb-api.com";
            static final String DATA_DEV_DEFAULT = "https://data-dev.emb-api.com";
            static final String IMAGES_DEFAULT = "https://images.emb-api.com";

            /**
             * Data base URL.
             */
            @SerializedName(BASE_URL_DATA_CONFIG_KEY)
            private String data;

            /**
             * Data dev base URL.
             */
            @SerializedName(BASE_URL_DATA_DEV_CONFIG_KEY)
            private String dataDev;

            /**
             * Config base URL.
             */
            @SerializedName(BASE_URL_CONFIG_KEY)
            private String config;

            /**
             * Images base URL.
             */
            @SerializedName(BASE_URL_IMAGES_CONFIG_KEY)
            private String images;

            BaseUrls() {
                this(null, null, null, null);
            }

            BaseUrls(String config, String data, String dataDev, String images) {
                this.config = Optional.fromNullable(config).or(CONFIG_DEFAULT);
                this.data = Optional.fromNullable(data).or(DATA_DEFAULT);
                this.dataDev = Optional.fromNullable(dataDev).or(DATA_DEV_DEFAULT);
                this.images = Optional.fromNullable(images).or(IMAGES_DEFAULT);
            }

            String getConfig() {
                return config;
            }

            String getData() {
                return data;
            }

            String getDataDev() {
                return dataDev;
            }

            String getImages() {
                return images;
            }

            public List<String> getNonDefaultValues() {
                List<String> nonDefaultValues = new ArrayList<>();
                if (!config.equals(CONFIG_DEFAULT)) {
                    nonDefaultValues.add("base_urls.config: " + this.config);
                }
                if (!data.equals(DATA_DEFAULT)) {
                    nonDefaultValues.add("base_urls.data: " + this.data);
                }
                if (!dataDev.equals(DATA_DEV_DEFAULT)) {
                    nonDefaultValues.add("base_urls.data_dev: " + this.dataDev);
                }
                if (!images.equals(IMAGES_DEFAULT)) {
                    nonDefaultValues.add("base_urls.images: " + this.images);
                }
                return nonDefaultValues;
            }
        }

        /**
         * Represents the crash handler element specified in the Embrace config file.
         */
        static class CrashHandler implements ConfigElement {

            static final Boolean ENABLED_DEFAULT = true;

            /**
             * Control whether the Embrace SDK automatically attaches to the uncaught exception handler.
             */
            @SerializedName(CRASH_HANDLER_ENABLED_CONFIG_KEY)
            private Boolean enabled;

            CrashHandler() {
                this(null);
            }

            CrashHandler(Boolean enabled) {
                this.enabled = Optional.fromNullable(enabled).or(ENABLED_DEFAULT);
            }

            Boolean getEnabled() {
                return enabled;
            }

            public List<String> getNonDefaultValues() {
                List<String> nonDefaultValues = new ArrayList<>();
                if (!enabled.equals(ENABLED_DEFAULT)) {
                    nonDefaultValues.add("crash_handler.enabled: " + enabled);
                }
                return nonDefaultValues;
            }
        }

        /**
         * Represents the startup moment configuration element specified in the Embrace config file.
         */
        static class StartupMoment implements ConfigElement {

            static final Boolean AUTOMATICALLY_END_DEFAULT = true;
            static final Boolean TAKE_SCREENSHOT_DEFAULT = true;

            /**
             * Control whether the startup moment is automatically ended.
             */
            @SerializedName(STARTUP_MOMENT_AUTOMATICALLY_END_CONFIG_KEY)
            private Boolean automaticallyEnd;

            /**
             * Control whether startup moment screenshots are taken.
             */
            @SerializedName(STARTUP_MOMENT_TAKE_SCREENSHOT_CONFIG_KEY)
            private Boolean takeScreenshot;


            StartupMoment() {
                this(null, null);
            }

            StartupMoment(Boolean automaticallyEnd, Boolean takeScreenshot) {
                this.automaticallyEnd = Optional.fromNullable(automaticallyEnd).or(AUTOMATICALLY_END_DEFAULT);
                this.takeScreenshot = Optional.fromNullable(takeScreenshot).or(TAKE_SCREENSHOT_DEFAULT);
            }

            Boolean getAutomaticallyEnd() {
                return automaticallyEnd;
            }

            Boolean getTakeScreenshot() {
                return takeScreenshot;
            }

            public List<String> getNonDefaultValues() {
                List<String> nonDefaultValues = new ArrayList<>();
                if (!automaticallyEnd.equals(AUTOMATICALLY_END_DEFAULT)) {
                    nonDefaultValues.add("startup_moment.automatically_end: " + automaticallyEnd);
                }
                if (!takeScreenshot.equals(TAKE_SCREENSHOT_DEFAULT)) {
                    nonDefaultValues.add("startup_moment.take_screenshot: " + takeScreenshot);
                }
                return nonDefaultValues;
            }
        }

        /**
         * Represents the networking configuration element specified in the Embrace config file.
         */
        static class Networking implements ConfigElement {

            /**
             * Sets the default name of the HTTP request header to extract trace ID from.
             */
            static final String CONFIG_TRACE_ID_HEADER_DEFAULT_VALUE = "x-emb-trace-id";

            /**
             * Capture request content length by default.
             */
            static final Boolean CAPTURE_REQUEST_CONTENT_LENGTH = false;

            /**
             * Enable native monitoring by default.
             */
            static final Boolean ENABLE_NATIVE_MONITORING_DEFAULT = true;

            /**
             * The default capture limit for the specified domains.
             */
            @SerializedName(NETWORKING_DEFAULT_CAPTURE_LIMIT_CONFIG_KEY)
            private Integer defaultCaptureLimit;

            /**
             * List of domains to be limited for tracking.
             */
            @SerializedName(NETWORKING_DOMAINS_CONFIG_KEY)
            private List<Domain> domains;

            /**
             * The Trace ID Header that can be used to trace a particular request.
             */
            @SerializedName(NETWORKING_TRACE_ID_CONFIG_KEY)
            private String traceIdHeader;

            /**
             * Control whether request size for native Android requests is captured.
             */
            @SerializedName(NETWORKING_CAPTURE_REQUEST_LENGTH_CONFIG_KEY)
            private Boolean captureRequestContentLength;

            /**
             * URLs that should not be captured.
             */
            @SerializedName(NETWORKING_DISABLE_URL_PATTERNS_CONFIG_KEY)
            private List<String> disabledUrlPatterns;

            /**
             * Enable the native monitoring.
             */
            @SerializedName(NETWORKING_ENABLE_MONITORING_CONFIG_KEY)
            private Boolean enableNativeMonitoring;

            private Set<Pattern> disabledUrlRegexPatterns;

            private ArrayList<String> nonDefaultValues;

            Networking() {
                this(null, null, null, null, null, null);
            }

            Networking(String traceIdHeader, Integer defaultCaptureLimit, List<Domain> domains,
                       Boolean captureRequestContentLength, List<String> disabledUrlPatterns,
                       Boolean enableNativeMonitoring) {
                this.traceIdHeader = Optional.fromNullable(traceIdHeader).or(CONFIG_TRACE_ID_HEADER_DEFAULT_VALUE);
                this.defaultCaptureLimit = defaultCaptureLimit;
                this.domains = domains;
                this.captureRequestContentLength = Optional.fromNullable(captureRequestContentLength).or(CAPTURE_REQUEST_CONTENT_LENGTH);
                this.disabledUrlPatterns = disabledUrlPatterns;
                this.enableNativeMonitoring = Optional.fromNullable(enableNativeMonitoring).or(ENABLE_NATIVE_MONITORING_DEFAULT);
            }

            String getTraceIdHeader() {
                return traceIdHeader;
            }

            synchronized Set<Pattern> getDisabledUrlRegexPatterns() {
                if (disabledUrlRegexPatterns == null) {
                    if (disabledUrlPatterns != null) {
                        disabledUrlRegexPatterns = new HashSet<>();
                        for (String url : disabledUrlPatterns) {
                            try {
                                disabledUrlRegexPatterns.add(Pattern.compile(url));
                            } catch (PatternSyntaxException e) {
                                EmbraceLogger.logError("Skipping invalid blocked URL: " + url, e);
                            }
                        }
                    } else {
                        disabledUrlRegexPatterns = Collections.emptySet();
                    }
                }
                return disabledUrlRegexPatterns;
            }

            Boolean getCaptureRequestContentLength() {
                return captureRequestContentLength;
            }

            public Optional<Integer> getDefaultCaptureLimit() {
                return Optional.fromNullable(defaultCaptureLimit);
            }

            public List<Domain> getDomains() {
                return domains != null ? domains : Collections.emptyList();
            }

            public Boolean isNativeNetworkingMonitoringEnabled() {
                return enableNativeMonitoring;
            }

            synchronized public List<String> getNonDefaultValues() {
                if (nonDefaultValues == null) {
                    nonDefaultValues = new ArrayList<>();
                    if (!traceIdHeader.equals(CONFIG_TRACE_ID_HEADER_DEFAULT_VALUE)) {
                        nonDefaultValues.add("networking.trace_id_header: " + traceIdHeader);
                    }
                    if (!captureRequestContentLength.equals(CAPTURE_REQUEST_CONTENT_LENGTH)) {
                        nonDefaultValues.add("networking.capture_request_content_length: " + captureRequestContentLength);
                    }
                    if (!enableNativeMonitoring.equals(ENABLE_NATIVE_MONITORING_DEFAULT)) {
                        nonDefaultValues.add("networking.enable_native_monitoring: " + enableNativeMonitoring);
                    }
                    if (disabledUrlPatterns != null) {
                        for (String pattern : disabledUrlPatterns) {
                            nonDefaultValues.add("networking.disabled_url_pattern: " + pattern);
                        }
                    }
                }
                return nonDefaultValues;
            }

            /**
             * Represents each domain element specified in the Embrace config file.
             */
            public static class Domain {

                /**
                 * Url for the domain.
                 */
                @SerializedName(NETWORKING_DOMAIN_NAME_CONFIG_KEY)
                private String domain;

                /**
                 * Limit for the number of requests to be tracked.
                 */
                @SerializedName(NETWORKING_DOMAIN_LIMIT_CONFIG_KEY)
                private Integer limit;

                Domain() {
                    this(null, null);
                }

                Domain(String domain, Integer limit) {
                    this.domain = domain;
                    this.limit = limit;
                }

                String getDomain() {
                    return domain;
                }

                Optional<Integer> getLimit() {
                    return Optional.of(limit);
                }
            }
        }

        /**
         * Represents the session configuration element specified in the Embrace config file.
         */
        static class SessionConfig implements ConfigElement {

            /**
             * Default minimum allowed end session time.
             */
            static final int MINIMUM_SESSION_SECONDS_DEFAULT = 60;

            /**
             * Do not use async mode for session end messages by default.
             */
            static final boolean ASYNC_END_DEFAULT = false;

            /**
             * Specify a maximum time before a session is allowed to exist before it is ended.
             */
            @SerializedName(SESSION_MAX_SESSION_SEC_CONFIG_KEY)
            private Integer maxSessionSeconds;

            /**
             * End session messages are sent asynchronously.
             */
            @SerializedName(SESSION_ASYNC_END_CONFIG_KEY)
            private Boolean asyncEnd;

            SessionConfig() {
                this(null, null);
            }

            SessionConfig(Integer maxSessionSeconds, Boolean asyncEnd) {
                this.maxSessionSeconds = maxSessionSeconds;
                this.asyncEnd = asyncEnd;
            }

            Optional<Integer> getMaxSessionSeconds() {
                if (maxSessionSeconds != null) {
                    if (maxSessionSeconds >= MINIMUM_SESSION_SECONDS_DEFAULT) {
                        return Optional.of(maxSessionSeconds);
                    } else {
                        EmbraceLogger.logWarning("Automatic end session disabled. Config max_session_seconds should be more than 60 seconds.");
                        return Optional.absent();
                    }
                } else {
                    return Optional.absent();
                }
            }

            Boolean getAsyncEnd() {
                return asyncEnd != null ? asyncEnd : ASYNC_END_DEFAULT;
            }

            public List<String> getNonDefaultValues() {
                List<String> nonDefaultValues = new ArrayList<>();
                if (maxSessionSeconds != null) {
                    nonDefaultValues.add("session.max_session_seconds: " + maxSessionSeconds);
                }
                if (getAsyncEnd() != ASYNC_END_DEFAULT) {
                    nonDefaultValues.add("session.async_end: " + getAsyncEnd());
                }
                return nonDefaultValues;
            }
        }

        static class Taps implements ConfigElement {

            static final Boolean CAPTURE_COORDINATES_DEFAULT = true;

            /**
             * Control whether tap coordindates are captured.
             */
            @SerializedName(TAPS_CAPTURE_COORDINATES_CONFIG_KEY)
            private Boolean captureCoordinates;

            Taps() {
                this(null);
            }

            Taps(Boolean captureCoordinates) {
                this.captureCoordinates = Optional.fromNullable(captureCoordinates).or(CAPTURE_COORDINATES_DEFAULT);
            }

            Boolean getCaptureCoordinates() {
                return captureCoordinates;
            }

            public List<String> getNonDefaultValues() {
                List<String> nonDefaultValues = new ArrayList<>();
                if (!captureCoordinates.equals(CAPTURE_COORDINATES_DEFAULT)) {
                    nonDefaultValues.add("taps.capture_coordinates: " + captureCoordinates);
                }
                return nonDefaultValues;
            }
        }

        static class WebView implements ConfigElement {

            static final Boolean WEB_VIEW_CAPTURE_DEFAULT = true;
            static final Boolean WEB_VIEW_QUERY_PARAMS_CAPTURE_DEFAULT = true;

            /**
             * Controls whether webviews are captured.
             */
            @SerializedName(WEBVIEW_ENABLE_CAPTURE_CONFIG_KEY)
            private Boolean captureWebViews;


            /**
             * Control whether query params for webviews are captured.
             */
            @SerializedName(WEBVIEW_CAPTURE_QUERY_PARAMS_CONFIG_KEY)
            private Boolean captureQueryParams;

            WebView() {
                this(null, null);
            }

            public WebView(Boolean captureWebViews, Boolean captureQueryParams) {
                this.captureWebViews = Optional.fromNullable(captureWebViews).or(WEB_VIEW_CAPTURE_DEFAULT);
                this.captureQueryParams = Optional.fromNullable(captureQueryParams).or(WEB_VIEW_QUERY_PARAMS_CAPTURE_DEFAULT);
            }

            public Boolean isWebViewsCaptureEnabled() {
                return captureWebViews;
            }

            public Boolean isQueryParamsCaptureEnabled() {
                return captureQueryParams;
            }

            @Override
            public List<String> getNonDefaultValues() {
                List<String> nonDefaultValues = new ArrayList<>();
                if (!captureWebViews.equals(WEB_VIEW_CAPTURE_DEFAULT)) {
                    nonDefaultValues.add("web_view.enabled: " + captureWebViews);
                }
                if (!captureQueryParams.equals(WEB_VIEW_QUERY_PARAMS_CAPTURE_DEFAULT)) {
                    nonDefaultValues.add("web_view.capture_query_params: " + captureQueryParams);
                }
                return nonDefaultValues;
            }
        }
    }
}
