package io.embrace.android.embracesdk;

import java.io.Closeable;

/**
 * Provides access to the configuration for the customer's app.
 * <p>
 * Configuration is configured for the user's app, and exposed via the API.
 */
interface ConfigService extends Closeable {

    /**
     * Gets the current configuration for the user's app, refreshing the configuration from the
     * server if it has expired.
     * <p>
     * If the configuration cannot be retrieved from the remote server, default configuration is
     * used.
     *
     * @return the app's Embrace configuration
     */
    Config getConfig();

    /**
     * Checks if the event name allows the capture of screenshot to be reported based on the
     * specified disabled pattern.
     *
     * @param eventName name for the event
     * @return true if screenshots are disabled, false otherwise
     */
    boolean isScreenshotDisabledForEvent(String eventName);

    /**
     * Checks if the event name is allowed to be reported based on the specified disabled pattern.
     *
     * @param eventName name for the event
     * @return true if events reports are disabled, false otherwise
     */
    boolean isEventDisabled(String eventName);

    /**
     * Checks if the log message is allowed to be reported based on the specified disabled pattern.
     *
     * @param logMessage name for the event
     * @return true if logs reports are disabled, false otherwise
     */
    boolean isLogMessageDisabled(String logMessage);

    /**
     * Checks if the message type is allowed to be reported based on the specified disabled pattern.
     *
     * @param type type of message
     * @return true if the message type is disabled for reporting, false otherwise
     */
    boolean isMessageTypeDisabled(MessageType type);

    /**
     * Checks if the url is allowed to be reported based on the specified disabled pattern.
     *
     * @param url the url to test
     * @return true if the url is disabled for reporting, false otherwise
     */
    boolean isUrlDisabled(String url);

    /**
     * Checks if the SDK is disabled.
     * <p>
     * The SDK can be configured to disable a percentage of devices based on the normalization of
     * their device ID between 1-100. This threshold is set in {@link Config}.
     *
     * @return true if the sdk is disabled, false otherwise
     */
    boolean isSdkDisabled();

    /**
     * Checks if the Anr Capture is enabled.
     * <p>
     * The Anr capture can be configured to enable a percentage of devices based on
     * the normalization of their device ID between 1-100.
     *
     * @return true if the anr capture is enabled, false otherwise
     */
    boolean isAnrCaptureEnabled();

    /**
     * Adds a listener for changes to the {@link Config}. The listeners will be notified when the
     * {@link ConfigService} refreshes its configuration.
     *
     * @param configListener the listener to add
     */
    void addListener(ConfigListener configListener);

    /**
     * Removes an existing listener, if it exists, from the {@link ConfigService} so that it will no
     * longer be notified of changes to the {@link Config}.
     *
     * @param configListener the listener to remove
     */
    void removeListener(ConfigListener configListener);


    /**
     * Checks if the session controls is allowed for the user to manage;
     *
     * @return true if the session control is enabled.
     */
    boolean isSessionControlEnabled();

    /**
     * Checks if internal exception capture is enabled;
     *
     * @return true if internal exception capture is enabled.
     */
    boolean isInternalExceptionCaptureEnabled();
}
