package io.embrace.android.embracesdk;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.app.usage.StorageStats;
import android.app.usage.StorageStatsManager;
import android.content.ContentResolver;
import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Process;
import android.os.StatFs;
import android.os.SystemClock;
import android.os.storage.StorageManager;
import android.provider.Settings;
import android.util.DisplayMetrics;
import android.view.Display;
import android.view.WindowManager;

import com.fernandocejas.arrow.optional.Optional;

import java.io.File;
import java.math.BigInteger;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.List;
import java.util.Locale;
import java.util.TimeZone;

import java9.util.Lists;
import java9.util.stream.StreamSupport;

/**
 * Utilities for retrieving metadata from the device's {@link Context}. This metadata is passed
 * to the API with certain messages to provide device information.
 */
final class MetadataUtils {
    private static final String OS_VERSION = "Android OS";

    private static final String ENVIRONMENT_DEV = "dev";

    private static final String ENVIRONMENT_PROD = "prod";

    private static final List<String> JAILBREAK_LOCATIONS = Lists.of(
            "/sbin/",
            "/system/bin/",
            "/system/xbin/",
            "/data/local/xbin/",
            "/data/local/bin/",
            "/system/sd/xbin/",
            "/system/bin/failsafe/",
            "/data/local/");

    /**
     * Gets the name of the manufacturer of the device.
     *
     * @return the name of the device manufacturer
     */
    static String getDeviceManufacturer() {
        return Build.MANUFACTURER;
    }

    /**
     * Gets the name of the model of the device.
     *
     * @return the name of the model of the device
     */
    static String getModel() {
        return Build.MODEL;
    }

    /**
     * Gets the architecture of the device.
     *
     * @return the device architecture
     */
    static String getArchitecture() {
        return Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP ?
                Build.SUPPORTED_ABIS[0] :
                Build.CPU_ABI;
    }

    /**
     * Gets the locale of the device, represented as "language_country".
     *
     * @return the locale of the device
     */
    static String getLocale() {
        return Locale.getDefault().getLanguage() + "_" + Locale.getDefault().getCountry();
    }

    /**
     * Gets the operating system of the device. This is hard-coded to Android OS.
     *
     * @return the device's operating system
     */
    static String getOperatingSystemType() {
        return OS_VERSION;
    }

    /**
     * Gets the version of the installed operating system on the device.
     *
     * @return the version of the operating system
     */
    static String getOperatingSystemVersion() {
        return String.valueOf(Build.VERSION.RELEASE);
    }

    /**
     * Gets the version code of the running Android SDK.
     *
     * @return the running Android SDK version code
     */
    static int getOperatingSystemVersionCode() {
        return Build.VERSION.SDK_INT;
    }

    /**
     * Formats the Android SDK version code in a format expected by the Embrace server for retrieving
     * device configuration.
     *
     * @return the formatted version code
     */
    static String getOperatingSystemVersionForRequest() {
        return String.format(Locale.US, "%d.0.0", Build.VERSION.SDK_INT);
    }

    /**
     * Gets the system uptime in milliseconds.
     *
     * @return the system uptime in milliseconds
     */
    static Long getSystemUptime() {
        return SystemClock.uptimeMillis();
    }

    /**
     * Gets the device's screen resolution.
     *
     * @param windowManager the {@link WindowManager} from the {@link Context}
     * @return the device's screen resolution
     */
    static Optional<String> getScreenResolution(WindowManager windowManager) {
        try {
            Display display = windowManager.getDefaultDisplay();
            DisplayMetrics displayMetrics = new DisplayMetrics();
            display.getMetrics(displayMetrics);
            return Optional.of(
                    String.format(Locale.US, "%dx%d", displayMetrics.widthPixels, displayMetrics.heightPixels));
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Could not determine screen resolution", ex);
            return Optional.absent();
        }
    }

    /**
     * Gets a ID of the device's timezone, e.g. 'Europe/London'.
     *
     * @return the ID of the device's timezone
     */
    static String getTimezoneId() {
        return TimeZone.getDefault().getID();
    }

    /**
     * Gets the total storage capacity of the device.
     *
     * @param statFs the {@link StatFs} service for the device
     * @return the total storage capacity in bytes
     */
    static long getInternalStorageTotalCapacity(StatFs statFs) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {
            return statFs.getTotalBytes();
        } else {
            return (long) statFs.getBlockCount() * (long) statFs.getBlockSize();
        }
    }

    /**
     * Gets the free capacity of the internal storage of the device.
     *
     * @param statFs the {@link StatFs} service for the device
     * @return the total free capacity of the internal storage of the device in bytes
     */
    static long getInternalStorageFreeCapacity(StatFs statFs) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {
            return statFs.getFreeBytes();
        } else {
            return (long) statFs.getFreeBlocks() * (long) statFs.getBlockSize();
        }
    }

    /**
     * Attempts to determine the disk usage of the app on the device.
     * <p>
     * If the disk usage cannot be determined, an absent {@link Optional} is returned.
     *
     * @param storageStatsManager the {@link StorageStatsManager}
     * @param packageManager the {@link PackageManager}
     * @param contextPackageName the name of the package from the {@link Context}
     * @return optionally the disk usage of the app on the device
     */
    @TargetApi(Build.VERSION_CODES.O)
    static Optional<Long> getDeviceDiskAppUsage(
            StorageStatsManager storageStatsManager,
            PackageManager packageManager,
            String contextPackageName) {

            try {
                PackageInfo packageInfo = packageManager.getPackageInfo(contextPackageName, 0);
                if (packageInfo != null && packageInfo.packageName != null) {
                    StorageStats stats = storageStatsManager.queryStatsForPackage(
                            StorageManager.UUID_DEFAULT,
                            packageInfo.packageName,
                            Process.myUserHandle());
                    return Optional.of(stats.getAppBytes() + stats.getDataBytes() + stats.getCacheBytes());
                }
            } catch (Exception ex) {
                // The package name and storage volume should always exist
                EmbraceLogger.logDebug("Error retrieving device disk usage", ex);
            }
        return Optional.absent();
    }

    /**
     * Tries to determine whether the device is jailbroken by looking for specific directories which
     * exist on jailbroken devices. Emulators are excluded and will always return false.
     *
     * @return true if the device is jailbroken and not an emulator, false otherwise
     */
    static boolean isJailbroken() {
        if (isEmulator()) {
            return false;
        }
        return StreamSupport.stream(JAILBREAK_LOCATIONS)
                .anyMatch(path -> new File(path + "su").exists());
    }

    /**
     * Tries to determine whether the device is an emulator by looking for known models and
     * manufacturers which correspond to emulators.
     *
     * @return true if the device is detected to be an emulator, false otherwise
     */
    static boolean isEmulator() {
        return Build.FINGERPRINT.startsWith("generic")
                || Build.FINGERPRINT.startsWith("unknown")
                || Build.MODEL.contains("google_sdk")
                || Build.MODEL.contains("Emulator")
                || Build.MODEL.contains("Android SDK built for x86")
                || Build.MANUFACTURER.contains("Genymotion")
                || (Build.BRAND.startsWith("generic") && Build.DEVICE.startsWith("generic"))
                || "google_sdk".equals(Build.PRODUCT);
    }

    /**
     * Attempts to determine the formatted version code of the package expected by the Embrace SDK.
     *
     * @param packageManager the {@link PackageManager}
     * @param packageName the name of the package from the {@link Context}
     * @return optionally the formatted version code
     */
    static Optional<String> getVersionCodeForRequest(PackageManager packageManager, String packageName) {
        try {
            PackageInfo packageInfo = packageManager.getPackageInfo(packageName, 0);
            String result = String.format(Locale.US, "%d.0.0", packageInfo.versionCode);
            return Optional.of(result);
        } catch (Exception ex) {
            // Package name of the running app should always exist
            EmbraceLogger.logDebug("Error retrieving version information from the package", ex);
            return Optional.absent();
        }
    }

    /**
     * Generates a device ID based on an MD5 of the secure hardware ID of the device.
     * <p>
     * If the secure ID cannot be determined, a UUID is used.
     *
     * @param contentResolver the content resolver used for resolving the secure device ID
     * @return the device ID
     */
    static String createDeviceId(ContentResolver contentResolver) {
        @SuppressLint("HardwareIds")
        String secureId = Settings.Secure.getString(contentResolver, Settings.Secure.ANDROID_ID);
        MessageDigest messageDigest = null;

        try {
            messageDigest = MessageDigest.getInstance("MD5");
        } catch (NoSuchAlgorithmException e) {
            // Shouldn't be triggered as the MD5 algo should always be available
        }

        if (messageDigest != null && secureId != null) {
            // Return the MD5 value of the secure Android ID and use that value as the hash.
            messageDigest.update(secureId.getBytes(), 0, secureId.length());
            return Uuid.getEmbUuid(new BigInteger(1, messageDigest.digest()).toString(16));
        } else {
            // Return the Embrace UUID based on the secure ID. If secure ID is null, return the Embrace
            // UUID using a random UUID.
            return Uuid.getEmbUuid(secureId);
        }
    }

    /**
     * Determines whether or not this is a debug build of the app.
     *
     * @param applicationInfo the application info used for introspecting flags
     * @return true if this is a debug build
     */
    static boolean isDebug(ApplicationInfo applicationInfo) {
        return (applicationInfo.flags & applicationInfo.FLAG_DEBUGGABLE) != 0;
    }

    /**
     * Creates a "prod" or "dev" environment string, depending on whether or not this is a debug
     * build of the app.
     *
     * @param applicationInfo the application info used for introspecting flags
     * @return a string representation of the environment name
     */
    static String appEnvironment(ApplicationInfo applicationInfo) {
        return isDebug(applicationInfo) ? ENVIRONMENT_DEV : ENVIRONMENT_PROD;
    }

    private MetadataUtils() {
        // Restricted constructor
    }
}
