package io.embrace.android.embracesdk;

import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.Message;

import com.fernandocejas.arrow.checks.Preconditions;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Set;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.regex.Pattern;

/**
 * Checks whether the main thread is still responding by using the following strategy:
 * <ol>
 * <li>Creating two {@link Handler}s, one on the main thread, and one on a new thread</li>
 * <li>Using the 'monitoring' thread to message the main thread with a heartbeat</li>
 * <li>Determining whether the main thread responds in time, and if not logging an ANR</li>
 * </ol>
 */
final class EmbraceAnrService implements AnrService, MemoryCleanerListener, ActivityListener {

    private static final long ANR_THRESHOLD_INTERVAL = 1000L;

    private static final int HEALTHCHECK_EXECUTE = 34592;

    private static final int HEALTHCHECK_REQUEST = 34593;

    private static final int HEALTHCHECK_RESPONSE = 34594;

    private static final long BACKGROUND_ANR_SAFE_INTERVAL_MS = 10L;

    private final Clock clock;

    private final ConfigService configService;

    private final NavigableMap<Long, AnrInterval> anrIntervals;
    private int anrIntervalsStackTraceCount;

    private final MainThreadHandler mainThreadHandler;

    private final MonitoringThreadHandler monitoringThreadHandler;

    private final HandlerThread monitoringThread;
    private final HashMap<Long, Integer> currentStacktraceStates = new HashMap<>();
    private volatile long lastTimeThreadUnblocked;
    private volatile boolean anrInProgress;
    private AnrStacktraces stacktraces;

    EmbraceAnrService(
            Clock clock,
            MemoryCleanerService memoryCleanerService,
            ConfigService configService,
            EmbraceActivityService activityService) {
        this.clock = clock;
        this.configService = Preconditions.checkNotNull(configService);
        this.anrIntervals = new ConcurrentSkipListMap<>();
        anrIntervalsStackTraceCount = 0;
        this.monitoringThread = new HandlerThread("Embrace ANR Healthcheck");
        this.monitoringThread.start();
        Looper callingThreadLooper = this.monitoringThread.getLooper();
        this.monitoringThreadHandler = new MonitoringThreadHandler(callingThreadLooper);
        this.mainThreadHandler = new MainThreadHandler(Looper.getMainLooper());
        this.monitoringThreadHandler.runHealthCheck();
        Preconditions.checkNotNull(memoryCleanerService).addListener(this);
        Preconditions.checkNotNull(activityService).addListener(this, true);

        stacktraces = new AnrStacktraces();
    }

    @Override
    public void onForeground(boolean coldStart, long startupTime) {
        if (!coldStart) {
            forceHealthCheck();
        }
    }

    @Override
    public List<AnrInterval> getAnrIntervals(long startTime, long endTime) {
        synchronized (this) {
            ArrayList<AnrInterval> results;
            long safeStartTime = startTime + BACKGROUND_ANR_SAFE_INTERVAL_MS;

            if (safeStartTime < endTime) {
                Collection<AnrInterval> intervals = anrIntervals.subMap(safeStartTime, endTime).values();
                results = new ArrayList<>(intervals);
            } else {
                results = new ArrayList<>();
            }

            if (anrInProgress) {
                long intervalEndTime = clock.now();
                long duration = intervalEndTime - lastTimeThreadUnblocked;

                AnrInterval.Builder anrIntervalBuilder = AnrInterval.newBuilder();
                anrIntervalBuilder
                        .withStartTime(lastTimeThreadUnblocked)
                        .withLastKnownTime(intervalEndTime)
                        .withEndTime(null)
                        .withType(AnrInterval.Type.UI);

                if (configService.isAnrCaptureEnabled() && isMinimumCaptureDurationExceeded(duration)) {
                    anrIntervalBuilder.withStacktraces(stacktraces);
                }

                results.add(anrIntervalBuilder.build());
            }

            return results;
        }
    }

    @Override
    public void close() {
        try {
            monitoringThread.quit();
            monitoringThread.join();
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to cleanly shut down EmbraceAnrService");
        }
    }

    @Override
    public void cleanCollections() {
        anrIntervals.clear();
        anrIntervalsStackTraceCount = 0;
    }

    private void forceHealthCheck() {
        monitoringThreadHandler.handleHealthCheckResponse(clock.now());
    }

    private boolean isMinimumCaptureDurationExceeded(long duration) {
        return duration >= configService.getConfig().getAnrStacktraceMinimumDuration();
    }

    /**
     * Performs the monitoring by submitting heartbeats to the {@link MainThreadHandler}, then
     * re-scheduling itself for another heartbeat by sending a message to itself with a delay.
     */
    class MonitoringThreadHandler extends Handler {

        public MonitoringThreadHandler(Looper looper) {
            super(looper);
        }

        @Override
        public void handleMessage(Message msg) {
            try {
                long timestamp = clock.now();
                if (msg.what == EmbraceAnrService.HEALTHCHECK_RESPONSE) {
                    handleHealthCheckResponse(timestamp);
                } else if (msg.what == EmbraceAnrService.HEALTHCHECK_EXECUTE) {
                    // Application is not responding
                    if (isAnrDurationThresholdExceeded(timestamp) && !anrInProgress) {
                        EmbraceLogger.logDebug("Main thread not responding for > 1s");
                        currentStacktraceStates.clear();
                        anrInProgress = true;
                    }

                    if (anrInProgress) {
                        processAnrTick();
                    }
                    runHealthCheck();
                }
            } catch (OutOfMemoryError oome) {
                // do nothing since we do not want to incur more memory usage
            } catch (Exception ex) {
                EmbraceLogger.logDebug("ANR healthcheck failed in monitoring thread", ex);
            }
        }

        void runHealthCheck() {
            mainThreadHandler.sendMessage(obtainMessage(EmbraceAnrService.HEALTHCHECK_REQUEST));
            monitoringThreadHandler.sendMessageDelayed(
                    obtainMessage(EmbraceAnrService.HEALTHCHECK_EXECUTE),
                    configService.getConfig().getCaptureAnrIntervalMs());
        }

        public synchronized void handleHealthCheckResponse(long timestamp) {
            if (isAnrDurationThresholdExceeded(timestamp)) {
                // Application was not responding, but recovered
                EmbraceLogger.logDebug("Main thread recovered from not responding for > 1s");
                // Finalize AnrInterval
                AnrInterval.Builder anrIntervalBuilder = AnrInterval.newBuilder();
                anrIntervalBuilder
                        .withStartTime(lastTimeThreadUnblocked)
                        .withLastKnownTime(timestamp)
                        .withEndTime(timestamp)
                        .withType(AnrInterval.Type.UI);

                long duration = timestamp - lastTimeThreadUnblocked;
                boolean shouldCaptureStacktrace = configService.isAnrCaptureEnabled() && isMinimumCaptureDurationExceeded(duration);
                if (shouldCaptureStacktrace) {
                    anrIntervalBuilder.withStacktraces(stacktraces);
                    if (reachedAnrCaptureLimit()) {
                        anrIntervals.lastEntry().getValue().removeStacktraces();
                    } else {
                        anrIntervalsStackTraceCount += 1;
                    }
                }
                anrIntervals.put(timestamp, anrIntervalBuilder.build());
                // Reset counters
                stacktraces = new AnrStacktraces();
                currentStacktraceStates.clear();
            }
            lastTimeThreadUnblocked = timestamp;
            anrInProgress = false;
        }

        private synchronized void processAnrTick() {
            // Check if ANR capture is enabled
            if (!configService.isAnrCaptureEnabled()) {
                return;
            }

            // Tick limit
            if (stacktraces.size() >= configService.getConfig().getStacktracesPerInterval()) {
                EmbraceLogger.logDebug("ANR stacktrace not captured. Maximum allowed ticks per ANR interval reached.");
                return;
            }

            AnrTick anrTick = new AnrTick(clock.now());

            for (ThreadInfo threadInfo : getAllowedThreads()) {
                // Compares every thread with the last known thread state via hashcode. If hashcode changed
                // it should be added to the anrInfo list and also the currentAnrInfoState must be updated.
                Integer threadHash = currentStacktraceStates.get(threadInfo.getThreadId());
                if (threadHash != null) {
                    if (threadHash != threadInfo.hashCode()) {
                        updateThread(threadInfo, anrTick);
                    }
                } else {
                    updateThread(threadInfo, anrTick);
                }
            }

            stacktraces.add(anrTick);
        }

        private void updateThread(ThreadInfo threadInfo, AnrTick anrTick) {
            currentStacktraceStates.put(threadInfo.getThreadId(), threadInfo.hashCode());
            anrTick.add(threadInfo);
        }

        /**
         * Filter the thread list based on allow/block list get by config.
         *
         * @return filtered threads
         */
        private Set<ThreadInfo> getAllowedThreads() {
            Set<ThreadInfo> allowed = new HashSet<>();
            List<Pattern> blockList = configService.getConfig().getAnrBlockPatternList();
            List<Pattern> allowList = configService.getConfig().getAnrAllowPatternList();
            int anrStacktracesMaxLength = configService.getConfig().getAnrStacktracesMaxDepth();
            int priority = configService.getConfig().getAnrThreadCapturePriority();

            if (configService.getConfig().captureMainThreadOnly()) {
                Thread mainThread = Looper.getMainLooper().getThread();
                ThreadInfo threadInfo = ThreadInfo.ofThread(
                        mainThread,
                        mainThread.getStackTrace(),
                        anrStacktracesMaxLength);

                allowed.add(threadInfo);
            } else {
                for (Map.Entry<Thread, StackTraceElement[]> entry : Thread.getAllStackTraces().entrySet()) {
                    ThreadInfo threadInfo = ThreadInfo.ofThread(
                            entry.getKey(),
                            entry.getValue(),
                            anrStacktracesMaxLength);

                    if (allowList != null && !allowList.isEmpty()) {
                        if (isAllowedByList(allowList, threadInfo) &&
                                isAllowedByPriority(priority, threadInfo)) {
                            allowed.add(threadInfo);
                        }
                    } else if (blockList != null && !blockList.isEmpty()) {
                        if (!isAllowedByList(blockList, threadInfo)
                                && isAllowedByPriority(priority, threadInfo)) {
                            allowed.add(threadInfo);
                        }
                    } else {
                        if (isAllowedByPriority(priority, threadInfo)) {
                            allowed.add(threadInfo);
                        }
                    }
                }
            }

            return allowed;
        }

        private boolean isAllowedByList(List<Pattern> allowed, ThreadInfo threadInfo) {
            for (Pattern pattern : allowed) {
                if (pattern.matcher(threadInfo.getName()).find()) {
                    return true;
                }
            }

            return false;
        }

        private boolean isAllowedByPriority(int priority, ThreadInfo threadInfo) {
            if (priority != 0) {
                return threadInfo.getPriority() >= priority;
            }

            return true;
        }

        private boolean reachedAnrCaptureLimit() {
            return anrIntervalsStackTraceCount >= configService.getConfig().getMaxAnrCapturedIntervalsPerSession();
        }

        private boolean isAnrDurationThresholdExceeded(long timestamp) {
            return lastTimeThreadUnblocked > 0 && (timestamp - lastTimeThreadUnblocked) > ANR_THRESHOLD_INTERVAL;
        }
    }

    /**
     * Handles healthcheck messages sent to the main thread. Responds with an acknowledgement to the
     * calling thread.
     */
    class MainThreadHandler extends Handler {

        public MainThreadHandler(Looper looper) {
            super(looper);
        }

        @Override
        public void handleMessage(Message msg) {
            try {
                if (msg.what == EmbraceAnrService.HEALTHCHECK_REQUEST) {
                    monitoringThreadHandler.sendMessage(obtainMessage(EmbraceAnrService.HEALTHCHECK_RESPONSE));
                }
            } catch (Exception ex) {
                EmbraceLogger.logDebug("ANR healthcheck failed in main (monitored) thread", ex);
            }
            super.handleMessage(msg);
        }
    }
}
