package io.embrace.android.embracesdk;

import com.fernandocejas.arrow.optional.Optional;

import java.util.List;

/**
 * Handles the caching of objects.
 */
interface CacheService {

    /**
     * Caches the specified object.
     *
     * @param name   the name of the object to cache
     * @param object the object to cache
     * @param clazz  the class of the object to cache
     * @param <T>    the type of the object
     */
    <T> void cacheObject(String name, T object, Class<T> clazz);

    /**
     * Reads the specified object from the cache, if it exists.
     *
     * @param name  the name of the object to read from the cache
     * @param clazz the class of the cached object
     * @param <T>   the type of the cached object
     * @return optionally the object, if it can be read successfully
     */
    <T> Optional<T> loadObject(String name, Class<T> clazz);

    /**
     * Reads the objects which names match with the specified regex from the cache, if they exist.
     *
     * @param regex the regex to match to the name of the object to read from the cache
     * @param clazz the class of the cached object
     * @param <T>   the type of the cached object
     * @return optionally a list of objects, if they can be read successfully
     */
    <T> List<T> loadObjectsByRegex(String regex, Class<T> clazz);

    /**
     * Deletes the specified object from the cache.
     *
     * @param name the name of the object to delete
     * @return true if the file was successfully deleted, false otherwise
     */
    boolean deleteObject(String name);

    /**
     * Deletes the objects which names match with the specified regex from the cache.
     *
     * @param regex the regex to match to the name of the object to delete
     * @return true if the files were successfully deleted, false otherwise
     */
    boolean deleteObjectsByRegex(String regex);

    /**
     * Moves the object using the current name to a new file called name.
     *
     * @param src the source file name
     * @param dst the destination file name
     * @return true if the file was successfully moved, false otherwise
     */
    boolean moveObject(String src, String dst);
}
