package io.embrace.android.embracesdk;

import android.content.Context;
import android.os.Debug;

import com.fernandocejas.arrow.checks.Preconditions;
import com.fernandocejas.arrow.strings.Charsets;
import com.google.gson.Gson;
import com.google.gson.stream.JsonReader;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.Future;
import java.util.zip.GZIPOutputStream;

import io.embrace.android.embracesdk.network.http.HttpMethod;
import io.embrace.android.embracesdk.networking.EmbraceConnection;
import io.embrace.android.embracesdk.networking.EmbraceUrl;
import io.embrace.android.embracesdk.utils.exceptions.Unchecked;
import java9.util.function.BiConsumer;
import java9.util.stream.Collectors;
import java9.util.stream.StreamSupport;

/**
 * Client for calling the Embrace API.
 */
class ApiClient implements Closeable {

    /**
     * The version of the API message format.
     */
    static final int MESSAGE_VERSION = 13;

    private static final int CONFIG_API_VERSION = 2;

    private static final int API_VERSION = 1;

    private static final String SCREENSHOT_URL_FORMAT = "%s/%s/%s/%s.jpg";

    private static final String URL_FORMAT = "%s/v%s/%s";

    private static final String CONFIG_URL_FORMAT = "%s?appId=%s&osVersion=%s&appVersion=%s&deviceId=%s";

    private static final String ID_HEADER_FORMAT = "%s:%s"; // [event type]:[story id]

    private final String screenshotUrl;

    private final String configBaseUrl;

    private final String coreBaseUrl;

    private final String appId;

    private final String deviceId;

    private final String operatingSystemCode = String.format(Locale.US, "%d.0.0", MetadataUtils.getOperatingSystemVersionCode());

    private final String appVersion;

    private final BackgroundWorker worker = BackgroundWorker.ofSingleThread("API Client");

    private final Gson gson = new Gson();

    private final ScheduledWorker retryWorker = ScheduledWorker.ofSingleThread("API Retry");

    private final ApiClientRetryWorker<SessionMessage> sessionRetryWorker;

    private final ApiClientRetryWorker<EventMessage> eventRetryWorker;

    /**
     * Creates an instance of the API client. This service handles all calls to the Embrace API.
     * <p>
     * Sessions can be sent to either the production or development endpoint. The development
     * endpoint shows sessions on the 'integration testing' screen within the dashboard, whereas
     * the production endpoint sends sessions to 'recent sessions'.
     * <p>
     * The development endpoint is only used if the build is a debug build, and if integration
     * testing is enabled when calling {@link Embrace#start(Context, boolean)} )}.
     *
     * @param metadataService          the metadata service
     * @param cacheService             the cache service
     * @param enableIntegrationTesting true if sessions should be sent to the integration testing
     *                                 screen in the dashboard for debug builds, false if they
     *                                 should be sent to 'recent sessions'
     */
    ApiClient(
            LocalConfig localConfig,
            MetadataService metadataService,
            CacheService cacheService,
            boolean enableIntegrationTesting) {

        Preconditions.checkNotNull(localConfig, "localConfig must not be null");
        Preconditions.checkNotNull(metadataService, "metadataService must not be null");
        coreBaseUrl = metadataService.isDebug()
                && enableIntegrationTesting
                && (Debug.isDebuggerConnected() || Debug.waitingForDebugger()) ?
                localConfig.getConfigurations().getBaseUrls().getDataDev() :
                localConfig.getConfigurations().getBaseUrls().getData();
        configBaseUrl = String.format(URL_FORMAT, localConfig.getConfigurations().getBaseUrls().getConfig(), CONFIG_API_VERSION, "config");
        screenshotUrl = String.format(URL_FORMAT, localConfig.getConfigurations().getBaseUrls().getImages(), API_VERSION, "screenshot");
        appId = metadataService.getAppId();

        appVersion = metadataService.getAppVersionCodeForRequest().or("");
        deviceId = metadataService.getDeviceId();
        sessionRetryWorker =
                new ApiClientRetryWorker<>(SessionMessage.class, cacheService, this, retryWorker);
        eventRetryWorker =
                new ApiClientRetryWorker<>(EventMessage.class, cacheService, this, retryWorker);
    }


    /**
     * Asynchronously gets the app's SDK configuration.
     * <p>
     * These settings define app-specific settings, such as disabled log patterns, whether
     * screenshots are enabled, as well as limits and thresholds.
     *
     * @return a future containing the configuration.
     */
    Future<Config> getConfig() {
        String url = String.format(CONFIG_URL_FORMAT, configBaseUrl, appId, operatingSystemCode, appVersion, deviceId);
        final EmbraceConnection connection = ApiRequest.newBuilder()
                .withHttpMethod(HttpMethod.GET)
                .withUrl(Unchecked.wrap(() -> EmbraceUrl.getUrl(url)))
                .build()
                .toConnection();
        return worker.submit(() -> {
            connection.connect();
            return httpCall(connection, Config.class);
        });

    }

    /**
     * Asynchronously sends a screenshot corresponding to a log entry.
     *
     * @param screenshot the screenshot payload
     * @param logId      the ID of the corresponding log
     * @return a future containing the path to the screenshot on the remote server
     */
    Future<String> sendLogScreenshot(byte[] screenshot, String logId) {
        EmbraceUrl url = Unchecked.wrap(() ->
                EmbraceUrl.getUrl(String.format(SCREENSHOT_URL_FORMAT, screenshotUrl, appId, "logs", logId)));
        ApiRequest request = screenshotBuilder(url)
                .withLogId(logId)
                .build();
        return rawPost(request, screenshot);
    }

    /**
     * Sends a screenshot corresponding to a moment.
     *
     * @param screenshot the screenshot payload
     * @param eventId    the ID of the moment
     * @return a future containing the path to the screenshot on the remote server
     */
    Future<String> sendMomentScreenshot(byte[] screenshot, String eventId) {
        EmbraceUrl url = Unchecked.wrap(() ->
                EmbraceUrl.getUrl(String.format(SCREENSHOT_URL_FORMAT, screenshotUrl, appId, "moments", eventId)));
        ApiRequest request = screenshotBuilder(url)
                .withEventId(eventId)
                .build();
        return rawPost(request, screenshot);
    }

    /**
     * Sends a session message to the API.
     *
     * @param sessionMessage the session to send
     * @return a future containing the response body from the server
     */
    Future<String> sendSession(SessionMessage sessionMessage) {
        EmbraceUrl url = Unchecked.wrap(() -> EmbraceUrl.getUrl(String.format(
                URL_FORMAT,
                coreBaseUrl,
                API_VERSION,
                "log/sessions")));
        ApiRequest request = eventBuilder(url)
                .withDeviceId(deviceId)
                .withAppId(appId)
                .withUrl(url)
                .withHttpMethod(HttpMethod.POST)
                .withContentEncoding("gzip")
                .build();
        return jsonPost(request, sessionMessage, SessionMessage.class, sessionRetryWorker::addFailedCall);
    }

    /**
     * Sends a log message to the API.
     *
     * @param eventMessage the event message containing the log entry
     * @return a future containing the response body from the server
     */
    Future<String> sendLogs(EventMessage eventMessage) {
        Preconditions.checkNotNull(eventMessage.getEvent(), "event must be set");
        Event event = eventMessage.getEvent();
        Preconditions.checkNotNull(event.getType(), "event type must be set");
        Preconditions.checkNotNull(event.getEventId(), "event ID must be set");
        EmbraceUrl url = Unchecked.wrap(() -> EmbraceUrl.getUrl(String.format(
                URL_FORMAT,
                coreBaseUrl,
                API_VERSION,
                "log/logging")));
        String abbreviation = event.getType().getAbbreviation();
        String logIdentifier = String.format(ID_HEADER_FORMAT, abbreviation, event.getMessageId());
        ApiRequest request = eventBuilder(url)
                .withLogId(logIdentifier)
                .build();
        return sendEvent(eventMessage, request);
    }

    /**
     * Sends an event to the API.
     *
     * @param eventMessage the event message containing the event
     * @return a future containing the response body from the server
     */
    Future<String> sendEvent(EventMessage eventMessage) {
        Preconditions.checkNotNull(eventMessage.getEvent(), "event must be set");
        Event event = eventMessage.getEvent();
        Preconditions.checkNotNull(event.getType(), "event type must be set");
        Preconditions.checkNotNull(event.getEventId(), "event ID must be set");
        EmbraceUrl url = Unchecked.wrap(() -> EmbraceUrl.getUrl(String.format(
                URL_FORMAT,
                coreBaseUrl,
                API_VERSION,
                "log/events")));
        String abbreviation = event.getType().getAbbreviation();
        String eventIdentifier;
        if (event.getType().equals(EmbraceEvent.Type.CRASH)) {
            eventIdentifier = createCrashActiveEventsHeader(abbreviation, event.getActiveEventIds());
        } else {
            eventIdentifier = String.format(
                    ID_HEADER_FORMAT,
                    abbreviation,
                    event.getEventId());
        }
        ApiRequest request = eventBuilder(url)
                .withEventId(eventIdentifier)
                .build();
        return sendEvent(eventMessage, request);
    }

    /**
     * Posts a JSON payload to the API.
     * <p>
     * If the request fails for any reason, such as a non-200 response, or an exception being thrown
     * during the connection, the failer handler will be invoked. This can be used to re-attempt
     * sending the message.
     *
     * @param request        the details of the API request
     * @param payload        the body of the API request
     * @param clazz          the class of the payload
     * @param failureHandler callback if the request fails
     * @param <T>            the type of the payload
     * @return a future containing the response body from the server
     */
    <T> Future<String> jsonPost(
            ApiRequest request,
            T payload,
            Class<T> clazz,
            BiConsumer<ApiRequest, T> failureHandler) {

        return worker.submit(() -> {
            try {
                final byte[] bytes = gson.toJson(payload, clazz.getGenericSuperclass()).getBytes(Charsets.UTF_8);
                return sendBytes(request, gzip(bytes));
            } catch (Exception ex) {
                EmbraceLogger.logDebug("Failed to post Embrace API call. Will retry.");
                failureHandler.accept(request, payload);
                throw Unchecked.propagate(ex);
            }
        });
    }

    private Future<String> sendEvent(EventMessage eventMessage, ApiRequest request) {
        return jsonPost(request, eventMessage, EventMessage.class, eventRetryWorker::addFailedCall);
    }

    private Future<String> rawPost(ApiRequest request, byte[] payload) {
        return worker.submit(() -> sendBytes(request, payload));
    }

    private String sendBytes(ApiRequest request, byte[] payload) {
        return Unchecked.wrap(() -> {
            EmbraceConnection connection = request.toConnection();
            if (payload != null) {
                OutputStream outputStream = connection.getOutputStream();
                outputStream.write(payload);
                connection.connect();
            }
            return httpCall(connection);
        });
    }

    /**
     * Compresses a given byte array using the GZIP compression algorithm.
     *
     * @param bytes the byte array to compress
     * @return the compressed byte array
     */
    private static byte[] gzip(byte[] bytes) {
        ByteArrayOutputStream baos = null;
        GZIPOutputStream os = null;
        try {
            baos = new ByteArrayOutputStream();
            os = new GZIPOutputStream(baos);
            os.write(bytes);
            os.finish();
            return baos.toByteArray();
        } catch (IOException ex) {
            throw Unchecked.propagate(ex);
        } finally {
            try {
                if (baos != null) {
                    baos.close();
                }
                if (os != null) {
                    os.close();
                }
            } catch (IOException ex) {
                EmbraceLogger.logDebug("Failed to close streams when gzipping payload", ex);
            }
        }
    }

    /**
     * Executes a HTTP call using the specified connection, returning the JSON response from the
     * server as an object of the specified type.
     *
     * @param connection the HTTP connection to use
     * @param clazz      the class representing the type of response
     * @param <T>        the type of the object being returned
     * @return the Java representation of the JSON object
     */
    private <T> T httpCall(EmbraceConnection connection, Class<T> clazz) {
        handleHttpResponseCode(connection);
        return readJsonObject(connection, clazz);
    }

    /**
     * Executes a HTTP call using the specified connection, returning the response from the server
     * as a string.
     *
     * @param connection the HTTP connection to use
     * @return a string containing the response from the server to the request
     */
    private String httpCall(EmbraceConnection connection) {
        handleHttpResponseCode(connection);
        return Unchecked.wrap(() -> readString(connection.getInputStream()));
    }

    /**
     * Checks that the HTTP response was 200, otherwise throws an exception.
     * <p>
     * Attempts to read the error message payload from the response, if present, and writes it to
     * the logs at warning level.
     *
     * @param connection the connection containing the HTTP response
     */
    private void handleHttpResponseCode(EmbraceConnection connection) {
        Integer responseCode = null;
        try {
            responseCode = connection.getResponseCode();
        } catch (IOException ex) {
            // Connection failed or unexpected response code
        }

        if (responseCode == null || responseCode != HttpURLConnection.HTTP_OK) {
            if (responseCode != null) {
                String errorMessage = readString(connection.getErrorStream());
                EmbraceLogger.logDebug("Embrace API request failed. HTTP response code: " + responseCode + ", message: " + errorMessage);
            }
            throw new IllegalStateException("Failed to retrieve from Embrace server.");
        }
    }

    /**
     * Reads the JSON response payload of a {@link HttpURLConnection} and converts it to a Java
     * object of the specified type.
     *
     * @param connection the connection to read
     * @param clazz      the class representing the type
     * @param <T>        the type of object to return
     * @return the Java representation of the JSON object
     */
    private <T> T readJsonObject(EmbraceConnection connection, Class<T> clazz) {
        InputStreamReader inputStreamReader = null;
        JsonReader reader = null;
        try {
            inputStreamReader = new InputStreamReader(connection.getInputStream());
            reader = new JsonReader(inputStreamReader);
            return gson.fromJson(reader, clazz);
        } catch (IOException ex) {
            throw Unchecked.propagate(ex);
        } finally {
            try {
                if (reader != null) {
                    reader.close();
                }
                if (inputStreamReader != null) {
                    inputStreamReader.close();
                }
            } catch (IOException ex) {
                EmbraceLogger.logDebug("Failed to close HTTP reader", ex);
            }
        }
    }

    /**
     * Reads an {@link InputStream} into a String.
     *
     * @param inputStream the input stream to read
     * @return the string representation
     */
    private String readString(InputStream inputStream) {
        InputStreamReader inputStreamReader = null;
        BufferedReader reader = null;
        try {
            inputStreamReader = new InputStreamReader(inputStream);
            reader = new BufferedReader(inputStreamReader);
            char[] buffer = new char[4096];
            StringBuilder sb = new StringBuilder();
            for (int len; (len = reader.read(buffer)) > 0; )
                sb.append(buffer, 0, len);
            return sb.toString();
        } catch (IOException ex) {
            throw Unchecked.propagate(ex);
        } finally {
            try {
                if (reader != null) {
                    reader.close();
                }
                if (inputStreamReader != null) {
                    inputStreamReader.close();
                }
            } catch (IOException ex) {
                EmbraceLogger.logDebug("Failed to close HTTP reader", ex);
            }
        }
    }

    private ApiRequest.Builder screenshotBuilder(EmbraceUrl url) {
        return ApiRequest.newBuilder()
                .withContentType("application/octet-stream")
                .withHttpMethod(HttpMethod.POST)
                .withAppId(appId)
                .withDeviceId(deviceId)
                .withUrl(url);
    }

    private ApiRequest.Builder eventBuilder(EmbraceUrl url) {
        return ApiRequest.newBuilder()
                .withUrl(url)
                .withHttpMethod(HttpMethod.POST)
                .withAppId(appId)
                .withDeviceId(deviceId)
                .withContentEncoding("gzip");
    }

    @Override
    public void close() {
        EmbraceLogger.logDebug("Shutting down ApiClient");
        retryWorker.close();
        worker.close();
    }

    /**
     * Crashes are sent with a header containing the list of active stories.
     *
     * @param abbreviation the abbreviation for the event type
     * @param eventIds     the list of story IDs
     * @return the header
     */
    private String createCrashActiveEventsHeader(String abbreviation, List<String> eventIds) {
        String stories = "";
        if (eventIds != null) {
            stories = StreamSupport.stream(eventIds)
                    .collect(Collectors.joining(","));
        }
        return String.format(
                ID_HEADER_FORMAT,
                abbreviation,
                stories);
    }
}
