package io.embrace.android.embracesdk;

import com.fernandocejas.arrow.optional.Optional;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

class EmbraceSessionProperties {

    private final PreferencesService preferencesService;

    private final Map<String, String> temporary;
    private final Map<String, String> permanent;

    /**
     * The maximum number of properties that can be attached to a session
     */
    private static final int SESSION_PROPERTY_LIMIT = 10;
    private static final int SESSION_PROPERTY_KEY_LIMIT = 128;
    private static final int SESSION_PROPERTY_VALUE_LIMIT = 1024;

    EmbraceSessionProperties(PreferencesService preferencesService) {
        this.preferencesService = preferencesService;
        this.temporary = new HashMap<>();

        Optional<Map<String, String>> existingPermanent = preferencesService.getPermanentSessionProperties();

        if (existingPermanent.isPresent()) {
            this.permanent = new HashMap<>(existingPermanent.get());
        } else {
            this.permanent = new HashMap<>();
        }
    }

    private boolean haveKey(String key) {
        return permanent.containsKey(key) || temporary.containsKey(key);
    }

    private boolean isValidKey(String key) {
        if (key == null) {
            EmbraceLogger.logError("Session property key cannot be null");
            return false;
        }
        if (key.equals("")) {
            EmbraceLogger.logError("Session property key cannot be empty string");
            return false;
        }
        return true;
    }

    private boolean isValidValue(String key) {
        if (key == null) {
            EmbraceLogger.logError("Session property value cannot be null");
            return false;
        }
        return true;
    }

    private String enforceLength(String value, int maxLength) {
        if (value.length() <= maxLength) {
            return value;
        }

        String endChars = "...";
        return String.format("%s%s", value.substring(0, maxLength - endChars.length()), endChars);
    }

    synchronized boolean add(String key, String value, boolean isPermanent) {
        if (!isValidKey(key)) {
            return false;
        }
        key = enforceLength(key, SESSION_PROPERTY_KEY_LIMIT);

        if (!isValidValue(value)) {
            return false;
        }
        value = enforceLength(value, SESSION_PROPERTY_VALUE_LIMIT);

        if (size() > SESSION_PROPERTY_LIMIT || (size() == SESSION_PROPERTY_LIMIT && !haveKey(key))) {
            EmbraceLogger.logError("Session property count is at its limit. Rejecting.");
            return false;
        }

        // add to selected destination, deleting the key if it exists in the other destination
        if (isPermanent) {
            permanent.put(key, value);
            temporary.remove(key);
            this.preferencesService.setPermanentSessionProperties(permanent);
        } else {
            // only save the permanent values if the key existed in the permanent map
            if (permanent.remove(key) != null) {
                this.preferencesService.setPermanentSessionProperties(permanent);
            }
            temporary.put(key, value);
        }

        return true;
    }

    synchronized boolean remove(String key) {
        if (!isValidKey(key)) {
            return false;
        }
        key = enforceLength(key, SESSION_PROPERTY_KEY_LIMIT);

        boolean existed = false;
        if (temporary.remove(key) != null) {
            existed = true;
        }
        if (permanent.remove(key) != null) {
            this.preferencesService.setPermanentSessionProperties(permanent);
            existed = true;
        }

        return existed;
    }

    synchronized Map<String, String> get() {
        Map<String, String> combined = new HashMap<>();
        combined.putAll(permanent);
        combined.putAll(temporary);
        return Collections.unmodifiableMap(combined);
    }

    private int size() {
        return permanent.size() + temporary.size();
    }

    void clearTemporary() {
        temporary.clear();
    }
}
