package io.embrace.android.embracesdk;

import com.google.gson.annotations.SerializedName;

/**
 * The session message, containing the session itself, as well as performance information about the
 * device which occurred during the session.
 */
public final class SessionMessage {

    /** The session information. */
    @SerializedName("s")
    private final Session session;

    /** The user information. */
    @SerializedName("u")
    private final UserInfo userInfo;

    /** The app information. */
    @SerializedName("a")
    private final AppInfo appInfo;

    /** The device information. */
    @SerializedName("d")
    private final DeviceInfo deviceInfo;

    /** The device's performance info, such as power, cpu, network. */
    @SerializedName("p")
    private final SessionPerformanceInfo performanceInfo;

    /** Breadcrumbs which occurred as part of this session. */
    @SerializedName("br")
    private final Breadcrumbs breadcrumbs;

    @SerializedName("v")
    private final int version = ApiClient.MESSAGE_VERSION;

    private SessionMessage(Builder builder) {
        session = builder.session;
        userInfo = builder.userInfo;
        appInfo = builder.appInfo;
        deviceInfo = builder.deviceInfo;
        performanceInfo = builder.performanceInfo;
        breadcrumbs = builder.breadcrumbs;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static Builder newBuilder(SessionMessage copy) {
        Builder builder = new Builder();
        builder.session = copy.getSession();
        builder.userInfo = copy.getUserInfo();
        builder.appInfo = copy.getAppInfo();
        builder.deviceInfo = copy.getDeviceInfo();
        builder.performanceInfo = copy.getPerformanceInfo();
        builder.breadcrumbs = copy.getBreadcrumbs();
        return builder;
    }


    public Session getSession() {
        return session;
    }

    public UserInfo getUserInfo() {
        return userInfo;
    }

    public AppInfo getAppInfo() {
        return appInfo;
    }

    public DeviceInfo getDeviceInfo() {
        return deviceInfo;
    }

    public SessionPerformanceInfo getPerformanceInfo() {
        return performanceInfo;
    }

    public Breadcrumbs getBreadcrumbs() {
        return breadcrumbs;
    }


    public static final class Builder {
        private Session session;
        private UserInfo userInfo;
        private AppInfo appInfo;
        private DeviceInfo deviceInfo;
        private SessionPerformanceInfo performanceInfo;
        private Breadcrumbs breadcrumbs;

        private Builder() {
        }

        public Builder withSession(Session session) {
            this.session = session;
            return this;
        }

        public Builder withUserInfo(UserInfo userInfo) {
            this.userInfo = userInfo;
            return this;
        }

        public Builder withAppInfo(AppInfo appInfo) {
            this.appInfo = appInfo;
            return this;
        }

        public Builder withDeviceInfo(DeviceInfo deviceInfo) {
            this.deviceInfo = deviceInfo;
            return this;
        }

        public Builder withPerformanceInfo(SessionPerformanceInfo performanceInfo) {
            this.performanceInfo = performanceInfo;
            return this;
        }

        public Builder withBreadcrumbs(Breadcrumbs breadcrumbs) {
            this.breadcrumbs = breadcrumbs;
            return this;
        }

        public SessionMessage build() {
            return new SessionMessage(this);
        }
    }
}
