package io.embrace.android.embracesdk;

import com.google.gson.annotations.SerializedName;

import java.util.List;

/**
 * Contains lists of {@link ViewBreadcrumb}, {@link TapBreadcrumb}, {@link CustomBreadcrumb},
 * and {@link WebViewBreadcrumb} within a particular time window, created by the
 * {@link EmbraceBreadcrumbService}.
 * <p>
 * Breadcrumbs are used to track user journeys throughout the apps, such as transitions between
 * screens or taps on particular UI elements. A developer can create a {@link CustomBreadcrumb} if
 * they would like to label some particular event or interaction within their app on the timeline.
 */
class Breadcrumbs {

    /** List of breadcrumbs which relate to views. */
    @SerializedName("vb")
    private final List<ViewBreadcrumb> viewBreadcrumbs;

    /** List of breadcrumbs which relate to screen taps. */
    @SerializedName("tb")
    private final List<TapBreadcrumb> tapBreadcrumbs;

    /** List of custom breadcrumbs defined by the developer. */
    @SerializedName("cb")
    private final List<CustomBreadcrumb> customBreadcrumbs;

    /** List of webview breadcrumbs. */
    @SerializedName("wv")
    private final List<WebViewBreadcrumb> webViewBreadcrumbs;

    /** List of fragment (custom view) breadcrumbs. */
    @SerializedName("cv")
    private final List<FragmentBreadcrumb> fragmentBreadcrumbs;

    private Breadcrumbs(Builder builder) {
        viewBreadcrumbs = builder.viewBreadcrumbs;
        tapBreadcrumbs = builder.tapBreadcrumbs;
        customBreadcrumbs = builder.customBreadcrumbs;
        webViewBreadcrumbs = builder.webViewBreadcrumbs;
        fragmentBreadcrumbs = builder.fragmentBreadcrumbs;
    }

    static Builder newBuilder() {
        return new Builder();
    }


    static final class Builder {
        private List<ViewBreadcrumb> viewBreadcrumbs;
        private List<TapBreadcrumb> tapBreadcrumbs;
        private List<CustomBreadcrumb> customBreadcrumbs;
        private List<WebViewBreadcrumb> webViewBreadcrumbs;
        private List<FragmentBreadcrumb> fragmentBreadcrumbs;

        private Builder() {
        }

        public Builder withViewBreadcrumbs(List<ViewBreadcrumb> viewBreadcrumbs) {
            this.viewBreadcrumbs = viewBreadcrumbs;
            return this;
        }

        public Builder withTapBreadcrumbs(List<TapBreadcrumb> tapBreadcrumbs) {
            this.tapBreadcrumbs = tapBreadcrumbs;
            return this;
        }

        public Builder withCustomBreadcrumbs(List<CustomBreadcrumb> customBreadcrumbs) {
            this.customBreadcrumbs = customBreadcrumbs;
            return this;
        }

        public Builder withWebViewBreadcrumbs(List<WebViewBreadcrumb> webViewBreadcrumbs) {
            this.webViewBreadcrumbs = webViewBreadcrumbs;
            return this;
        }

        public Builder withFragmentBreadcrumbs(List<FragmentBreadcrumb> fragmentBreadcrumbs) {
            this.fragmentBreadcrumbs = fragmentBreadcrumbs;
            return this;
        }

        public Breadcrumbs build() {
            return new Breadcrumbs(this);
        }
    }
}
