package io.embrace.android.embracesdk;

import com.google.gson.annotations.SerializedName;

import java.util.List;

/**
 * Describes information about how the device is performing.
 */
abstract class PerformanceInfo {
    /** Current battery level of the device. */
    @SerializedName("bt")
    private final float batteryLevel;

    /** Current disk space usage of the app, and free space on the device. */
    @SerializedName("ds")
    private final DiskUsage diskUsage;

    /** Memory usage at various points in time on the device. */
    @SerializedName("me")
    private final List<MemorySample> memorySamples;

    /** Occasions where the device reported that the memory is running low. */
    @SerializedName("mw")
    private final List<MemoryWarning> memoryWarnings;

    /** Periods during which the device was charging. */
    @SerializedName("bc")
    private final List<Interval> chargingIntervals;

    /** Periods during which the CPU usage was high. */
    @SerializedName("cp")
    private final List<Interval> criticalCpuIntervals;

    /** Periods during which the device was connected to WIFI, WAN, or no network. */
    @SerializedName("ns")
    private final List<Interval> networkInterfaceIntervals;

    /** Measurements of the battery level at various points in time. */
    @SerializedName("bm")
    private final List<BatteryMeasurement> batteryMeasurements;

    /** Periods during which the application was not responding (UI thread blocked for > 1 sec). */
    @SerializedName("anr")
    private final List<AnrInterval> anrIntervals;

    /** Measurements of signal strength on each change. */
    @SerializedName("ss")
    private final List<SignalStrength> signalStrength;

    /** Measurements of the connection quality at various points in time. */
    @SerializedName("cq")
    private final List<ConnectionQualityInterval> connectionQualityIntervals;


    protected PerformanceInfo(Builder builder) {
        batteryLevel = builder.batteryLevel;
        diskUsage = builder.diskUsage;
        memorySamples = builder.memorySamples;
        memoryWarnings = builder.memoryWarnings;
        chargingIntervals = builder.chargingIntervals;
        criticalCpuIntervals = builder.criticalCpuIntervals;
        networkInterfaceIntervals = builder.networkInterfaceIntervals;
        batteryMeasurements = builder.batteryMeasurements;
        anrIntervals = builder.anrIntervals;
        signalStrength = builder.signalStrength;
        connectionQualityIntervals = builder.connectionQualityIntervals;
    }



    abstract static class Builder {
        private float batteryLevel;
        private DiskUsage diskUsage;
        private List<MemorySample> memorySamples;
        private List<MemoryWarning> memoryWarnings;
        private List<Interval> chargingIntervals;
        private List<Interval> criticalCpuIntervals;
        private List<Interval> networkInterfaceIntervals;
        private List<BatteryMeasurement> batteryMeasurements;
        private List<AnrInterval> anrIntervals;
        private List<SignalStrength> signalStrength;
        private List<ConnectionQualityInterval> connectionQualityIntervals;

        protected Builder() {
        }

        public Builder withBatteryLevel(float batteryLevel) {
            this.batteryLevel = batteryLevel;
            return this;
        }

        public Builder withDiskUsage(DiskUsage diskUsage) {
            this.diskUsage = diskUsage;
            return this;
        }

        public Builder withMemorySamples(List<MemorySample> memorySamples) {
            this.memorySamples = memorySamples;
            return this;
        }

        public Builder withMemoryWarnings(List<MemoryWarning> memoryWarnings) {
            this.memoryWarnings = memoryWarnings;
            return this;
        }

        public Builder withChargingIntervals(List<Interval> chargingIntervals) {
            this.chargingIntervals = chargingIntervals;
            return this;
        }

        public Builder withCriticalCpuIntervals(List<Interval> criticalCpuIntervals) {
            this.criticalCpuIntervals = criticalCpuIntervals;
            return this;
        }

        public Builder withNetworkInterfaceIntervals(List<Interval> networkInterfaceIntervals) {
            this.networkInterfaceIntervals = networkInterfaceIntervals;
            return this;
        }

        public Builder withBatteryMeasurements(List<BatteryMeasurement> batteryMeasurements) {
            this.batteryMeasurements = batteryMeasurements;
            return this;
        }

        public Builder withAnrIntervals(List<AnrInterval> anrIntervals) {
            this.anrIntervals = anrIntervals;
            return this;
        }

        public Builder withSignalStrength(List<SignalStrength> signalStrength) {
            this.signalStrength = signalStrength;
            return this;
        }

        public Builder withConnectionQualityIntervals(List<ConnectionQualityInterval> connectionQualityIntervals) {
            this.connectionQualityIntervals = connectionQualityIntervals;
            return this;
        }
    }
}
