package io.embrace.android.embracesdk;

import android.app.Activity;

import com.fernandocejas.arrow.checks.Preconditions;
import com.fernandocejas.arrow.optional.Optional;

import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import io.embrace.android.embracesdk.utils.exceptions.Unchecked;

import static io.embrace.android.embracesdk.EmbracePreferencesService.SDK_STARTUP_COMPLETED;
import static io.embrace.android.embracesdk.Session.Builder;
import static io.embrace.android.embracesdk.Session.SessionLifeEventType;
import static io.embrace.android.embracesdk.Session.newBuilder;

/**
 * Handles the lifecycle of an Embrace session.
 * <p>
 * A session encapsulates all metrics gathered between the app being foregrounded and the app being
 * backgrounded. A caching service runs on an interval of every
 * {@value EmbraceSessionService#SESSION_CACHING_INTERVAL} seconds which saves the current session
 * to disk, so if the app is terminated, the session can resume from where it left off.
 */
final class EmbraceSessionService implements SessionService, ActivityListener {

    /**
     * Signals to the API that the application was in the foreground.
     */
    public static final String APPLICATION_STATE_ACTIVE = "active";
    /**
     * Signals to the API that the application was in the background.
     */
    public static final String APPLICATION_STATE_BACKGROUND = "background";

    /**
     * The minimum threshold for how long a session must last. Package-private for test accessibility
     */
    static Long minSessionTime = 5000L;
    /**
     * Signals to the API the start of a session.
     */
    private static final String SESSION_START_TYPE = "st";
    /**
     * Signals to the API the end of a session.
     */
    private static final String SESSION_END_TYPE = "en";
    /**
     * The name of the file containing the cached session.
     */
    private static final String SESSION_FILE_NAME = "last_session.json";
    /**
     * Session caching interval in seconds.
     */
    private static final int SESSION_CACHING_INTERVAL = 2;

    /**
     * ApiClient timeout for the active session send service
     */
    private static final int SEND_SESSION_API_CLIENT_TIMEOUT = 2;
    /**
     * Synchronization lock.
     */
    private final Object lock = new Object();
    /**
     * Embrace service dependencies of the session service.
     */
    private final PreferencesService preferencesService;
    private final PerformanceInfoService performanceInfoService;
    private final MetadataService metadataService;
    private final NetworkConnectivityService networkConnectivityService;
    private final BreadcrumbService breadcrumbService;
    private final PowerService powerService;
    private final ActivityService activityService;
    private final ApiClient apiClient;
    private final EventService eventService;
    private final EmbraceRemoteLogger remoteLogger;
    private final UserService userService;
    private final ConfigService configService;
    private final CacheService cacheService;
    private final EmbraceExceptionService exceptionService;
    private final LocalConfig localConfig;
    private final MemoryCleanerService memoryCleanerService;
    private final OrientationService orientationService;
    private final NdkService ndkService;

    /**
     * Asynchronous workers.
     */
    private final BackgroundWorker sessionBackgroundWorker;
    private final BackgroundWorker nativeCrashSendBackgroundWorker;
    private final BackgroundWorker nativeCrashSearchBackgroundWorker;
    private volatile ScheduledWorker sessionCacheWorker;
    private volatile ScheduledWorker automaticSessionCloser;

    /**
     * The currently active session.
     */
    private volatile Session activeSession;

    /**
     * Session properties
     */
    private final EmbraceSessionProperties sessionProperties;

    /**
     * Allows to the last session message to be send to backend.
     * Is set in true as default until the native crash searching is finished.
     */
    private boolean isLookingForNativeCrash;

    /**
     * Synchronization lock to handle native crash searching.
     */
    private final Lock nativeCrashSearchLock = new ReentrantLock();
    private final Condition searchEndedCondition = nativeCrashSearchLock.newCondition();

    EmbraceSessionService(
            PreferencesService preferencesService,
            PerformanceInfoService performanceInfoService,
            MetadataService metadataService,
            NetworkConnectivityService networkConnectivityService,
            BreadcrumbService breadcrumbService,
            PowerService powerService,
            ActivityService activityService,
            ApiClient apiClient,
            EventService eventService,
            EmbraceRemoteLogger remoteLogger,
            UserService userService,
            ConfigService configService,
            CacheService cacheService,
            EmbraceExceptionService exceptionService,
            LocalConfig localConfig,
            MemoryCleanerService memoryCleanerService,
            OrientationService orientationService,
            NdkService ndkService,
            EmbraceSessionProperties sessionProperties) {

        this.preferencesService = Preconditions.checkNotNull(preferencesService);
        this.performanceInfoService = Preconditions.checkNotNull(performanceInfoService);
        this.metadataService = Preconditions.checkNotNull(metadataService);
        this.networkConnectivityService = Preconditions.checkNotNull(networkConnectivityService);
        this.breadcrumbService = Preconditions.checkNotNull(breadcrumbService);
        this.powerService = Preconditions.checkNotNull(powerService);
        this.activityService = Preconditions.checkNotNull(activityService);
        this.apiClient = Preconditions.checkNotNull(apiClient);
        this.eventService = Preconditions.checkNotNull(eventService);
        this.remoteLogger = Preconditions.checkNotNull(remoteLogger);
        this.userService = Preconditions.checkNotNull(userService);
        this.configService = Preconditions.checkNotNull(configService);
        this.cacheService = Preconditions.checkNotNull(cacheService);
        this.exceptionService = Preconditions.checkNotNull(exceptionService);
        this.activityService.addListener(this, true);
        this.sessionBackgroundWorker = BackgroundWorker.ofSingleThread("Session");
        this.nativeCrashSendBackgroundWorker = BackgroundWorker.ofSingleThread("Native Crash Send");
        this.nativeCrashSearchBackgroundWorker = BackgroundWorker.ofSingleThread("Native Crash Search");
        this.localConfig = Preconditions.checkNotNull(localConfig);
        this.memoryCleanerService = Preconditions.checkNotNull(memoryCleanerService);
        this.orientationService = Preconditions.checkNotNull(orientationService);
        this.ndkService = Preconditions.checkNotNull(ndkService);

        // Session properties
        this.sessionProperties = Preconditions.checkNotNull(sessionProperties);

        // Check if NDK crash capture is enabled
        this.isLookingForNativeCrash = localConfig.isNdkEnabled();
    }

    /**
     * @return session number incremented by 1
     */
    private int incrementAndGetSessionNumber() {
        if (preferencesService.getSessionNumber().isPresent()) {
            preferencesService.setSessionNumber(preferencesService.getSessionNumber().get() + 1);
        } else {
            preferencesService.setSessionNumber(1);
        }
        return preferencesService.getSessionNumber().get();
    }

    private void startCaching() {
        this.sessionCacheWorker = ScheduledWorker.ofSingleThread("Session Caching Service");
        this.sessionCacheWorker.scheduleAtFixedRate(
                this::cacheActiveSession,
                0,
                SESSION_CACHING_INTERVAL,
                TimeUnit.SECONDS);
    }

    private void stopCaching() {
        if (this.sessionCacheWorker != null) {
            this.sessionCacheWorker.close();
        }
    }

    private void startSessionCloser(int maxSessionSeconds) {
        if (localConfig.getConfigurations().getSessionConfig().getAsyncEnd() || configService.getConfig().endSessionInBackgroundThread()) {
            EmbraceLogger.logWarning("Can't close the session. Automatic session closing disabled since async session send is enabled.");
            return;
        }

        this.automaticSessionCloser = ScheduledWorker.ofSingleThread("Session Closer Service");
        this.automaticSessionCloser.scheduleAtFixedRate(
                () -> {
                    try {
                        synchronized (lock) {
                            EmbraceLogger.logInfo("Automatic session closing triggered.");
                            triggerStatelessSessionEnd(SessionLifeEventType.TIMED);
                        }
                    } catch (Exception ex) {
                        EmbraceLogger.logError("Error while trying to close the session " +
                                "automatically", ex);
                    }
                },
                maxSessionSeconds, maxSessionSeconds, TimeUnit.SECONDS);
    }

    private void stopSessionCloser() {
        if (this.automaticSessionCloser != null) {
            this.automaticSessionCloser.close();
        }
    }

    @Override
    public void startSession(boolean coldStart, SessionLifeEventType startType) {

        if (configService.isMessageTypeDisabled(MessageType.SESSION)) {
            EmbraceLogger.logWarning("Session messages disabled. Ignoring all sessions.");
            return;
        }

        String sessionId = Uuid.getEmbUuid();
        SessionMessage message = sessionStartMessage(coldStart, sessionId, startType);
        this.activeSession = message.getSession();
        metadataService.setActiveSessionId(sessionId);
        apiClient.sendSession(message);

        //If getMaxSessionSeconds is not null, schedule the session closer.
        LocalConfig.SdkConfigs.SessionConfig sessionConfig = localConfig.getConfigurations().getSessionConfig();
        if (sessionConfig.getMaxSessionSeconds().isPresent()) {
            startSessionCloser(sessionConfig.getMaxSessionSeconds().get());
        }

        addViewBreadcrumbForResumedSession();
        startCaching();
        if (localConfig.isNdkEnabled()) {
            ndkService.updateSessionId(this.activeSession.getSessionId());
        }
    }

    @Override
    public boolean sdkStartupFailedLastSession() {
        Optional<String> lastSessionStartupProgress = preferencesService.getSDKStartupStatus();
        return lastSessionStartupProgress.isPresent() && !lastSessionStartupProgress.get().equals(SDK_STARTUP_COMPLETED);
    }

    @Override
    public void handleCrash(String crashId) {
        synchronized (lock) {
            if (this.activeSession != null) {
                SessionMessage sessionMessage = sessionEndMessage(
                        this.activeSession,
                        false,
                        false,
                        crashId,
                        SessionLifeEventType.STATE);
                Unchecked.wrap(() -> apiClient.sendSession(sessionMessage).get());
                cacheService.deleteObject(SESSION_FILE_NAME);
            }
        }
    }

    private SessionMessage sessionStartMessage(boolean coldStart, String id, SessionLifeEventType startType) {
        Builder builder = newBuilder()
                .withSessionId(id)
                .withStartTime(System.currentTimeMillis())
                .withNumber(incrementAndGetSessionNumber())
                .withColdStart(coldStart)
                .withStartType(startType)
                .withOrientations(orientationService.getOrientations())
                .withProperties(sessionProperties)
                .withSessionType(SESSION_START_TYPE)
                .withStartingBatteryLevel(powerService.getLatestBatteryLevel())
                .withUserInfo(userService.loadUserInfoFromDisk());

        Session session = builder.build();

        // Record the connection type at the start of the session.
        networkConnectivityService.networkStatusOnSessionStarted(session.getStartTime());

        return SessionMessage.newBuilder()
                .withSession(session)
                .withDeviceInfo(metadataService.getDeviceInfo())
                .withAppInfo(metadataService.getAppInfo())
                .build();
    }

    /**
     * Caches the session, with performance information generated up to the current point.
     */
    private void cacheActiveSession() {
        try {
            synchronized (lock) {
                if (activeSession != null) {
                    cacheSessionMessage(sessionEndMessage(activeSession, false, true, SessionLifeEventType.STATE));
                }
            }
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Error while caching active session", ex);
        }
    }

    private SessionMessage sessionEndMessage(Session session,
                                             boolean endedCleanly,
                                             boolean forceQuit,
                                             SessionLifeEventType endType) {
        return this.sessionEndMessage(session, endedCleanly, forceQuit, null, endType);
    }

    private SessionMessage sessionEndMessage(Session session,
                                             boolean endedCleanly,
                                             boolean forceQuit,
                                             String crashId,
                                             SessionLifeEventType endType) {

        Long startTime = session.getStartTime();
        long endTime = System.currentTimeMillis();
        Builder builder = newBuilder(session)
                .withEndedCleanly(endedCleanly)
                .withLastState(getApplicationState())
                .withSessionType(SESSION_END_TYPE)
                .withEventIds(eventService.findEventIdsForSession(startTime, endTime))
                .withInfoLogIds(remoteLogger.findInfoLogIds(startTime, endTime))
                .withWarningLogIds(remoteLogger.findWarningLogIds(startTime, endTime))
                .withErrorLogIds(remoteLogger.findErrorLogIds(startTime, endTime))
                .withExceptionErrors(exceptionService.getCurrentExceptionError())
                .withLastHeartbeatTime(System.currentTimeMillis())
                .withOrientations(orientationService.getOrientations())
                .withProperties(sessionProperties)
                .withEndType(endType);

        if (crashId != null && !crashId.isEmpty()) {
            // if it's a crash session, then add the stacktrace to the session payload
            builder.withCrashReportId(crashId);
        }
        if (forceQuit) {
            builder
                    .withTerminationTime(System.currentTimeMillis())
                    .withReceivedTermination(true);
        } else {
            // We don't set end time for force-quit, as the API interprets this to be a clean
            // termination
            builder.withEndTime(endTime);
        }

        StartupEventInfo startupEventInfo = eventService.getStartupMomentInfo();
        if (startupEventInfo != null && session.isColdStart()) {
            builder.withStartupDuration(startupEventInfo.getDuration());
            builder.withStartupThreshold(startupEventInfo.getThreshold());
        }

        Session currentSession = builder.build();

        return SessionMessage.newBuilder()
                .withUserInfo(currentSession.getUser())
                .withAppInfo(metadataService.getAppInfo())
                .withDeviceInfo(metadataService.getDeviceInfo())
                .withPerformanceInfo(performanceInfoService.getSessionPerformanceInfo(startTime, endTime))
                .withBreadcrumbs(breadcrumbService.getBreadcrumbs(startTime, endTime))
                .withSession(currentSession)
                .build();
    }

    private String getApplicationState() {
        return activityService.isInBackground() ? APPLICATION_STATE_BACKGROUND : APPLICATION_STATE_ACTIVE;
    }

    @Override
    public void onForeground(boolean coldStart, long startupTime) {
        synchronized (lock) {
            if (coldStart) {
                sendCachedSession();
            }
            startSession(coldStart, SessionLifeEventType.STATE);
        }
    }

    @Override
    public void onBackground() {
        endSession(SessionLifeEventType.STATE);
    }

    @Override
    public void triggerStatelessSessionEnd(SessionLifeEventType endType) {

        if (!configService.isSessionControlEnabled()) {
            EmbraceLogger.logWarning("Session control disabled from remote configuration.");
            return;
        }

        if (activeSession == null) {
            EmbraceLogger.logError("There's no active session to end.");
            return;
        }

        // If less than 5 seconds, then the session cannot be finished manually.
        if (System.currentTimeMillis() - activeSession.getStartTime() < minSessionTime) {
            EmbraceLogger.logError("The session has to be of at least 5 seconds to be ended manually.");
            return;
        }

        // Ends active session.
        endSession(endType);

        // Starts a new session.
        if (!activityService.isInBackground()) {
            startSession(false, endType);
        }

        EmbraceLogger.logInfo("Session successfully closed.");
    }

    /**
     * This will trigger all necessary events to end the current session and send it to the server.
     *
     * @param endType the origin of the event that ends the session.
     */
    private synchronized void endSession(SessionLifeEventType endType) {

        stopCaching();
        stopSessionCloser();

        if (localConfig.getConfigurations().getSessionConfig().getAsyncEnd() || configService.getConfig().endSessionInBackgroundThread()) {
            sessionBackgroundWorker.submit(() -> {
                runEndSession(endType);
                return null;
            });
        } else {
            runEndSession(endType);
        }
    }

    private void runEndSession(SessionLifeEventType endType) {
        if (activeSession != null) {
            if (configService.isMessageTypeDisabled(MessageType.SESSION)) {
                EmbraceLogger.logWarning("Session messages disabled. Ignoring all Sessions.");
                return;
            }
            // Cache the active session to ensure that the cache agrees with the data we are
            // about to send. We do this since:
            //
            // 1. If the session data is sent, but the completion of the send is not registered
            //    by the SDK, then we will re-send an incomplete set of session data later.
            // 2. If the app crashes before we send the session message, the data since the last
            //    cache interval is lost.
            SessionMessage sessionMessage = sessionEndMessage(activeSession, true, false, endType);
            cacheSessionMessage(sessionMessage);
            activeSession = null;
            // Clean every collection of those services which have collections in memory.
            memoryCleanerService.cleanServicesCollections();
            try {
                // During power saving mode, requests are hanged forever causing an ANR when
                // Foregrounded, so the Timeout prevents this scenario from happening.
                apiClient.sendSession(sessionMessage).get(SEND_SESSION_API_CLIENT_TIMEOUT, TimeUnit.SECONDS);
                cacheService.deleteObject(SESSION_FILE_NAME);
            } catch (Exception ex) {
                EmbraceLogger.logError("Failed to send session end message", ex);
            }
        }

        clearProperties();
    }

    /**
     * Caches a generated session message, with performance information generated up to the current
     * point.
     */
    private void cacheSessionMessage(SessionMessage sessionMessage) {
        if (isLookingForNativeCrash) {
            return;
        }

        try {
            cacheService.cacheObject(SESSION_FILE_NAME, sessionMessage, SessionMessage.class);
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to cache session");
        }
    }

    /**
     * As when resuming the app a new session is created, the screen would be the same as the
     * one the app had when was backgrounded. In this case, the SDK skips the "duplicate view
     * breadcrumb" scenario. This method, forces the repetition of the view breadcrumb when the
     * app's being resumed.
     */
    private void addViewBreadcrumbForResumedSession() {
        Optional<String> screen = breadcrumbService.getLastViewBreadcrumbScreenName();
        if (screen.isPresent()) {
            breadcrumbService.forceLogView(screen.orNull(), System.currentTimeMillis());
        } else {
            Optional<Activity> foregroundActivity = activityService.getForegroundActivity();
            if (foregroundActivity.isPresent()) {
                breadcrumbService.forceLogView(foregroundActivity.get().getLocalClassName(), System.currentTimeMillis());
            }
        }
    }

    /**
     * Send the cached session on a background thread, so that the main thread does not get
     * blocked during foreground.
     */
    private void sendCachedSession() {
        try {
            nativeCrashSendBackgroundWorker.submit(() -> {

                if (localConfig.isNdkEnabled()) {
                    nativeCrashSearchLock.tryLock(1000, TimeUnit.MILLISECONDS);

                    while (isLookingForNativeCrash) {
                        searchEndedCondition.await();
                    }

                    try {
                        sendLastMessage();
                    } catch (Exception ex) {
                        EmbraceLogger.logDebug("Failed to send cached session message with native crash", ex);
                    } finally {
                        nativeCrashSearchLock.unlock();
                    }
                } else {
                    sendLastMessage();
                }

                return null;
            });
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to create background worker to send cached session during resume", ex);
        }
    }

    private void sendLastMessage() {
        try {
            Optional<SessionMessage> optionalSession = getLastSessionMessage();
            if (optionalSession.isPresent()) {
                apiClient.sendSession(optionalSession.get());
                cacheService.deleteObject(SESSION_FILE_NAME);
            }
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to send cached session message on resume", ex);
        }
    }

    @Override
    public void close() {
        EmbraceLogger.logInfo("Shutting down EmbraceSessionService");
        stopCaching();
        stopSessionCloser();
        sessionBackgroundWorker.close();
    }

    Optional<SessionMessage> getLastSessionMessage() {
        try {
            return cacheService.loadObject(SESSION_FILE_NAME, SessionMessage.class);
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to send cached session message on resume", ex);
        }

        return Optional.absent();
    }

    Session getActiveSession() {
        return activeSession;
    }

    @Override
    public boolean addProperty(String key, String value, boolean permanent) {
        boolean added = sessionProperties.add(key, value, permanent);
        if (added) {
            ndkService.onSessionPropertiesUpdate(sessionProperties.get());
        }
        return added;
    }

    @Override
    public boolean removeProperty(String key) {
        boolean removed = sessionProperties.remove(key);
        if (removed) {
            ndkService.onSessionPropertiesUpdate(sessionProperties.get());
        }
        return removed;
    }

    @Override
    public Map<String, String> getProperties() {
        return this.sessionProperties.get();
    }

    @Override
    public void handleNativeCrash(Optional<NativeCrashData> nativeCrashData) {
        try {
            nativeCrashSearchBackgroundWorker.submit(() -> {

                nativeCrashSearchLock.tryLock(1000, TimeUnit.MILLISECONDS);

                try {
                    if (nativeCrashData.isPresent()) {
                        // retrieve last session message and update crash report id if the session id matches
                        // with the session id attached to the crash
                        Optional<SessionMessage> lastSessionMessage = getLastSessionMessage();
                        if (lastSessionMessage.isPresent()) {
                            SessionMessage lastMessage = lastSessionMessage.get();
                            NativeCrashData crash = nativeCrashData.get();
                            Session lastSession = lastMessage.getSession();
                            if (lastSession.getSessionId().equals(crash.getSessionId())) {
                                SessionMessage.Builder messageBuilder = SessionMessage.newBuilder(lastMessage);
                                Session.Builder sessionBuilder = Session.newBuilder(lastSession);
                                sessionBuilder.withCrashReportId(crash.getNativeCrashId());
                                messageBuilder.withSession(sessionBuilder.build());

                                cacheService.cacheObject(SESSION_FILE_NAME, messageBuilder.build(), SessionMessage.class);
                            } else {
                                EmbraceLogger.logDebug(String.format("Crash report ID %s did not match session ID %s. Not updating cached session.",
                                        nativeCrashData.get().getNativeCrashId(),
                                        lastSession.getSessionId()));
                            }
                        } else {
                            EmbraceLogger.logInfo(String.format("Could not find session to try to match native crash with report ID %s",
                                    nativeCrashData.get().getNativeCrashId()));
                        }
                    }

                    this.isLookingForNativeCrash = false;
                    searchEndedCondition.signal();

                } catch (Exception ex) {
                    EmbraceLogger.logDebug("Failed to update cached session message with native crash report id.", ex);
                } finally {
                    nativeCrashSearchLock.unlock();
                }
                return null;
            });
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to create background worker to update pending session with the native crash report id.", ex);
        }
    }

    void clearProperties() {
        this.sessionProperties.clearTemporary();
    }

}