package io.embrace.android.embracesdk;

import android.util.Base64;

import com.google.gson.Gson;
import com.google.gson.annotations.SerializedName;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * Describes a Crash (uncaught {@link Throwable} by a unique ID, and a list of exceptions (causes).
 */
final class Crash {
    @SerializedName("id")
    private final String id;

    @SerializedName("ex")
    private final List<ExceptionInfo> exceptions;

    @SerializedName("rep_js")
    private final List<String> jsExceptions;

    @SerializedName("th")
    private final List<ThreadInfo> threads;

    private Crash(String id, List<ExceptionInfo> exceptions, List<ThreadInfo> threads, List<String> jsExceptions) {
        this.id = id;
        this.exceptions = exceptions;
        this.jsExceptions = jsExceptions;
        this.threads = threads;
    }

    /**
     * Creates a crash from a {@link Throwable}. Extracts each cause and converts it to
     * {@link ExceptionInfo}. Optionally includes a {@link JsException}.
     *
     * @param throwable   the throwable to parse
     * @param jsException an optional JS exception that is associated with the crash
     * @return a crash
     */
    static Crash ofThrowable(Throwable throwable, JsException jsException) {
        return new Crash(Uuid.getEmbUuid(),
                exceptionInfo(throwable),
                threadsInfo(),
                jsExceptions(jsException));
    }

    /**
     * @param ex the throwable to parse
     * @return a list of {@link ExceptionInfo} elements of the throwable.
     */
    private static List<ExceptionInfo> exceptionInfo(Throwable ex) {
        List<ExceptionInfo> result = new ArrayList<>();
        Throwable throwable = ex;
        while (throwable != null && !throwable.equals(throwable.getCause())) {
            ExceptionInfo exceptionInfo = ExceptionInfo.ofThrowable(throwable);
            result.add(0, exceptionInfo);
            throwable = throwable.getCause();
        }
        return result;
    }

    /**
     * @return a list of {@link ThreadInfo} elements of the current thread list.
     */
    private static List<ThreadInfo> threadsInfo() {
        List<ThreadInfo> threads = new ArrayList<>();
        for (Map.Entry<Thread, StackTraceElement[]> entry : Thread.getAllStackTraces().entrySet()) {
            threads.add(ThreadInfo.ofThread(entry.getKey(), entry.getValue()));
        }
        return threads;
    }

    /**
     * @param jsException the {@link JsException} coming from the React Native layer.
     * @return a list of {@link String} representing the javascript stacktrace of the crash.
     */
    private static List<String> jsExceptions(JsException jsException) {
        List<String> jsExceptions = null;
        if (jsException != null) {
            try {
                byte[] jsonException = new Gson().toJson(jsException, jsException.getClass()).getBytes();
                String encodedString = Base64.encodeToString(jsonException, Base64.NO_WRAP);
                jsExceptions = Collections.singletonList(encodedString);
            } catch (Exception ex) {
                EmbraceLogger.logError("Failed to parse javascript exception", ex, true);
            }
        }
        return jsExceptions;
    }

    String getCause() {
        return exceptions != null && !exceptions.isEmpty() ? this.exceptions.get(0).getName() : null;
    }

    String getCrashId() {
        return id;
    }
}
