package io.embrace.android.embracesdk;

import android.content.Context;
import android.content.res.Resources;

import com.fernandocejas.arrow.optional.Optional;

/**
 * Specifies the application ID and build ID.
 */
public final class BuildInfo {

    static final String BUILD_INFO_BUILD_ID = "emb_build_id";
    static final String BUILD_INFO_BUILD_TYPE = "emb_build_type";
    static final String BUILD_INFO_BUILD_FLAVOR = "emb_build_flavor";

    private static final String RES_TYPE_STRING = "string";

    /**
     * The ID of the particular build, generated at compile-time.
     */
    private final String buildId;

    /**
     * The BuildType name of the particular build, extracted at compile-time.
     */
    private final String buildType;

    /**
     * The Flavor name of the particular build, extracted at compile-time.
     */
    private final String buildFlavor;

    BuildInfo(String buildId, String buildType, String buildFlavor) {
        this.buildId = buildId;
        this.buildType = buildType;
        this.buildFlavor = Optional.fromNullable(buildFlavor).orNull();
    }

    /**
     * Loads the build information from resources provided by the config file packaged within the application by Gradle at
     * build-time.
     *
     * @return the build information
     */
    static BuildInfo fromResources(Context context) {
        try {
            int buildFlavorId = getResourcesIdentifier(context, BUILD_INFO_BUILD_FLAVOR, RES_TYPE_STRING);

            return new BuildInfo(
                    context.getResources().getString(getResourcesIdentifier(context, BUILD_INFO_BUILD_ID, RES_TYPE_STRING)),
                    context.getResources().getString(getResourcesIdentifier(context, BUILD_INFO_BUILD_TYPE, RES_TYPE_STRING)),
                    buildFlavorId != 0 ? context.getResources().getString(buildFlavorId) : null);
        } catch (Exception ex) {
            throw new RuntimeException("Failed to read build_info.xml", ex);
        }
    }

    /**
     * Given a build property name and a build property type, retrieves the embrace build resource id.
     *
     * @param context
     * @param buildProperty
     * @param type
     * @return
     */
    private static int getResourcesIdentifier(Context context, String buildProperty, String type) {
        Resources resources = context.getResources();
        return resources.getIdentifier(buildProperty, type, context.getPackageName());
    }


    String getBuildId() {
        return buildId;
    }

    String getBuildType() {
        return buildType;
    }

    String getBuildFlavor() {
        return buildFlavor;
    }
}
