package io.embrace.android.embracesdk;

import android.content.Context;

import com.fernandocejas.arrow.checks.Preconditions;
import com.fernandocejas.arrow.optional.Optional;
import com.fernandocejas.arrow.strings.Charsets;
import com.google.gson.Gson;
import com.google.gson.stream.JsonReader;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

/**
 * Handles the reading and writing of objects from the app's cache.
 */
class EmbraceCacheService implements CacheService {

    private static final String EMBRACE_PREFIX = "emb_";

    private final Gson gson = new Gson();

    private final Context context;

    EmbraceCacheService(Context context) {
        this.context = Preconditions.checkNotNull(context, "context must not be null");
    }

    /**
     * Writes a file to the cache. Must be serializable by GSON.
     * <p>
     * If writing the object to the cache fails, an exception is logged.
     *
     * @param name   the name of the object to write
     * @param object the object to write
     * @param clazz  the class of the object to write
     * @param <T>    the type of the object to write
     */
    @Override
    public <T> void cacheObject(String name, T object, Class<T> clazz) {
        BufferedWriter bw = null;
        FileWriter fw = null;
        File file = new File(context.getCacheDir(), EMBRACE_PREFIX + name);
        try {
            fw = new FileWriter(file.getAbsoluteFile());
            bw = new BufferedWriter(fw);
            bw.write(gson.toJson(object, clazz));
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to store cache object " + file.getPath(), ex);
        } finally {
            // BufferedWriter closes the FileWriter it wraps, so no need to close fw
            try {
                if (bw != null) {
                    bw.close();
                }
            } catch (Exception ex) {
                EmbraceLogger.logDebug("Failed to close cache writer " + file.getPath(), ex);
            }
        }
    }

    @Override
    public <T> Optional<T> loadObject(String name, Class<T> clazz) {
        File file = new File(context.getCacheDir(), EMBRACE_PREFIX + name);
        try (FileInputStream fileInputStream = new FileInputStream(file);
             InputStreamReader inputStreamReader = new InputStreamReader(fileInputStream, Charsets.UTF_8);
             JsonReader jsonreader = new JsonReader(inputStreamReader)) {
            jsonreader.setLenient(true);
            T obj = gson.fromJson(jsonreader, clazz);
            if (obj != null) {
                return Optional.of(obj);
            }
        } catch (FileNotFoundException ex) {
            EmbraceLogger.logDebug("Cache file cannot be found " + file.getPath());
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to read cache object " + file.getPath(), ex);
        }
        return Optional.absent();
    }

    @Override
    public <T> List<T> loadObjectsByRegex(String regex, Class<T> clazz) {
        Pattern pattern = Pattern.compile(regex);
        List<T> objects = new ArrayList<>();

        for (File cache : context.getCacheDir().listFiles()) {
            if (pattern.matcher(cache.getName()).find()) {
                try (FileInputStream fileInputStream = new FileInputStream(cache);
                     InputStreamReader inputStreamReader = new InputStreamReader(fileInputStream, Charsets.UTF_8);
                     JsonReader jsonreader = new JsonReader(inputStreamReader)) {
                    jsonreader.setLenient(true);
                    T obj = gson.fromJson(jsonreader, clazz);
                    if (obj != null) {
                        objects.add(obj);
                    }
                } catch (FileNotFoundException ex) {
                    EmbraceLogger.logDebug("Cache file cannot be found " + cache.getPath());
                } catch (Exception ex) {
                    EmbraceLogger.logDebug("Failed to read cache object " + cache.getPath(), ex);
                }
            }
        }

        return objects;
    }

    @Override
    public boolean deleteObject(String name) {
        File file = new File(context.getCacheDir(), EMBRACE_PREFIX + name);
        try {
            return file.delete();
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to delete cache object " + file.getPath());
        }
        return false;
    }

    @Override
    public boolean deleteObjectsByRegex(String regex) {
        Pattern pattern = Pattern.compile(regex);
        boolean result = false;
        for (File cache : context.getCacheDir().listFiles()) {
            if (pattern.matcher(cache.getName()).find()) {
                try {
                    result = cache.delete();
                } catch (Exception ex) {
                    EmbraceLogger.logDebug("Failed to delete cache object " + cache.getPath());
                }
            }
        }

        return result;
    }
}
