package io.embrace.android.embracesdk;

import com.google.gson.annotations.SerializedName;

import java.util.HashSet;
import java.util.Set;

/**
 * Information about the user of the app, provided by the developer performing the integration.
 */
final class UserInfo {

    static final String PERSONA_NEW_USER = "new_user";
    static final String PERSONA_POWER_USER = "power_user";
    static final String PERSONA_LOGGED_IN = "logged_in";
    static final String PERSONA_VIP = "vip";
    static final String PERSONA_CREATOR = "content_creator";
    static final String PERSONA_TESTER = "tester";
    static final String PERSONA_PAYER = "payer";
    static final String PERSONA_FIRST_DAY_USER = "first_day";

    private UserInfo(Builder builder) {
        userId = builder.userId;
        email = builder.email;
        username = builder.username;
        personas = builder.personas;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static Builder newBuilder(UserInfo copy) {
        Builder builder = new Builder();
        builder.userId = copy.getUserId();
        builder.email = copy.getEmail();
        builder.username = copy.getUsername();
        builder.personas = copy.getPersonas();
        return builder;
    }

    @SerializedName("id")
    private String userId;

    @SerializedName("em")
    private String email;

    @SerializedName("un")
    private String username;

    @SerializedName("per")
    private Set<String> personas;

    UserInfo(String userId, String email, String username, Set<String> personas) {
        this.userId = userId;
        this.email = email;
        this.username = username;
        this.personas = personas;
    }

    public String getUserId() {
        return userId;
    }

    public String getEmail() {
        return email;
    }

    public String getUsername() {
        return username;
    }

    public Set<String> getPersonas() {
        return personas;
    }

    void setUserId(String userId) {
        this.userId = userId;
    }

    void setEmail(String email) {
        this.email = email;
    }

    void setUsername(String username) {
        this.username = username;
    }

    void setPersonas(Set<String> personas) {
        this.personas = personas;
    }

    /**
     * Creates an empty instance.
     *
     * @return an empty instance
     */
    public static UserInfo ofEmpty() {
        return new UserInfo(null, null, null, null);
    }

    /**
     * Creates an instance of {@link UserInfo} from the cache.
     *
     * @param preferencesService the preferences service
     * @return user info created from the cache and configuration
     */
    public static UserInfo ofStored(PreferencesService preferencesService) {
        Builder builder = newBuilder();
        Set<String> personas = new HashSet<>();
        if (preferencesService.getUserEmailAddress().isPresent()) {
            builder = builder.withEmail(preferencesService.getUserEmailAddress().get());
        }

        if (preferencesService.getUserIdentifier().isPresent()) {
            builder = builder.withUserId(preferencesService.getUserIdentifier().get());
        }

        if (preferencesService.getUsername().isPresent()) {
            builder = builder.withUsername(preferencesService.getUsername().get());
        }

        if (preferencesService.getUserPersonas().isPresent()) {
            personas.addAll(preferencesService.getUserPersonas().get());
        }

        if (preferencesService.getCustomPersonas().isPresent()) {
            personas.addAll(preferencesService.getCustomPersonas().get());
        }
        personas.remove(PERSONA_PAYER);
        if (preferencesService.getUserPayer()) {
            personas.add(PERSONA_PAYER);
        }
        personas.remove(PERSONA_FIRST_DAY_USER);
        if (preferencesService.isUsersFirstDay()) {
            personas.add(PERSONA_FIRST_DAY_USER);
        }

        builder = builder.withPersonas(personas);
        return builder.build();
    }


    public static final class Builder {
        private String userId;
        private String email;
        private String username;
        private Set<String> personas;

        private Builder() {
        }

        public Builder withUserId(String userId) {
            this.userId = userId;
            return this;
        }

        public Builder withEmail(String email) {
            this.email = email;
            return this;
        }

        public Builder withUsername(String username) {
            this.username = username;
            return this;
        }

        public Builder withPersonas(Set<String> personas) {
            this.personas = personas;
            return this;
        }

        public String getUserId() {
            return userId;
        }

        public String getEmail() {
            return email;
        }

        public String getUsername() {
            return username;
        }

        public Set<String> getPersonas() {
            return personas;
        }

        public UserInfo build() {
            return new UserInfo(this);
        }
    }
}
