package io.embrace.android.embracesdk.network.http;

import com.fernandocejas.arrow.checks.Preconditions;

import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.concurrent.atomic.AtomicLong;

import java9.util.function.Consumer;

/**
 * Counts the bytes read from an input stream and invokes a callback once the stream has reached
 * the end.
 */
final class CountingInputStreamWithCallback extends FilterInputStream {
    /** The count of the number of bytes which have been read. */
    private volatile AtomicLong count;
    /** The mark. */
    private volatile long mark = -1;
    /** The callback to be invoked with num of bytes after reaching the end of the stream. */
    private final Consumer<Long> callback;
    /** true if the callback has been invoked, false otherwise. */
    private volatile boolean callbackCompleted;

    /**
     * Wraps another input stream, counting the number of bytes read.
     *
     * @param in the input stream to be wrapped
     */
    public CountingInputStreamWithCallback(InputStream in, Consumer<Long> callback) {
        super(Preconditions.checkNotNull(in));
        this.callback = Preconditions.checkNotNull(callback);
    }

    /** Returns the number of bytes read. */
    public long getCount() {
        return count.longValue();
    }

    @Override
    public int read() throws IOException {
        int result = in.read();
        if (result != -1) {
            count.incrementAndGet();
        } else if (!callbackCompleted){
            callbackCompleted = true;
            callback.accept(count.longValue());
        }
        return result;
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        int result = in.read(b, off, len);
        if (result != -1) {
            count.addAndGet(result);
        } else if (!callbackCompleted) {
            callbackCompleted = true;
            callback.accept(count.longValue());
        }
        return result;
    }

    @Override
    public long skip(long n) throws IOException {
        long result = in.skip(n);
        count.addAndGet(result);
        return result;
    }

    @Override
    public synchronized void mark(int readlimit) {
        in.mark(readlimit);
        mark = count.longValue();
        // it's okay to mark even if mark isn't supported, as reset won't work
    }

    @Override
    public synchronized void reset() throws IOException {
        if (!in.markSupported()) {
            throw new IOException("Mark not supported");
        }
        if (mark == -1) {
            throw new IOException("Mark not set");
        }

        in.reset();
        count.set(mark);
        callbackCompleted = false;
    }
}
