package io.embrace.android.embracesdk;

import com.fernandocejas.arrow.optional.Optional;

import java.util.HashSet;
import java.util.Set;
import java.util.regex.Pattern;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

class EmbraceUserService implements ActivityListener, UserService {

    // Valid persona regex representation.
    static final Pattern VALID_PERSONA = Pattern.compile("^[a-zA-Z0-9_]{1,32}$");
    // Maximum number of allowed personas.
    static final int PERSONA_LIMIT = 10;

    private final PreferencesService preferencesService;
    private volatile UserInfo userInfo;

    EmbraceUserService(
            PreferencesService preferencesService) {

        this.preferencesService = preferencesService;
        this.userInfo = UserInfo.ofStored(preferencesService);
    }

    @Override
    public Optional<UserInfo> loadUserInfoFromDisk() {
        try {
            return Optional.fromNullable(UserInfo.ofStored(preferencesService));
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to load user info from persistent storage.");
            return Optional.absent();
        }
    }

    @Override
    public UserInfo getUserInfo() {
        return userInfo;
    }

    @Override
    public void setUserIdentifier(@Nullable String userId) {
        UserInfo.Builder builder = UserInfo.newBuilder(userInfo);
        this.userInfo = builder
                .withUserId(userId)
                .build();
        try {
            if (userId != null) {
                Embrace.getInstance().getSessionService().getActiveSession().getUser().setUserId(userId);
            }
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to save user's id on the current Session. User info is null.");
        }
        preferencesService.setUserIdentifier(userId);
    }

    @Override
    public void clearUserIdentifier() {
        setUserIdentifier(null);
    }

    @Override
    public void setUsername(@Nullable String username) {
        UserInfo.Builder builder = UserInfo.newBuilder(userInfo);
        this.userInfo = builder
                .withUsername(username)
                .build();
        try {
            if (username != null) {
                Embrace.getInstance().getSessionService().getActiveSession().getUser().setUsername(username);
            }
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to save user's username on the current Session. User info is null.");
        }
        preferencesService.setUsername(username);
    }

    @Override
    public void clearUsername() {
        setUsername(null);
    }

    @Override
    public void setUserEmail(@Nullable String email) {
        UserInfo.Builder builder = UserInfo.newBuilder(userInfo);
        this.userInfo = builder
                .withEmail(email)
                .build();
        try {
            if (email != null) {
                Embrace.getInstance().getSessionService().getActiveSession().getUser().setEmail(email);
            }
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to save user's email on the current Session. User info is null.");
        }
        preferencesService.setUserEmailAddress(email);
    }

    @Override
    public void clearUserEmail() {
        setUserEmail(null);
    }

    @Override
    public void setUserAsPayer() {
        setUserPersona(UserInfo.PERSONA_PAYER);
    }

    @Override
    public void clearUserAsPayer() {
        clearUserPersona(UserInfo.PERSONA_PAYER);
    }

    @Override
    public void setUserPersona(@NonNull String persona) {
        if (!VALID_PERSONA.matcher(persona).matches()) {
            EmbraceLogger.logWarning("Ignoring persona " + persona + " as it does not match " + VALID_PERSONA.pattern());
            return;
        }
        if (userInfo.getPersonas().size() >= PERSONA_LIMIT) {
            EmbraceLogger.logWarning("Cannot set persona as the limit of " + PERSONA_LIMIT + " has been reached");
            return;
        }
        UserInfo.Builder builder = UserInfo.newBuilder(userInfo);
        Set<String> personas = new HashSet<>();
        if (builder.getPersonas() != null) {
            personas.addAll(builder.getPersonas());
        }
        personas.add(persona);

        this.userInfo = builder
                .withPersonas(personas)
                .build();
        updatePersonas(personas);
    }

    @Override
    public void clearUserPersona(@NonNull String persona) {
        UserInfo.Builder builder = UserInfo.newBuilder(userInfo);
        Set<String> personas = new HashSet<>();
        if (builder.getPersonas() != null) {
            personas.addAll(builder.getPersonas());
        }
        personas.remove(persona);
        this.userInfo = builder
                .withPersonas(personas)
                .build();
        updatePersonas(personas);
    }

    @Override
    public void clearAllUserPersonas() {
        UserInfo.Builder builder = UserInfo.newBuilder(userInfo);
        Set<String> personas = new HashSet<>();
        if (preferencesService.getUserPayer()) {
            personas.add(UserInfo.PERSONA_PAYER);
        }
        if (preferencesService.isUsersFirstDay()) {
            personas.add(UserInfo.PERSONA_FIRST_DAY_USER);
        }
        this.userInfo = builder
                .withPersonas(personas)
                .build();
        updatePersonas(personas);
    }

    private void updatePersonas(Set<String> personas) {
        try {
            Embrace.getInstance().getSessionService().getActiveSession().getUser().setPersonas(personas);
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to save user's personas on the current Session. User info is null.");
        }
        preferencesService.setUserPersonas(personas);
    }

    @Override
    public void clearAllUserInfo() {
        clearUserIdentifier();
        clearUserEmail();
        clearUsername();
        clearAllUserPersonas();
    }
}
