package io.embrace.android.embracesdk;

import android.content.Context;
import android.content.res.Resources;
import android.os.Build;
import android.text.TextUtils;
import android.util.Base64;

import com.fernandocejas.arrow.checks.Preconditions;
import com.fernandocejas.arrow.optional.Optional;
import com.fernandocejas.arrow.strings.Charsets;
import com.google.gson.Gson;
import com.google.gson.stream.JsonReader;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import java9.util.stream.Collectors;
import java9.util.stream.StreamSupport;

class EmbraceNdkService implements NdkService, ActivityListener {

    /**
     * Signals to the API that the application was in the foreground.
     */
    private static final String APPLICATION_STATE_ACTIVE = "active";
    /**
     * Signals to the API that the application was in the background.
     */
    private static final String APPLICATION_STATE_BACKGROUND = "background";

    private static final String CRASH_REPORT_EVENT_NAME = "_crash_report";

    private static final String NATIVE_CRASH_FILE_PREFIX = "emb_ndk";

    private static final String NATIVE_CRASH_FILE_FOLDER = "ndk";

    private static final int MAX_NATIVE_CRASH_FILES_ALLOWED = 4;

    /**
     * Synchronization lock.
     */
    private final Object lock = new Object();
    /**
     * Whether or not the NDK has been installed.
     */
    private static boolean isInstalled = false;

    private final Context context;

    private final MetadataService metadataService;

    private final CacheService cacheService;

    private final ApiClient apiClient;

    private final UserService userService;

    private Gson gson;

    private final BackgroundWorker cleanCacheWorker;
    private final BackgroundWorker crashFetchWorker;

    EmbraceNdkService(
            Context context,
            MetadataService metadataService,
            ActivityService activityService,
            LocalConfig localConfig,
            CacheService cacheService,
            ApiClient apiClient,
            UserService userService) {

        this.context = Preconditions.checkNotNull(context);
        this.metadataService = Preconditions.checkNotNull(metadataService);
        this.cacheService = Preconditions.checkNotNull(cacheService);
        this.apiClient = Preconditions.checkNotNull(apiClient);
        this.userService = Preconditions.checkNotNull(userService);
        Preconditions.checkNotNull(activityService);
        Preconditions.checkNotNull(localConfig);

        this.cleanCacheWorker = BackgroundWorker.ofSingleThread("Native Crash Cleaner");
        this.crashFetchWorker = BackgroundWorker.ofSingleThread("Native Crash Fetch");

        if (localConfig.isNdkEnabled()) {
            activityService.addListener(this, true);
            this.gson = new Gson();

            startNdk();
            cleanOldCrashFiles();
            checkForNativeCrash();
        }
    }

    @Override
    public void testCrash(boolean isCpp) {
        if (isCpp) {
            testCrashCPP();
        } else {
            testCrashC();
        }

    }

    @Override
    public void updateSessionId(String newSessionId) {
        if (isInstalled) {
            _updateSessionId(newSessionId);
        }
    }

    @Override
    public void onBackground() {
        synchronized (lock) {
            if (isInstalled) {
                updateAppState(APPLICATION_STATE_BACKGROUND);
            }
        }
    }

    @Override
    public void onForeground(boolean coldStart, long startupTime) {
        synchronized (lock) {
            if (isInstalled) {
                updateAppState(APPLICATION_STATE_ACTIVE);
            }
        }
    }

    private void startNdk() {
        try {
            loadNDKLibrary();
            installSignals();
            createCrashReportDirectory();
            EmbraceLogger.logInfo("NDK library successfully loaded");
        } catch (Exception ex) {
            EmbraceLogger.logError("Failed to load NDK library", ex);
        }
    }

    private void createCrashReportDirectory() {
        String directory = String.format("%s/%s", context.getCacheDir(), NATIVE_CRASH_FILE_FOLDER);
        File directoryFile = new File(directory);

        if (directoryFile.exists()) {
            return;
        }

        if (!directoryFile.mkdirs()) {
            EmbraceLogger.logError("Failed to create crash report directory");
        }
    }

    private void loadNDKLibrary() throws UnsatisfiedLinkError {
        System.loadLibrary("embrace-native");
    }

    private void installSignals() {
        boolean is32bit;
        String reportPath = String.format("%s/%s", context.getCacheDir().getAbsolutePath(), NATIVE_CRASH_FILE_FOLDER);

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            is32bit = !TextUtils.join(", ", Build.SUPPORTED_ABIS).contains("64");
        } else {
            is32bit = true;
        }

        NativeCrashData.NativeCrashMetadata metadata = new NativeCrashData.NativeCrashMetadata(
                metadataService.getAppInfo(),
                metadataService.getDeviceInfo());

        _installSignalHandlers(
                reportPath,
                gson.toJson(metadata),
                "null",
                metadataService.getAppState(),
                Uuid.getEmbUuid(),
                Build.VERSION.SDK_INT, is32bit);

        isInstalled = true;
    }

    /**
     * Check if a native crash file exists. Also checks for the symbols file in the build dir.
     * If so, attempt to send an event message and call {@link SessionService} to update the crash
     * report id in the appropriate pending session.
     */
    private void checkForNativeCrash() {
        crashFetchWorker.submit(() -> {
            NativeCrashData nativeCrash = null;
            List<File> matchingFiles = sortNativeCrashes(false);

            for (File crashFile : matchingFiles) {
                try (FileInputStream fileInputStream = new FileInputStream(crashFile);
                     InputStreamReader inputStreamReader = new InputStreamReader(fileInputStream, Charsets.UTF_8);
                     JsonReader jsonreader = new JsonReader(inputStreamReader)) {
                    jsonreader.setLenient(true);

                    nativeCrash = new Gson().fromJson(jsonreader, NativeCrashData.class);

                    //retrieve deobfusacted symbols
                    NativeSymbols nativeSymbols = getNativeSymbols();
                    if (nativeSymbols != null) {
                        String arch = MetadataUtils.getArchitecture();
                        nativeCrash.setSymbols(nativeSymbols.getSymbolByArchitecture(arch));
                    }

                    sendNativeCrash(nativeCrash);

                    if (!crashFile.delete()) {
                        EmbraceLogger.logWarning("Failed to delete native crash file.");
                    }
                } catch (FileNotFoundException ex) {
                    EmbraceLogger.logDebug("Native crash file not found.", ex);
                } catch (Exception ex) {
                    crashFile.deleteOnExit();
                    EmbraceLogger.logDebug("Failed to read native crash file " + crashFile.getPath(), ex);
                }
            }

            Embrace.getInstance().getSessionService().handleNativeCrash(Optional.fromNullable(nativeCrash));
            return null;
        });
    }

    private NativeSymbols getNativeSymbols() {
        Resources resources = context.getResources();
        int resourceId = resources.getIdentifier("symbols", "string", context.getPackageName());

        if (resourceId != 0) {
            try {
                String encodedSymbols = new String(Base64.decode(context.getResources().getString(resourceId), Base64.DEFAULT));
                return gson.fromJson(encodedSymbols, NativeSymbols.class);
            } catch (Exception ex) {
                EmbraceLogger.logError("Failed to decode symbols from resources.", ex);
            }
        } else {
            EmbraceLogger.logError("Failed to find symbols in resources.");
        }

        return null;
    }

    private File[] getNativeCrashFiles() {
        File[] matchingFiles = null;

        for (File cached : context.getCacheDir().listFiles()) {
            if (cached.isDirectory() && cached.getName().equals(NATIVE_CRASH_FILE_FOLDER)) {
                matchingFiles = cached.listFiles(file -> file.getName().startsWith(NATIVE_CRASH_FILE_PREFIX));
                break;
            }
        }

        return matchingFiles;
    }

    private void cleanOldCrashFiles() {
        cleanCacheWorker.submit(() -> {
            List<File> sortedFiles = sortNativeCrashes(true);

            int deleteCount = sortedFiles.size() - MAX_NATIVE_CRASH_FILES_ALLOWED;

            if (deleteCount > 0) {
                LinkedList<File> files = new LinkedList<>(sortedFiles);

                try {
                    for (int i = 0; i < deleteCount; i++) {
                        File removed = files.get(i);
                        if (files.get(i).delete()) {
                            EmbraceLogger.logDebug(String.format("Native crash file %s removed from cache", removed.getName()));
                        }
                    }
                } catch (Exception ex) {
                    EmbraceLogger.logError("Failed to delete native crash from cache.", ex);
                }
            }

            return null;
        });
    }

    private List<File> sortNativeCrashes(boolean byOldest) {
        final File[] nativeCrashFiles = getNativeCrashFiles();
        final List<File> nativeCrashList = new ArrayList<>();

        if (nativeCrashFiles == null) {
            return nativeCrashList;
        }

        nativeCrashList.addAll(Arrays.asList(nativeCrashFiles));
        final Map<File, Long> sorted = new HashMap<>();

        try {
            for (final File f : nativeCrashList) {
                sorted.put(f, f.lastModified());
            }

            return StreamSupport.stream(nativeCrashList)
                    .sorted((first, next) -> byOldest ? sorted.get(first).compareTo(sorted.get(next)) : sorted.get(next).compareTo(sorted.get(first)))
                    .collect(Collectors.toList());
        } catch (Exception ex) {
            EmbraceLogger.logError("Failed to sort native crashes to remove oldest.", ex);
        }

        return nativeCrashList;
    }

    private void sendNativeCrash(NativeCrashData nativeCrash) {

        Event.Builder nativeCrashEventBuilder = Event.newBuilder()
                .withName(CRASH_REPORT_EVENT_NAME)
                .withType(EmbraceEvent.Type.CRASH)
                .withSessionId(nativeCrash.getSessionId())
                .withTimestamp(nativeCrash.getTimestamp())
                .withAppState(nativeCrash.getAppState());

        EventMessage.Builder nativeCrashMessageEventBuilder = EventMessage.newBuilder()
                .withAppInfo(nativeCrash.getMetadata().getAppInfo())
                .withDeviceInfo(nativeCrash.getMetadata().getDeviceInfo())
                .withUserInfo(userService.getUserInfo())
                .withNativeCrash(nativeCrash.getCrash())
                .withEvent(nativeCrashEventBuilder.build());
        try {
            apiClient.sendEvent(nativeCrashMessageEventBuilder.build()).get();
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to report native crash to the api", ex);
        }
    }

    private void updateAppState(String newAppState) {
        _updateAppState(newAppState);
    }

    private void updateDeviceMetaData(NativeCrashData.NativeCrashMetadata newDeviceMetaData) {
        _updateMetaData(gson.toJson(newDeviceMetaData));
    }

    private void uninstallSignals() {
        _uninstallSignals();
    }

    private void testCrashC() {
        _testNativeCrash_C();
    }

    private void testCrashCPP() {
        _testNativeCrash_CPP();
    }

    public native void _installSignalHandlers(String report_path, String device_meta_data, String session_id, String app_state, String report_id, int api_level, boolean is_32bit);

    public native void _updateMetaData(String new_device_meta_data);

    public native void _updateSessionId(String new_session_id);

    public native void _updateAppState(String new_app_state);

    public native void _uninstallSignals();

    public native void _testNativeCrash_C();

    public native void _testNativeCrash_CPP();
}
