package io.embrace.android.embracesdk;

import com.fernandocejas.arrow.optional.Optional;

/**
 * Manages user information, allowing the setting and removing of various user attributes.
 */
interface UserService {
    /**
     * Gets information on the current user of the device. This is sent with all events and sessions.
     *
     * @return the current user information
     */
    UserInfo getUserInfo();

    /**
     * Cleans the user info.
     */
    void clearAllUserInfo();

    /**
     * Gets user information from persistent storage.
     *
     * @return the current user information from persistent storage
     */
    Optional<UserInfo> loadUserInfoFromDisk();

    /**
     * Sets the user's ID. This could be a UUID, for example.
     *
     * @param userId the user's unique identifier
     */
    void setUserIdentifier(String userId);

    /**
     * Removes the user's unique identifier.
     */
    void clearUserIdentifier();

    /**
     * Sets the user's email address.
     *
     * @param email the user's email address
     */
    void setUserEmail(String email);

    /**
     * Removes the user's email address.
     */
    void clearUserEmail();

    /**
     * Sets the user as a paying user, attaching the paying persona to the user.
     */
    void setUserAsPayer();

    /**
     * Unsets the user as a paying user, removing the paying persona from the user.
     */
    void clearUserAsPayer();

    /**
     * Attaches the specified persona to the user. This can be used for user segmentation.
     *
     * @param persona the persona to attach to the user
     */
    void setUserPersona(String persona);

    /**
     * Removes the specified user persona from the user.
     *
     * @param persona the persona to remove from the user
     */
    void clearUserPersona(String persona);

    /**
     * Clears all personas from the user, except for system personas.
     */
    void clearAllUserPersonas();

    /**
     * Sets the user's username.
     *
     * @param username the username to set
     */
    void setUsername(String username);

    /**
     * Removes the user's username.
     */
    void clearUsername();

}
