package io.embrace.android.embracesdk;

import android.content.Context;
import android.content.res.Resources;
import android.util.Base64;

import com.fernandocejas.arrow.optional.Optional;
import com.google.gson.Gson;
import com.google.gson.annotations.SerializedName;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import java9.util.stream.StreamSupport;

public final class LocalConfig {

    /**
     * Build info app id name.
     */
    static final String BUILD_INFO_APP_ID = "emb_app_id";

    /**
     * Build info ndk enabled.
     */
    private static final String BUILD_INFO_NDK_ENABLED = "emb_ndk_enabled";

    /**
     * Build info sdk config id name.
     */
    static final String BUILD_INFO_SDK_CONFIG = "emb_sdk_config";

    /**
     * The default value for native crash capture enabling
     */
    private static final Boolean NDK_ENABLED_DEFAULT = false;

    /**
     * The Embrace app ID. This is used to identify the app within the database.
     */
    private final String appId;

    /**
     * Control whether the Embrace SDK is able to capture native crashes.
     */
    @SerializedName("ndk_enabled")
    private Boolean ndkEnabled;

    /**
     * The Embrace sdk configurations. This is used to setup configurations.
     */
    private final SdkConfigs configs;

    LocalConfig(String appId, Boolean ndkEnabled, SdkConfigs configs) {
        this.appId = appId;
        this.ndkEnabled = ndkEnabled;
        this.configs = Optional.fromNullable(configs).orNull();
    }

    /**
     * Loads the build information from resources provided by the config file packaged within the application by Gradle at
     * build-time.
     *
     * @return the local configuration
     */

    static LocalConfig fromResources(Context context) {
        try {

            String appId = context.getResources().getString(getResourcesIdentifier(context, BUILD_INFO_APP_ID, "string"));

            boolean ndkEnabled = NDK_ENABLED_DEFAULT;
            int ndkEnabledJsonId = getResourcesIdentifier(context, BUILD_INFO_NDK_ENABLED, "string");
            if (ndkEnabledJsonId != 0) {
                ndkEnabled = Boolean.parseBoolean(context.getResources().getString(ndkEnabledJsonId));
            }

            String sdkConfigJson = null;
            int sdkConfigJsonId = getResourcesIdentifier(context, BUILD_INFO_SDK_CONFIG, "string");
            if (sdkConfigJsonId != 0) {
                String encodedConfig = context.getResources().getString(sdkConfigJsonId);
                //Decode sdkConfig
                sdkConfigJson = new String(Base64.decode(encodedConfig, Base64.DEFAULT));
            }

            return buildConfig(appId, ndkEnabled, sdkConfigJson);
        } catch (Exception ex) {
            throw new RuntimeException("Failed to load local config from resources.", ex);
        }
    }

    static LocalConfig buildConfig(String appId, boolean ndkEnabled, String sdkConfigs) {
        if (appId == null || appId.isEmpty()) {
            throw new IllegalArgumentException("Embrace AppId cannot be null or empty.");
        }

        EmbraceLogger.logInfo(String.format("Native crash capture is %s", ndkEnabled ? "enabled" : "disabled"));

        SdkConfigs configs;
        if (sdkConfigs != null && !sdkConfigs.isEmpty()) {
            configs = new Gson().fromJson(sdkConfigs, SdkConfigs.class);
        } else {
            configs = new SdkConfigs();
        }

        List<String> nonDefaultValues = configs.getNonDefaultValues();
        for (String nonDefaultValue : nonDefaultValues) {
            EmbraceLogger.logInfo("Custom config value " + nonDefaultValue);
        }
        return new LocalConfig(appId, ndkEnabled, configs);
    }

    /**
     * Given a config property name and a config property type, retrieves the embrace config resource id.
     *
     * @param context
     * @param configProperty
     * @param type
     * @return
     */
    private static int getResourcesIdentifier(Context context, String configProperty, String type) {
        Resources resources = context.getResources();
        return resources.getIdentifier(configProperty, type, context.getPackageName());
    }

    String getAppId() {
        return appId;
    }

    Boolean isNdkEnabled() {
        return ndkEnabled;
    }

    public SdkConfigs getConfigurations() {
        return configs;
    }

    /**
     * Checks if the url is allowed to be reported based on the specified disabled pattern.
     *
     * @param url the url to test
     * @return true if the url is disabled for reporting, false otherwise
     */
    boolean isUrlDisabled(String url) {
        Set<Pattern> patterns = getConfigurations().getNetworking().getDisabledUrlRegexPatterns();
        return StreamSupport.stream(patterns)
                .anyMatch(p -> p.matcher(url).find());
    }

    static class SdkConfigs {

        /**
         * App settings
         */
        @SerializedName("app")
        private final App app;

        /**
         * Base URL settings
         */
        @SerializedName("base_urls")
        private final BaseUrls baseUrls;

        /**
         * Crash handler settings
         */
        @SerializedName("crash_handler")
        private final CrashHandler crashHandler;

        /**
         * Startup moment settings
         */
        @SerializedName("startup_moment")
        private final StartupMoment startupMoment;

        /**
         * Networking moment settings
         */
        @SerializedName("networking")
        private final Networking networking;

        /**
         * Session config settings
         */
        @SerializedName("session")
        private final SessionConfig sessionConfig;

        /**
         * Taps
         */
        @SerializedName("taps")
        private final Taps taps;

        SdkConfigs() {
            this(null, null, null, null, null, null, null);
        }

        SdkConfigs(App app, BaseUrls baseUrls, CrashHandler crashHandler, StartupMoment startupMoment, Networking networking, SessionConfig sessionConfig, Taps taps) {

            this.app = Optional.fromNullable(app).or(new App());
            this.baseUrls = Optional.fromNullable(baseUrls).or(new BaseUrls());
            this.crashHandler = Optional.fromNullable(crashHandler).or(new CrashHandler());
            this.startupMoment = Optional.fromNullable(startupMoment).or(new StartupMoment());
            this.networking = Optional.fromNullable(networking).or(new Networking());
            this.sessionConfig = Optional.fromNullable(sessionConfig).or(new SessionConfig());
            this.taps = Optional.fromNullable(taps).or(new Taps());
        }

        App getApp() {
            return app;
        }

        BaseUrls getBaseUrls() {
            return baseUrls;
        }

        CrashHandler getCrashHandler() {
            return crashHandler;
        }

        StartupMoment getStartupMoment() {
            return startupMoment;
        }

        Networking getNetworking() {
            return networking;
        }

        SessionConfig getSessionConfig() {
            return sessionConfig;
        }

        Taps getTaps() {
            return taps;
        }

        List<String> getNonDefaultValues() {
            List<String> nonDefaultValues = new ArrayList<>();
            List<ConfigElement> elements = Arrays.asList(app, baseUrls, crashHandler, networking, sessionConfig, startupMoment, taps);
            for (ConfigElement element : elements) {
                if (element.getNonDefaultValues().isEmpty()) {
                    continue;
                }
                nonDefaultValues.addAll(element.getNonDefaultValues());
            }
            return nonDefaultValues;
        }

        interface ConfigElement {
            List<String> getNonDefaultValues();
        }

        /**
         * Represents the base URLs element specified in the Embrace config file.
         */
        static class App implements ConfigElement {
            static final boolean REPORT_DISK_USAGE_DEFAULT = true;

            /**
             * Control whether we scan for and report app disk usage. This can be a costly operation
             * for apps with a lot of local files.
             */
            @SerializedName("report_disk_usage")
            private boolean reportDiskUsage;

            App() {
                this(null);
            }

            App(Boolean reportDiskUsage) {
                this.reportDiskUsage = Optional.fromNullable(reportDiskUsage).or(REPORT_DISK_USAGE_DEFAULT);
            }

            boolean getReportDiskUsage() {
                return reportDiskUsage;
            }

            @Override
            public List<String> getNonDefaultValues() {
                List<String> nonDefaultValues = new ArrayList<>();
                if (REPORT_DISK_USAGE_DEFAULT != reportDiskUsage) {
                    nonDefaultValues.add("app.report_disk_usage: " + this.reportDiskUsage);
                }
                return nonDefaultValues;
            }
        }

        /**
         * Represents the base URLs element specified in the Embrace config file.
         */
        static class BaseUrls implements ConfigElement {
            static final String CONFIG_DEFAULT = "https://config.emb-api.com";
            static final String DATA_DEFAULT = "https://data.emb-api.com";
            static final String DATA_DEV_DEFAULT = "https://data-dev.emb-api.com";
            static final String IMAGES_DEFAULT = "https://images.emb-api.com";

            /**
             * Data base URL.
             */
            @SerializedName("data")
            private String data;

            /**
             * Data dev base URL.
             */
            @SerializedName("data_dev")
            private String dataDev;

            /**
             * Config base URL.
             */
            @SerializedName("config")
            private String config;

            /**
             * Images base URL.
             */
            @SerializedName("images")
            private String images;

            BaseUrls() {
                this(null, null, null, null);
            }

            BaseUrls(String config, String data, String dataDev, String images) {
                this.config = Optional.fromNullable(config).or(CONFIG_DEFAULT);
                this.data = Optional.fromNullable(data).or(DATA_DEFAULT);
                this.dataDev = Optional.fromNullable(dataDev).or(DATA_DEV_DEFAULT);
                this.images = Optional.fromNullable(images).or(IMAGES_DEFAULT);
            }

            String getConfig() {
                return config;
            }

            String getData() {
                return data;
            }

            String getDataDev() {
                return dataDev;
            }

            String getImages() {
                return images;
            }

            public List<String> getNonDefaultValues() {
                List<String> nonDefaultValues = new ArrayList<>();
                if (!config.equals(CONFIG_DEFAULT)) {
                    nonDefaultValues.add("base_urls.config: " + this.config);
                }
                if (!data.equals(DATA_DEFAULT)) {
                    nonDefaultValues.add("base_urls.data: " + this.data);
                }
                if (!dataDev.equals(DATA_DEV_DEFAULT)) {
                    nonDefaultValues.add("base_urls.data_dev: " + this.dataDev);
                }
                if (!images.equals(IMAGES_DEFAULT)) {
                    nonDefaultValues.add("base_urls.images: " + this.images);
                }
                return nonDefaultValues;
            }
        }

        /**
         * Represents the crash handler element specified in the Embrace config file.
         */
        static class CrashHandler implements ConfigElement {

            static final Boolean ENABLED_DEFAULT = true;

            /**
             * Control whether the Embrace SDK automatically attaches to the uncaught exception handler.
             */
            @SerializedName("enabled")
            private Boolean enabled;

            CrashHandler() {
                this(null);
            }

            CrashHandler(Boolean enabled) {
                this.enabled = Optional.fromNullable(enabled).or(ENABLED_DEFAULT);
            }

            Boolean getEnabled() {
                return enabled;
            }

            public List<String> getNonDefaultValues() {
                List<String> nonDefaultValues = new ArrayList<>();
                if (!enabled.equals(ENABLED_DEFAULT)) {
                    nonDefaultValues.add("crash_handler.enabled: " + enabled);
                }
                return nonDefaultValues;
            }
        }

        /**
         * Represents the startup moment configuration element specified in the Embrace config file.
         */
        static class StartupMoment implements ConfigElement {

            static final Boolean AUTOMATICALLY_END_DEFAULT = true;
            static final Boolean TAKE_SCREENSHOT_DEFAULT = true;

            /**
             * Control whether the startup moment is automatically ended.
             */
            @SerializedName("automatically_end")
            private Boolean automaticallyEnd;

            /**
             * Control whether startup moment screenshots are taken.
             */
            @SerializedName("take_screenshot")
            private Boolean takeScreenshot;


            StartupMoment() {
                this(null, null);
            }

            StartupMoment(Boolean automaticallyEnd, Boolean takeScreenshot) {
                this.automaticallyEnd = Optional.fromNullable(automaticallyEnd).or(AUTOMATICALLY_END_DEFAULT);
                this.takeScreenshot = Optional.fromNullable(takeScreenshot).or(TAKE_SCREENSHOT_DEFAULT);
            }

            Boolean getAutomaticallyEnd() {
                return automaticallyEnd;
            }

            Boolean getTakeScreenshot() {
                return takeScreenshot;
            }

            public List<String> getNonDefaultValues() {
                List<String> nonDefaultValues = new ArrayList<>();
                if (!automaticallyEnd.equals(AUTOMATICALLY_END_DEFAULT)) {
                    nonDefaultValues.add("startup_moment.automatically_end: " + automaticallyEnd);
                }
                if (!takeScreenshot.equals(TAKE_SCREENSHOT_DEFAULT)) {
                    nonDefaultValues.add("startup_moment.take_screenshot: " + takeScreenshot);
                }
                return nonDefaultValues;
            }
        }

        /**
         * Represents the networking configuration element specified in the Embrace config file.
         */
        static class Networking implements ConfigElement {

            /**
             * Sets the default name of the HTTP request header to extract trace ID from.
             */
            static final String CONFIG_TRACE_ID_HEADER_DEFAULT_VALUE = "x-emb-trace-id";

            /**
             * Capture request content length by default.
             */
            static final Boolean CAPTURE_REQUEST_CONTENT_LENGTH = false;
            /**
             * The default capture limit for the specified domains.
             */
            @SerializedName("default_capture_limit")
            private final Integer defaultCaptureLimit;
            /**
             * List of domains to be limited for tracking.
             */
            @SerializedName("domains")
            private final List<Domain> domains;
            /**
             * The Trace ID Header that can be used to trace a particular request.
             */
            @SerializedName("trace_id_header")
            private String traceIdHeader;
            /**
             * Control whether request size for native Android requests is captured.
             */
            @SerializedName("capture_request_content_length")
            private Boolean captureRequestContentLength;

            /**
             * URLs that should not be captured.
             */
            @SerializedName("disabled_url_patterns")
            private List<String> disabledUrlPatterns;

            private Set<Pattern> disabledUrlRegexPatterns;

            private ArrayList<String> nonDefaultValues;

            Networking() {
                this(null, null, null, null, null);
            }

            Networking(String traceIdHeader, Integer defaultCaptureLimit, List<Domain> domains,
                       Boolean captureRequestContentLength, List<String> disabledUrlPatterns) {
                this.traceIdHeader = Optional.fromNullable(traceIdHeader).or(CONFIG_TRACE_ID_HEADER_DEFAULT_VALUE);
                this.defaultCaptureLimit = defaultCaptureLimit;
                this.domains = domains;
                this.captureRequestContentLength = Optional.fromNullable(captureRequestContentLength).or(CAPTURE_REQUEST_CONTENT_LENGTH);
                this.disabledUrlPatterns = disabledUrlPatterns;
            }

            String getTraceIdHeader() {
                return traceIdHeader;
            }

            synchronized Set<Pattern> getDisabledUrlRegexPatterns() {
                if (disabledUrlRegexPatterns == null) {
                    if (disabledUrlPatterns != null) {
                        disabledUrlRegexPatterns = new HashSet<>();
                        for (String url : disabledUrlPatterns) {
                            try {
                                disabledUrlRegexPatterns.add(Pattern.compile(url));
                            } catch (PatternSyntaxException e) {
                                EmbraceLogger.logError("Skipping invalid blacklist URL: " + url, e);
                            }
                        }
                    } else {
                        disabledUrlRegexPatterns = Collections.emptySet();
                    }
                }
                return disabledUrlRegexPatterns;
            }

            Boolean getCaptureRequestContentLength() {
                return captureRequestContentLength;
            }

            public Optional<Integer> getDefaultCaptureLimit() {
                return Optional.fromNullable(defaultCaptureLimit);
            }

            public List<Domain> getDomains() {
                return domains != null ? domains : Collections.emptyList();
            }

            synchronized public List<String> getNonDefaultValues() {
                if (nonDefaultValues == null) {
                    nonDefaultValues = new ArrayList<>();
                    if (!traceIdHeader.equals(CONFIG_TRACE_ID_HEADER_DEFAULT_VALUE)) {
                        nonDefaultValues.add("networking.trace_id_header: " + traceIdHeader);
                    }
                    if (!captureRequestContentLength.equals(CAPTURE_REQUEST_CONTENT_LENGTH)) {
                        nonDefaultValues.add("networking.capture_request_content_length: " + captureRequestContentLength);
                    }
                    if (disabledUrlPatterns != null) {
                        for (String pattern : disabledUrlPatterns) {
                            nonDefaultValues.add("networking.disabled_url_pattern: " + pattern);
                        }
                    }
                }
                return nonDefaultValues;
            }

            /**
             * Represents each domain element specified in the Embrace config file.
             */
            class Domain {

                /**
                 * Url for the domain.
                 */
                @SerializedName("domain_name")
                private final String domain;

                /**
                 * Limit for the number of requests to be tracked.
                 */
                @SerializedName("domain_limit")
                private final Integer limit;

                Domain(String domain, Integer limit) {
                    this.domain = domain;
                    this.limit = limit;
                }

                public String getDomain() {
                    return domain;
                }

                public Integer getLimit() {
                    return limit;
                }
            }
        }

        /**
         * Represents the session configuration element specified in the Embrace config file.
         */
        static class SessionConfig implements ConfigElement {

            /**
             * Default minimum allowed end session time.
             */
            private static final int DEFAULT_MINIMUM_SESSION_SECONDS = 60;

            /**
             * Do not use async mode for session end messages by default.
             */
            private static final boolean DEFAULT_ASYNC_END = false;

            /**
             * Specify a maximum time before a session is allowed to exist before it is ended.
             */
            @SerializedName("max_session_seconds")
            private Integer maxSessionSeconds;

            /**
             * End session messages are sent asynchronously.
             */
            @SerializedName("async_end")
            private Boolean asyncEnd;

            SessionConfig() {
                this(null, null);
            }

            SessionConfig(Integer maxSessionSeconds, Boolean asyncEnd) {
                this.maxSessionSeconds = maxSessionSeconds;
                this.asyncEnd = asyncEnd;
            }

            Optional<Integer> getMaxSessionSeconds() {
                if (maxSessionSeconds != null) {
                    if (maxSessionSeconds >= DEFAULT_MINIMUM_SESSION_SECONDS) {
                        return Optional.of(maxSessionSeconds);
                    } else {
                        EmbraceLogger.logWarning("Automatic end session disabled. Config max_session_seconds should be more than 60 seconds.");
                        return Optional.absent();
                    }
                } else {
                    return Optional.absent();
                }
            }

            Boolean getAsyncEnd() {
                return asyncEnd != null ? asyncEnd : DEFAULT_ASYNC_END;
            }

            public List<String> getNonDefaultValues() {
                List<String> nonDefaultValues = new ArrayList<>();
                if (maxSessionSeconds != null) {
                    nonDefaultValues.add("session.max_session_seconds: " + maxSessionSeconds);
                }
                if (getAsyncEnd() != DEFAULT_ASYNC_END) {
                    nonDefaultValues.add("session.async_end: " + getAsyncEnd());
                }
                return nonDefaultValues;
            }
        }

        static class Taps implements ConfigElement {

            static final Boolean CAPTURE_COORDINATES_DEFAULT = true;

            /**
             * Control whether tap coordindates are captured.
             */
            @SerializedName("capture_coordinates")
            private Boolean captureCoordinates;

            Taps() {
                this(null);
            }

            Taps(Boolean captureCoordinates) {
                this.captureCoordinates = Optional.fromNullable(captureCoordinates).or(CAPTURE_COORDINATES_DEFAULT);
            }

            Boolean getCaptureCoordinates() {
                return captureCoordinates;
            }

            public List<String> getNonDefaultValues() {
                List<String> nonDefaultValues = new ArrayList<>();
                if (!captureCoordinates.equals(CAPTURE_COORDINATES_DEFAULT)) {
                    nonDefaultValues.add("taps.capture_coordinates: " + captureCoordinates);
                }
                return nonDefaultValues;
            }
        }
    }
}
