package io.embrace.android.embracesdk;

import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.Message;

import com.fernandocejas.arrow.checks.Preconditions;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Set;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.regex.Pattern;

/**
 * Checks whether the main thread is still responding by using the following strategy:
 * <ol>
 * <li>Creating two {@link Handler}s, one on the main thread, and one on a new thread</li>
 * <li>Using the 'monitoring' thread to message the main thread with a heartbeat</li>
 * <li>Determining whether the main thread responds in time, and if not logging an ANR</li>
 * </ol>
 */
final class EmbraceAnrService implements AnrService, MemoryCleanerListener {

    private static final String MAIN_THREAD_NAME = "main";

    private static final long ANR_THRESHOLD_INTERVAL = 1000L;

    private static final int HEALTHCHECK_EXECUTE = 34592;

    private static final int HEALTHCHECK_REQUEST = 34593;

    private static final int HEALTHCHECK_RESPONSE = 34594;

    private final Clock clock;

    private final ConfigService configService;

    private final NavigableMap<Long, AnrInterval> anrIntervals;

    private final MainThreadHandler mainThreadHandler;

    private final MonitoringThreadHandler monitoringThreadHandler;

    private final HandlerThread monitoringThread;

    private volatile long lastAlive;

    private volatile boolean anrInProgress;

    private final HashMap<Long, Integer> currentStacktraceStates = new HashMap<>();

    private AnrStacktraces stacktraces;

    EmbraceAnrService(
            Clock clock,
            MemoryCleanerService memoryCleanerService,
            ConfigService configService) {
        this.clock = clock;
        this.configService = Preconditions.checkNotNull(configService);
        this.anrIntervals = new ConcurrentSkipListMap<>();
        this.monitoringThread = new HandlerThread("Embrace ANR Healthcheck");
        this.monitoringThread.start();
        Looper callingThreadLooper = this.monitoringThread.getLooper();
        this.monitoringThreadHandler = new MonitoringThreadHandler(callingThreadLooper);
        this.mainThreadHandler = new MainThreadHandler(Looper.getMainLooper());
        this.monitoringThreadHandler.runHealthcheck();
        Preconditions.checkNotNull(memoryCleanerService).addListener(this);

        stacktraces = new AnrStacktraces();
    }

    @Override
    public List<AnrInterval> getAnrIntervals(long startTime, long endTime) {
        synchronized (this) {
            Collection<AnrInterval> intervals = anrIntervals.subMap(startTime, endTime).values();
            ArrayList<AnrInterval> results = new ArrayList<>(intervals);

            if (anrInProgress) {
                AnrInterval.Builder anrIntervalBuilder = AnrInterval.newBuilder();
                anrIntervalBuilder
                        .withStartTime(lastAlive)
                        .withLastKnownTime(clock.now())
                        .withEndTime(null)
                        .withType(AnrInterval.Type.UI);

                if (configService.isAnrCaptureEnabled() && !isAnrCaptureLimitReached()) {
                    anrIntervalBuilder.withStacktraces(stacktraces);
                }

                results.add(anrIntervalBuilder.build());
            }
            return results;
        }
    }

    @Override
    public void close() {
        try {
            monitoringThread.quit();
            monitoringThread.join();
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to cleanly shut down EmbraceAnrService");
        }
    }

    @Override
    public void cleanCollections() {
        anrIntervals.clear();
    }

    /**
     * Performs the monitoring by submitting heartbeats to the {@link MainThreadHandler}, then
     * re-scheduling itself for another heartbeat by sending a message to itself with a delay.
     */
    class MonitoringThreadHandler extends Handler {

        public MonitoringThreadHandler(Looper looper) {
            super(looper);
        }

        @Override
        public void handleMessage(Message msg) {
            try {
                long timestamp = clock.now();
                boolean anrThresholdExceeded = lastAlive > 0 && (timestamp - lastAlive) > ANR_THRESHOLD_INTERVAL;
                if (msg.what == EmbraceAnrService.HEALTHCHECK_RESPONSE) {
                    if (anrThresholdExceeded) {
                        // Application was not responding, but recovered
                        EmbraceLogger.logDebug("Main thread recovered from not responding for > 1s");
                        synchronized (EmbraceAnrService.this) {
                            // Finalize AnrInterval
                            AnrInterval.Builder anrIntervalBuilder = AnrInterval.newBuilder();
                            anrIntervalBuilder
                                    .withStartTime(lastAlive)
                                    .withLastKnownTime(timestamp)
                                    .withEndTime(timestamp)
                                    .withType(AnrInterval.Type.UI);

                            if (configService.isAnrCaptureEnabled() && !isAnrCaptureLimitReached()) {
                                anrIntervalBuilder.withStacktraces(stacktraces);
                            }

                            anrIntervals.put(timestamp, anrIntervalBuilder.build());

                            //reset counters
                            stacktraces = new AnrStacktraces();
                            currentStacktraceStates.clear();
                        }
                    }
                    lastAlive = timestamp;
                    anrInProgress = false;
                } else if (msg.what == EmbraceAnrService.HEALTHCHECK_EXECUTE) {
                    // Application is not responding
                    if (anrThresholdExceeded && !anrInProgress) {
                        EmbraceLogger.logDebug("Main thread not responding for > 1s");
                        currentStacktraceStates.clear();
                        anrInProgress = true;
                    }

                    if (anrInProgress) {
                        processAnrTick();
                    }

                    runHealthcheck();
                }
            } catch (Exception ex) {
                EmbraceLogger.logDebug("ANR healthcheck failed in monitoring thread", ex);
            }
        }

        void runHealthcheck() {
            mainThreadHandler.sendMessage(obtainMessage(EmbraceAnrService.HEALTHCHECK_REQUEST));
            monitoringThreadHandler.sendMessageDelayed(
                    obtainMessage(EmbraceAnrService.HEALTHCHECK_EXECUTE),
                    configService.getConfig().getCaptureAnrIntervalMs());
        }

        private synchronized void processAnrTick() {
            // Check if Anr capture is enabled
            if (!configService.isAnrCaptureEnabled()) {
                return;
            }

            // ANR Interval limit
            if (isAnrCaptureLimitReached()) {
                EmbraceLogger.logDebug("ANR stacktrace not captured. Maximum allowed anr intervals per session reached.");
                return;
            }

            // Tick limit
            if (stacktraces.size() >= configService.getConfig().getStacktracesPerInterval()) {
                EmbraceLogger.logDebug("ANR stacktrace not captured. Maximum allowed ticks per anr interval reached.");
                return;
            }

            AnrTick anrTick = new AnrTick(clock.now());

            for (ThreadInfo threadInfo : getAllowedThreads()) {
                // Compares every thread with the last known thread state via hashcode. If hashcode changed
                // it should be added to the anrInfo list and also the currentAnrInfoState must be updated.
                Integer threadHash = currentStacktraceStates.get(threadInfo.getThreadId());
                if (threadHash != null) {
                    if (threadHash != threadInfo.hashCode()) {
                        updateThread(threadInfo, anrTick);
                    }
                } else {
                    updateThread(threadInfo, anrTick);
                }
            }

            stacktraces.add(anrTick);
        }

        private void updateThread(ThreadInfo threadInfo, AnrTick anrTick) {
            currentStacktraceStates.put(threadInfo.getThreadId(), threadInfo.hashCode());
            anrTick.add(threadInfo);
        }

        /**
         * Filter the thread list based on black/white list get by config.
         *
         * @return filtered threads
         */
        private Set<ThreadInfo> getAllowedThreads() {
            Set<ThreadInfo> allowed = new HashSet<>();
            List<String> blackList = configService.getConfig().getAnrBlackList();
            List<String> whiteList = configService.getConfig().getAnrWhiteList();
            int anrStacktracesMaxLength = configService.getConfig().getAnrStacktracesMaxDepth();
            int priority = configService.getConfig().getAnrThreadCapturePriority();

            for (Map.Entry<Thread, StackTraceElement[]> entry : Thread.getAllStackTraces().entrySet()) {
                ThreadInfo threadInfo = ThreadInfo.ofThread(
                        entry.getKey(),
                        entry.getValue(),
                        anrStacktracesMaxLength);

                // First, check if only the main thread should be captured.
                if (configService.getConfig().captureMainThreadOnly()) {
                    if (entry.getKey().getName().equals(MAIN_THREAD_NAME)
                            && isAllowedByPriority(priority, threadInfo)) {
                        allowed.add(threadInfo);
                        break;
                    }
                } else {
                    if (whiteList != null && !whiteList.isEmpty()) {
                        if (isAllowedByList(whiteList, threadInfo) &&
                                isAllowedByPriority(priority, threadInfo)) {
                            allowed.add(threadInfo);
                        }
                    } else if (blackList != null && !blackList.isEmpty()) {
                        if (!isAllowedByList(blackList, threadInfo)
                                && isAllowedByPriority(priority, threadInfo)) {
                            allowed.add(threadInfo);
                        }
                    } else {
                        if (isAllowedByPriority(priority, threadInfo)) {
                            allowed.add(threadInfo);
                        }
                    }
                }
            }
            return allowed;
        }

        private boolean isAllowedByList(List<String> allowed, ThreadInfo threadInfo) {
            for (String regex : allowed) {
                if (Pattern.compile(regex).matcher(threadInfo.getName()).find()) {
                    return true;
                }
            }

            return false;
        }

        private boolean isAllowedByPriority(int priority, ThreadInfo threadInfo) {
            if (priority != 0) {
                return threadInfo.getPriority() >= priority;
            }

            return true;
        }
    }

    private boolean isAnrCaptureLimitReached() {
        return anrIntervals.size() >= configService.getConfig().getMaxAnrCapturedIntervalsPerSession();
    }

    /**
     * Handles healthcheck messages sent to the main thread. Responds with an acknowledgement to the
     * calling thread.
     */
    class MainThreadHandler extends Handler {

        public MainThreadHandler(Looper looper) {
            super(looper);
        }

        @Override
        public void handleMessage(Message msg) {
            try {
                if (msg.what == EmbraceAnrService.HEALTHCHECK_REQUEST) {
                    monitoringThreadHandler.sendMessage(obtainMessage(EmbraceAnrService.HEALTHCHECK_RESPONSE));
                }
            } catch (Exception ex) {
                EmbraceLogger.logDebug("ANR healthcheck failed in main (monitored) thread", ex);
            }
            super.handleMessage(msg);
        }
    }
}
