package io.embrace.android.embracesdk;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URL;

import io.embrace.android.embracesdk.network.http.HttpMethod;
import io.embrace.android.embracesdk.networking.EmbraceConnection;
import io.embrace.android.embracesdk.networking.EmbraceUrl;
import io.embrace.android.embracesdk.utils.exceptions.Unchecked;

/**
 * Describes a HTTP request to the Embrace API.
 * <p>
 * The request can be converted to a {@link HttpURLConnection}.
 */
final class ApiRequest {
    private final String contentType;

    private final String userAgent;

    private final String contentEncoding;

    private final String accept;

    private final String acceptEncoding;

    private final String appId;

    private final String deviceId;

    private final String eventId;

    private final String logId;

    private final EmbraceUrl url;

    private final HttpMethod httpMethod;

    private ApiRequest(Builder builder) {
        contentType = builder.contentType;
        userAgent = builder.userAgent;
        contentEncoding = builder.contentEncoding;
        acceptEncoding = builder.acceptEncoding;
        accept = builder.accept;
        appId = builder.appId;
        deviceId = builder.deviceId;
        eventId = builder.eventId;
        logId = builder.logId;
        url = builder.url;
        httpMethod = builder.httpMethod;
    }

    static Builder newBuilder() {
        return new Builder();
    }

    /**
     * Converts a {@link ApiRequest} to a {@link HttpURLConnection} to be used for sending a request
     * to the API server.
     *
     * @return an instance of {@link HttpURLConnection} created from an {@link ApiRequest}
     */
    EmbraceConnection toConnection() {
        try {
            EmbraceConnection connection = url.openConnection();
            if (accept != null) {
                connection.setRequestProperty("Accept", accept);
            }
            if (userAgent != null) {
                connection.setRequestProperty("User-Agent", userAgent);
            }
            if (contentType != null) {
                connection.setRequestProperty("Content-Type", contentType);
            }
            if (contentEncoding != null) {
                connection.setRequestProperty("Content-Encoding", contentEncoding);
            }
            if (acceptEncoding != null) {
                connection.setRequestProperty("Accept-Encoding", acceptEncoding);
            }
            if (appId != null) {
                connection.setRequestProperty("X-EM-AID", appId);
            }
            if (deviceId != null) {
                connection.setRequestProperty("X-EM-DID", deviceId);
            }
            if (eventId != null) {
                connection.setRequestProperty("X-EM-SID", eventId);
            }
            if (logId != null) {
                connection.setRequestProperty("X-EM-LID", logId);
            }
            connection.setRequestMethod(httpMethod.name());
            if (httpMethod.equals(HttpMethod.POST)) {
                connection.setDoOutput(true);
            }
            return connection;
        } catch (IOException ex) {
            throw Unchecked.propagate(ex);
        }
    }

    static final class Builder {
        private String contentType = "application/json";
        // Default user agent to the version name
        private String userAgent = String.format("Embrace/a/%s", BuildConfig.VERSION_NAME);
        private String contentEncoding;
        private String accept = "application/json";
        private String acceptEncoding;
        private String appId;
        private String deviceId;
        private String eventId;
        private String logId;
        private EmbraceUrl url;
        private HttpMethod httpMethod = HttpMethod.POST;

        private Builder() {
        }

        public Builder withContentType(String contentType) {
            this.contentType = contentType;
            return this;
        }

        public Builder withUserAgent(String userAgent) {
            this.userAgent = userAgent;
            return this;
        }

        public Builder withContentEncoding(String contentEncoding) {
            this.contentEncoding = contentEncoding;
            return this;
        }

        public Builder withAcceptEncoding(String acceptEncoding) {
            this.acceptEncoding = acceptEncoding;
            return this;
        }

        public Builder withAccept(String accept) {
            this.accept = accept;
            return this;
        }

        public Builder withAppId(String appId) {
            this.appId = appId;
            return this;
        }

        public Builder withDeviceId(String deviceId) {
            this.deviceId = deviceId;
            return this;
        }

        public Builder withEventId(String eventId) {
            this.eventId = eventId;
            return this;
        }

        public Builder withLogId(String logId) {
            this.logId = logId;
            return this;
        }

        public Builder withUrl(EmbraceUrl url) {
            this.url = url;
            return this;
        }

        public Builder withHttpMethod(HttpMethod httpMethod) {
            this.httpMethod = httpMethod;
            return this;
        }

        public ApiRequest build() {
            return new ApiRequest(this);
        }
    }
}
