package io.embrace.android.embracesdk.network;

import android.util.Base64;

import androidx.annotation.Nullable;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import io.embrace.android.embracesdk.EmbraceLogger;

public class NetworkCaptureData {

    /**
     * The request headers captured from the https connection.
     */
    private volatile HashMap<String, String> requestHeaders;

    /**
     * The request query params captured from the https connection.
     */
    private volatile String requestQueryParams;

    /**
     * The request body from the https connection.
     */
    private int requestBodyLength;

    /**
     * The request body captured from the https connection.
     */
    private String capturedRequestBody;

    /**
     * The response headers captured from the https connection.
     */
    private volatile HashMap<String, String> responseHeaders;

    /**
     * The response body from the https connection.
     */
    private int responseBodyLength;

    /**
     * The response body captured from the https connection.
     */
    private String capturedResponseBody;

    public NetworkCaptureData withRequestHeaders(Map<String, String> requestHeaders) {
        this.requestHeaders = (HashMap<String, String>) requestHeaders;
        return this;
    }

    public NetworkCaptureData withResponseHeaders(Map<String, String> responseHeaders) {
        this.responseHeaders = (HashMap<String, String>) responseHeaders;
        return this;
    }

    public NetworkCaptureData withRequestQueryParams(String requestQueryParams) {
        this.requestQueryParams = requestQueryParams;
        return this;
    }

    public NetworkCaptureData withRequestBody(byte[] requestBody) {
        this.requestBodyLength = requestBody != null ? requestBody.length : 0;
        this.capturedRequestBody = parseBody(requestBody);
        return this;
    }

    public NetworkCaptureData withResponseBody(byte[] responseBody) {
        this.responseBodyLength = responseBody != null ? responseBody.length : 0;
        this.capturedResponseBody = parseBody(responseBody);
        return this;
    }

    public @Nullable
    Map<String, String> getRequestHeaders() {
        return requestHeaders;
    }

    public String getRequestQueryParams() {
        return requestQueryParams;
    }

    public String getCapturedRequestBody() {
        return capturedRequestBody;
    }

    public Integer getCapturedRequestBodySize() {
        return capturedRequestBody != null ? capturedRequestBody.length() : 0;
    }

    public Integer getRequestBodySize() {
        return requestBodyLength;
    }

    public @Nullable
    Map<String, String> getResponseHeaders() {
        return responseHeaders;
    }

    public String getCapturedResponseBody() {
        return capturedResponseBody;
    }

    public Integer getCapturedResponseBodySize() {
        return capturedResponseBody != null ? capturedResponseBody.length() : 0;
    }

    public Integer getResponseBodySize() {
        return responseBodyLength;
    }

    public void limitBodySize(int size) {
        if (size != 0 && capturedRequestBody != null && capturedRequestBody.length() > size) {
            capturedRequestBody = capturedRequestBody.substring(0, size);
        }

        if (size != 0 && capturedResponseBody != null && capturedResponseBody.length() > size) {
            capturedResponseBody = capturedResponseBody.substring(0, size);
        }
    }

    @Nullable
    private String parseBody(@Nullable byte[] body) {

        if (body == null) {
            return null;
        }

        try {
            if (body.length > NetworkBodyRule.NETWORK_BODY_RULE_DEFAULT_MAX_SIZE) {
                body = Arrays.copyOf(body, NetworkBodyRule.NETWORK_BODY_RULE_DEFAULT_MAX_SIZE);
            }
            return Base64.encodeToString(body, Base64.NO_WRAP);
        } catch (Exception e) {
            EmbraceLogger.logDebug("Parsing body failed.", e);
            return null;
        }
    }
}
