package io.embrace.android.embracesdk.network;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.gson.JsonArray;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;

import java.lang.reflect.Type;
import java.util.HashSet;
import java.util.Locale;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import io.embrace.android.embracesdk.EmbraceLogger;

public class NetworkBodyRule {

    private static final int NETWORK_BODY_RULE_DEFAULT_MAX_COUNT = 5;

    private static final int NETWORK_BODY_RULE_MIN_DURATION_MS = 5000;

    public static final int NETWORK_BODY_RULE_DEFAULT_MAX_SIZE = 102400;

    private static final String NETWORK_BODY_RULE_ID_KEY = "id";
    private static final String NETWORK_BODY_RULE_URL_KEY = "url";
    private static final String NETWORK_BODY_RULE_METHOD_KEY = "method";
    private static final String NETWORK_BODY_RULE_STATUS_CODES_KEY = "status_codes";
    private static final String NETWORK_BODY_RULE_EXPIRES_IN_KEY = "expires_in";
    private static final String NETWORK_BODY_RULE_MAX_SIZE_KEY = "max_size";
    private static final String NETWORK_BODY_RULE_MAX_COUNT_KEY = "max_count";
    private static final String NETWORK_BODY_RULE_DURATION_KEY = "duration";

    private String url;

    private HashSet<Integer> statusCodes;

    private String method;

    private int maxSize;

    private int duration;

    private String ruleId;

    private Long expireDate;

    private int maxCount;

    private int remainingCount;

    private Matcher matcher;

    private NetworkBodyRule() {
    }

    public String getUrl() {
        return url;
    }

    public Set<Integer> getStatusCodes() {
        return statusCodes;
    }

    public String getMethod() {
        return method;
    }

    public Integer getMaxSize() {
        return maxSize;
    }

    public Integer getDuration() {
        return duration;
    }

    public String getRuleId() {
        return ruleId;
    }

    public Long getExpireDate() {
        return expireDate;
    }

    public Integer getMaxCount() {
        return maxCount;
    }

    public Integer getRemainingCount() {
        return remainingCount;
    }

    public synchronized void updateRemainingCount() {
        this.remainingCount -= 1;
    }

    // Call find() to scans the url looking for the next subsequence that matches the pattern.
    private boolean urlMatch(String url) {

        if (this.matcher == null) {
            return false;
        }

        this.matcher.reset(url);
        return matcher.find();
    }

    public void update(NetworkBodyRule existingRule) {
        if (existingRule.getMaxCount().equals(this.getMaxCount())) {
            this.remainingCount = existingRule.getRemainingCount();
        }
    }

    public boolean shouldBeUsedToCapture(@NonNull String url, @NonNull String method) {
        if (this.method == null) {
            return false;
        }

        return this.method.equals(method) && this.urlMatch(url);
    }

    public boolean isStatusCodeCapturable(int statusCode) {
        if (this.statusCodes == null || this.statusCodes.isEmpty()) {
            return false;
        }

        return this.statusCodes.contains(statusCode);
    }

    public static class NetworkBodyRuleDeserializer implements JsonDeserializer<NetworkBodyRule> {

        @Override
        public synchronized NetworkBodyRule deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context) throws JsonParseException {
            JsonObject jsonObject = json.getAsJsonObject();

            NetworkBodyRule rule = new NetworkBodyRule();

            if (jsonObject.has(NETWORK_BODY_RULE_ID_KEY)) {
                rule.ruleId = jsonObject.get(NETWORK_BODY_RULE_ID_KEY).getAsString();
            } else {
                rule.ruleId = "";
                EmbraceLogger.logWarning("One body capture rule has an empty id.");
            }

            String url = "";
            if (jsonObject.has(NETWORK_BODY_RULE_URL_KEY)) {
                url = jsonObject.get(NETWORK_BODY_RULE_URL_KEY).getAsString();
                if (url != null && !url.isEmpty()) {
                    rule.url = url;
                    // Initialize the matcher with a url.
                    rule.matcher = Pattern.compile(url, Pattern.DOTALL).matcher(url);
                } else {
                    EmbraceLogger.logWarning(String.format("The rule %s has an invalid url. The rule will not be used.", rule.ruleId));
                }
            } else {
                EmbraceLogger.logWarning(String.format("The rule %s doesn't contain an url. The rule will not be used.", rule.ruleId));
            }

            if (jsonObject.has(NETWORK_BODY_RULE_METHOD_KEY)) {
                String method = jsonObject.get(NETWORK_BODY_RULE_METHOD_KEY).getAsString();
                if (method != null && !method.isEmpty()) {
                    rule.method = method;
                } else {
                    EmbraceLogger.logWarning(String.format("The rule %s has an invalid http method. The rule will not be used.", rule.ruleId));
                }
            } else {
                EmbraceLogger.logWarning(String.format("The rule %s doesn't contain a http method. The rule will not be used.", rule.ruleId));
            }

            if (jsonObject.has(NETWORK_BODY_RULE_STATUS_CODES_KEY)) {
                rule.statusCodes = new HashSet<>();
                JsonArray statusCodes = jsonObject.getAsJsonArray(NETWORK_BODY_RULE_STATUS_CODES_KEY);

                for (int i = 0; i < statusCodes.size(); i++) {
                    rule.statusCodes.add(statusCodes.get(i).getAsInt());
                }
            }

            if (jsonObject.has(NETWORK_BODY_RULE_EXPIRES_IN_KEY)) {
                rule.expireDate = System.currentTimeMillis() +
                        TimeUnit.SECONDS.toMillis(jsonObject.get(NETWORK_BODY_RULE_EXPIRES_IN_KEY).getAsInt());
            } else {
                //By default this expiration time will be set to 1 day.
                rule.expireDate = System.currentTimeMillis() + TimeUnit.DAYS.toMillis(1);
            }

            if (jsonObject.has(NETWORK_BODY_RULE_MAX_SIZE_KEY)) {
                rule.maxSize = jsonObject.get(NETWORK_BODY_RULE_MAX_SIZE_KEY).getAsInt();
            } else {
                rule.maxSize = NETWORK_BODY_RULE_DEFAULT_MAX_SIZE;
            }

            if (jsonObject.has(NETWORK_BODY_RULE_DURATION_KEY)) {
                int duration = jsonObject.get(NETWORK_BODY_RULE_DURATION_KEY).getAsInt();
                if (duration >= NETWORK_BODY_RULE_MIN_DURATION_MS) {
                    rule.duration = duration;
                } else {
                    EmbraceLogger.logDebug(String.format(Locale.getDefault(),
                            "Network capture rule for %s has duration %s ms, which is less than the allowed minimum %d ms. Duration will be ignored.",
                            url,
                            duration,
                            NETWORK_BODY_RULE_MIN_DURATION_MS));
                }
            }

            if (jsonObject.has(NETWORK_BODY_RULE_MAX_COUNT_KEY)) {
                rule.maxCount = jsonObject.get(NETWORK_BODY_RULE_MAX_COUNT_KEY).getAsInt();
            } else {
                rule.maxCount = NETWORK_BODY_RULE_DEFAULT_MAX_COUNT;
            }

            rule.remainingCount = rule.maxCount;

            return rule;
        }
    }

    @Override
    public boolean equals(@Nullable Object obj) {
        if (this.ruleId == null) {
            return false;
        }

        if (obj == null) {
            return false;
        }

        if (this.getClass() != obj.getClass())
            return false;

        if (((NetworkBodyRule) obj).ruleId == null) {
            return false;
        }

        return ((NetworkBodyRule) obj).ruleId.equals(this.ruleId);
    }

    @Override
    public int hashCode() {
        return ruleId != null ? ruleId.hashCode() : 0;
    }
}
