package io.embrace.android.embracesdk;

import androidx.annotation.NonNull;

import java.util.List;

import io.embrace.android.embracesdk.network.NetworkCaptureData;

/**
 * Logs network calls made by the application. The Embrace SDK intercepts the calls and reports
 * them to the API.
 */
public interface NetworkLoggingService {

    /**
     * Get the calls and counts of network calls (which exceed the limit) within the specified time
     * range.
     *
     * @param startTime     the start time
     * @param lastKnownTime the end time
     * @return
     */
    NetworkSessionV2 getNetworkSession(long startTime, long lastKnownTime);

    /**
     * Get all of the network calls which occurred within the specified time range.
     *
     * @param startTime the start time of the range
     * @param endTime   the end time of the range
     * @return the list of network calls
     */
    List<NetworkCallV2> getNetworkCalls(long startTime, long endTime);

    /**
     * Logs a HTTP network call.
     *
     * @param url           the URL being called
     * @param httpMethod    the HTTP method
     * @param statusCode    the status code from the response
     * @param startTime     the start time of the request
     * @param endTime       the end time of the request
     * @param bytesSent     the number of bytes sent
     * @param bytesReceived the number of bytes received
     * @param traceId       optional trace ID that can be used to trace a particular request
     */
    void logNetworkCall(
            String url,
            String httpMethod,
            Integer statusCode,
            long startTime,
            long endTime,
            Long bytesSent,
            Long bytesReceived,
            String traceId);

    /**
     * Logs a HTTP network call.
     *
     * @param url                the URL being called
     * @param httpMethod         the HTTP method
     * @param statusCode         the status code from the response
     * @param startTime          the start time of the request
     * @param endTime            the end time of the request
     * @param bytesSent          the number of bytes sent
     * @param bytesReceived      the number of bytes received
     * @param traceId            optional trace ID that can be used to trace a particular request
     * @param networkCaptureData the additional data captured if network body capture is enabled for the URL
     */
    void logNetworkCall(
            String url,
            String httpMethod,
            Integer statusCode,
            long startTime,
            long endTime,
            Long bytesSent,
            Long bytesReceived,
            String traceId,
            NetworkCaptureData networkCaptureData);

    /**
     * Logs an exception which occurred when attempting to make a network call.
     *
     * @param url          the URL being called
     * @param httpMethod   the HTTP method
     * @param startTime    the start time of the request
     * @param endTime      the end time of the request
     * @param errorType    the type of error being thrown
     * @param errorMessage the error message
     * @param traceId      optional trace ID that can be used to trace a particular request
     */
    void logNetworkError(
            String url,
            String httpMethod,
            long startTime,
            long endTime,
            String errorType,
            String errorMessage,
            String traceId);

    /**
     * Logs an exception which occurred when attempting to make a network call.
     *
     * @param url                the URL being called
     * @param httpMethod         the HTTP method
     * @param startTime          the start time of the request
     * @param endTime            the end time of the request
     * @param errorType          the type of error being thrown
     * @param errorMessage       the error message
     * @param traceId            optional trace ID that can be used to trace a particular request
     * @param networkCaptureData the additional data captured if network body capture is enabled for the URL
     */
    void logNetworkError(
            String url,
            String httpMethod,
            long startTime,
            long endTime,
            String errorType,
            String errorMessage,
            String traceId,
            NetworkCaptureData networkCaptureData);

    /**
     * Check if the current network request should be captured base on the capturing rules provided
     * via configuration service.
     *
     * @param url    the url which will be matched to the rule's regex
     * @param method the method which will be compared to the rule's available methods
     * @return if the current network request should be captured
     */
    boolean shouldCaptureNetworkBody(@NonNull String url, @NonNull String method);
}
