package io.embrace.android.embracesdk;

import com.fernandocejas.arrow.checks.Preconditions;
import com.fernandocejas.arrow.optional.Optional;

import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.regex.Pattern;

import io.embrace.android.embracesdk.network.NetworkBodyRule;
import java9.util.stream.StreamSupport;

/**
 * Loads configuration for the app from the Embrace API.
 */
final class EmbraceConfigService implements ConfigService, ActivityListener {

    /**
     * Config lives for 1 hour before attempting to retrieve again.
     */
    private static final long CONFIG_TTL = 60 * 60 * 1000L;

    /**
     * Name of the file used to cache config values.
     */
    private static final String CONFIG_FILE_NAME = "config.json";

    /**
     * Config refresh default retry period.
     */
    private static final long DEFAULT_RETRY_WAIT_TIME = 2; // 2 seconds

    /**
     * Config max allowed refresh retry period.
     */
    private static final long MAX_ALLOWED_RETRY_WAIT_TIME = 300; // 5 minutes

    /**
     * The listeners subscribed to configuration changes.
     */
    private final Set<ConfigListener> listeners = new CopyOnWriteArraySet<>();
    private final BackgroundWorker worker = BackgroundWorker.ofSingleThread("SDK Configuration");
    private final ApiClient apiClient;
    private final CacheService cacheService;
    private final MetadataService metadataService;

    private final Object lock = new Object();

    private volatile Config config = Config.ofDefault();
    private volatile long lastUpdated;
    private volatile long lastRefreshConfigAttempt;
    private volatile double configRetrySafeWindow = DEFAULT_RETRY_WAIT_TIME;

    EmbraceConfigService(
            ApiClient apiClient,
            EmbraceActivityService activityService,
            CacheService cacheService,
            MetadataService metadataService) {

        this.apiClient = Preconditions.checkNotNull(apiClient, "apiClient must not be null");
        this.cacheService = Preconditions.checkNotNull(cacheService, "cacheService must not be null");
        this.metadataService = Preconditions.checkNotNull(metadataService, "metadataService must not be null");
        Preconditions.checkNotNull(activityService).addListener(this);
        Optional<Config> optionalConfig = cacheService.loadObject(CONFIG_FILE_NAME, Config.class);
        if (optionalConfig.isPresent()) {
            config = optionalConfig.get();
        }
        getConfig();
    }

    @Override
    public Config getConfig() {
        if (configRequiresRefresh() && configRetryIsSafe()) {
            // Attempt to asynchronously update the config if it is out of date
            refreshConfigByService();
        }
        return config;
    }

    private void refreshConfigByService() {
        worker.submit((Callable<Object>) () -> {
            synchronized (lock) {
                // Ensure that another thread didn't refresh it already in the meantime
                if (configRequiresRefresh() && configRetryIsSafe()) {
                    Config previousConfig = config;
                    try {
                        lastRefreshConfigAttempt = System.currentTimeMillis();
                        // Update new rules based on old rules.
                        config = apiClient.getConfig().get();
                        if (previousConfig.getCaptureRules() != null && !previousConfig.getCaptureRules().isEmpty()) {
                            config.updateNewRules(previousConfig.getCaptureRules());
                        }
                        cacheConfig(previousConfig);
                    } catch (Exception ex) {
                        configRetrySafeWindow = Math.min(MAX_ALLOWED_RETRY_WAIT_TIME, configRetrySafeWindow * 2);
                        EmbraceLogger.logWarning("Failed to load SDK config from the server. " +
                                "Trying again in " + configRetrySafeWindow + " seconds.");
                    }
                }
                return config;
            }
        });
    }

    // Should be called inside of a worker
    private void cacheConfig(Config previousConfig) {
        cacheService.cacheObject(CONFIG_FILE_NAME, config, Config.class);
        lastUpdated = System.currentTimeMillis();
        if (!config.equals(previousConfig)) {
            // Only notify listeners if the config has actually changed value
            notifyListeners(previousConfig, config);
        }
        configRetrySafeWindow = DEFAULT_RETRY_WAIT_TIME;
    }

    @Override
    public boolean isScreenshotDisabledForEvent(String eventName) {
        return doesStringMatchesPatternInSet(eventName, config.getDisabledScreenshotPatterns());
    }

    @Override
    public boolean isEventDisabled(String eventName) {
        return doesStringMatchesPatternInSet(eventName, config.getDisabledEventAndLogPatterns());
    }

    @Override
    public boolean isLogMessageDisabled(String logMessage) {
        return doesStringMatchesPatternInSet(logMessage, config.getDisabledEventAndLogPatterns());
    }

    @Override
    public boolean isMessageTypeDisabled(MessageType type) {
        return config.getDisabledMessageTypes().contains(type.name().toLowerCase());
    }

    @Override
    public boolean isUrlDisabled(String url) {
        return doesStringMatchesPatternInSet(url, config.getDisabledUrlPatterns());
    }

    @Override
    public boolean isInternalExceptionCaptureEnabled() {
        return config.getInternalExceptionCaptureEnabled();
    }

    @Override
    public boolean isSdkDisabled() {
        Config sdkConfig = getConfig();

        float result = getNormalizedDeviceId();
        //Check if this is lower than the threshold, to determine whether
        // we should enable/disable the SDK.
        int lowerBound = Math.max(0, sdkConfig.getOffset());
        int upperBound = Math.min(sdkConfig.getOffset() + sdkConfig.getThreshold(), 100);
        return lowerBound == upperBound || result < lowerBound || result > upperBound;
    }

    @Override
    public boolean isAnrCaptureEnabled() {
        return getConfig().getAnrUsersEnabledPercentage() >= getNormalizedDeviceId();
    }

    public void addListener(ConfigListener configListener) {
        listeners.add(configListener);
    }

    @Override
    public void removeListener(ConfigListener configListener) {
        listeners.remove(configListener);
    }

    @Override
    public boolean isSessionControlEnabled() {
        return config.getSessionControl();
    }

    @Override
    public void onForeground(boolean coldStart, long startupTime) {
        // Refresh the config on resume if it has expired
        getConfig();
        if (Embrace.getInstance().isStarted() && isSdkDisabled()) {
            EmbraceLogger.logInfo("Embrace SDK disabled by config");
            Embrace.getInstance().stop();
        }
    }

    @Override
    public void updateRuleInConfig(NetworkBodyRule rule) {
        worker.submit(() -> {
            if (config.updateRule(rule)) {
                synchronized (lock) {
                    // Attempt to asynchronously save the config
                    cacheConfig(config);
                }
            }
            return null;
        });
    }

    /**
     * Notifies the listeners that a new config was fetched from the server.
     *
     * @param previousConfig old config
     * @param newConfig      new config
     */
    private void notifyListeners(Config previousConfig, Config newConfig) {
        StreamSupport.stream(listeners).forEach(listener -> {
            try {
                listener.onConfigChange(previousConfig, newConfig);
            } catch (Exception ex) {
                EmbraceLogger.logDebug("Failed to notify ConfigListener", ex);
            }
        });
    }

    /**
     * Analyzes if the string matches any of the patterns provided in the set.
     *
     * @param string     string of characters to be analyzed.
     * @param patternSet set of patterns to analyze the string against.
     * @return if the string matches any of the patterns in the set.
     */
    private boolean doesStringMatchesPatternInSet(String string, Set<String> patternSet) {
        return StreamSupport.stream(patternSet)
                .map(Pattern::compile)
                .anyMatch(p -> p.matcher(string).matches());
    }

    /**
     * Checks if the time diff since the last fetch exceeds the
     * {@link EmbraceConfigService#CONFIG_TTL} millis.
     *
     * @return if the config requires to be fetched from the remote server again or not.
     */
    private boolean configRequiresRefresh() {
        return System.currentTimeMillis() - lastUpdated > CONFIG_TTL;
    }

    /**
     * Checks if the time diff since the last attempt is enough to try again.
     *
     * @return if the config can be fetched from the remote server again or not.
     */
    private boolean configRetryIsSafe() {
        return System.currentTimeMillis() > (lastRefreshConfigAttempt + configRetrySafeWindow * 1000);
    }

    private float getNormalizedDeviceId() {
        String deviceId = metadataService.getDeviceId();
        String finalChars = deviceId.substring(deviceId.length() - 2);

        // Normalize the device ID to a value between 0-100.
        return (((float) Integer.valueOf(finalChars, 16) / 255.0f) * 100);
    }

    @Override
    public void close() {
        EmbraceLogger.logDebug("Shutting down EmbraceConfigService");
        worker.close();
    }
}
