package io.embrace.android.embracesdk;

import android.app.Application;
import android.content.Context;
import androidx.annotation.Nullable;
import android.util.Pair;

import com.fernandocejas.arrow.checks.Preconditions;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import io.embrace.android.embracesdk.network.EmbraceNetworkRequest;
import io.embrace.android.embracesdk.network.NetworkCaptureData;
import io.embrace.android.embracesdk.network.http.HttpMethod;

import static io.embrace.android.embracesdk.EmbraceEventService.STARTUP_EVENT_NAME;

/**
 * Entry point for the SDK. This class is part of the Embrace Public API.
 * <p>
 * Contains a singleton instance of itself, and is used for initializing the SDK.
 */
public final class Embrace {

    /**
     * Singleton instance of the Embrace SDK.
     */
    private static final Embrace embrace = new Embrace();
    private static final String ERROR_NOT_INITIALIZED = "Embrace SDK is not initialized yet, cannot log event";
    private static final String ERROR_USER_UPDATES_DISABLED = "User updates are disabled, ignoring user persona update.";
    /**
     * The application being instrumented by the SDK.
     */
    private volatile Application application;
    /**
     * The clock used with the SDK.
     */
    private volatile Clock clock;
    /**
     * The object caching service.
     */
    private volatile CacheService cacheService;
    /**
     * The crash handling service.
     */
    private volatile CrashService crashService;
    /**
     * The breadcrumbs service.
     */
    private volatile BreadcrumbService breadcrumbService;
    /**
     * The session handling service.
     */
    private volatile EmbraceSessionService sessionService;
    /**
     * The device metadata service.
     */
    private volatile MetadataService metadataService;
    /**
     * The device's performance information service.
     */
    private volatile PerformanceInfoService performanceInfoService;
    /**
     * The power service.
     */
    private volatile PowerService powerService;
    /**
     * The memory service;
     */
    private volatile MemoryService memoryService;
    /**
     * The Activity service.
     */
    private volatile EmbraceActivityService activityService;
    /**
     * The Networking service.
     */
    private volatile NetworkConnectivityService networkConnectivityService;
    /**
     * The network call logging service.
     */
    private volatile NetworkLoggingService networkLoggingService;
    /**
     * The CPU service.
     */
    private volatile CpuService cpuService;
    /**
     * The Build info data accessor.
     */
    private volatile BuildInfo buildInfo;
    /**
     * The local config info data accessor.
     */
    private volatile LocalConfig localConfig;
    /**
     * The ANR Monitoring service.
     */
    private volatile AnrService anrService;
    /**
     * The EmbraceRemoteLogger.
     */
    private volatile EmbraceRemoteLogger remoteLogger;
    /**
     * The Configuration service.
     */
    private volatile ConfigService configService;
    /**
     * The Embrace prefences service.
     */
    private volatile PreferencesService preferencesService;
    /**
     * The Embrace screenshot service.
     */
    private volatile ScreenshotService screenshotService;
    /**
     * The Embrace event service.
     */
    private volatile EventService eventService;
    /**
     * The User service.
     */
    private volatile UserService userService;
    /**
     * The Embrace signal quality service.
     */
    private volatile SignalQualityService signalQualityService;
    /**
     * The Embrace exception class service.
     */
    private volatile EmbraceExceptionService exceptionsService;
    /**
     * The Embrace memory cleaner class service.
     */
    private volatile MemoryCleanerService memoryCleanerService;
    /**
     * The Embrace orientation class service.
     */
    private volatile OrientationService orientationService;
    /**
     * Whether the Embrace SDK has been started yet.
     */
    private volatile AtomicBoolean started = new AtomicBoolean(false);

    /**
     * Gets the singleton instance of the Embrace SDK.
     *
     * @return the instance of the Embrace SDK
     */
    public static Embrace getInstance() {
        return embrace;
    }

    /**
     * Starts instrumentation of the Android application using the Embrace SDK. This should be
     * called during creation of the application, as early as possible.
     * <p>
     * See <a href="https://docs.embrace.io/docs/android-integration-guide">Embrace Docs</a> for
     * integration instructions. For compatibility with other networking SDKs such as Akamai,
     * the Embrace SDK must be initialized after any other SDK.
     *
     * @param context an instance of the application context
     */
    public void start(Context context) {
        start(context, true, AppFramework.NATIVE);
    }

    /**
     * Starts instrumentation of the Android application using the Embrace SDK. This should be
     * called during creation of the application, as early as possible.
     * <p>
     * See <a href="https://docs.embrace.io/docs/android-integration-guide">Embrace Docs</a> for
     * integration instructions. For compatibility with other networking SDKs such as Akamai,
     * the Embrace SDK must be initialized after any other SDK.
     *
     * @param context                  an instance of the application context
     * @param enableIntegrationTesting if true, debug sessions (those which are not part of a
     *                                 release APK) will go to the live integration testing tab
     *                                 of the dashboard. If false, they will appear in 'recent
     *                                 sessions'.
     */
    public void start(Context context, boolean enableIntegrationTesting) {
        start(context, enableIntegrationTesting, AppFramework.NATIVE);
    }

    /**
     * Starts instrumentation of the Android application using the Embrace SDK. This should be
     * called during creation of the application, as early as possible.
     * <p>
     * See <a href="https://docs.embrace.io/docs/android-integration-guide">Embrace Docs</a> for
     * integration instructions. For compatibility with other networking SDKs such as Akamai,
     * the Embrace SDK must be initialized after any other SDK.
     *
     * @param context                  an instance of the application context
     * @param enableIntegrationTesting if true, debug sessions (those which are not part of a
     *                                 release APK) will go to the live integration testing tab
     *                                 of the dashboard. If false, they will appear in 'recent
     *                                 sessions'.
     */
    public void start(Context context, boolean enableIntegrationTesting, AppFramework appFramework) {
        if (context instanceof Application) {
            start((Application) context, enableIntegrationTesting, appFramework);
        } else {
            throw new IllegalArgumentException("To initialize EmbraceSDK, please use the Application's context.");
        }
    }

    /**
     * Starts instrumentation of the Android application using the Embrace SDK. This should be
     * called during creation of the application, as early as possible.
     * <p>
     * See <a href="https://docs.embrace.io/docs/android-integration-guide">Embrace Docs</a> for
     * integration instructions. For compatibility with other networking SDKs such as Akamai,
     * the Embrace SDK must be initialized after any other SDK.
     *
     * @param application an instance of the application to instrument
     */
    public void start(Application application) {
        start(application, true, AppFramework.NATIVE);
    }

    /**
     * Starts instrumentation of the Android application using the Embrace SDK. This should be
     * called during creation of the application, as early as possible.
     * <p>
     * See <a href="https://docs.embrace.io/docs/android-integration-guide">Embrace Docs</a> for
     * integration instructions. For compatibility with other networking SDKs such as Akamai,
     * the Embrace SDK must be initialized after any other SDK.
     *
     * @param application              an instance of the application to instrument
     * @param enableIntegrationTesting if true, debug sessions (those which are not part of a
     *                                 release APK) will go to the live integration testing tab
     *                                 of the dashboard. If false, they will appear in 'recent
     *                                 sessions'.
     */
    public void start(Application application, boolean enableIntegrationTesting) {
        start(application, enableIntegrationTesting, AppFramework.NATIVE);
    }

    /**
     * Starts instrumentation of the Android application using the Embrace SDK. This should be
     * called during creation of the application, as early as possible.
     * <p>
     * See <a href="https://docs.embrace.io/docs/android-integration-guide">Embrace Docs</a> for
     * integration instructions. For compatibility with other networking SDKs such as Akamai,
     * the Embrace SDK must be initialized after any other SDK.
     *
     * @param application              an instance of the application to instrument
     * @param enableIntegrationTesting if true, debug sessions (those which are not part of a
     *                                 release APK) will go to the live integration testing tab
     *                                 of the dashboard. If false, they will appear in 'recent
     *                                 sessions'.
     * @param appFramework             the type of application framework used by the app
     */
    public void start(Application application, boolean enableIntegrationTesting, AppFramework appFramework) {
        if (this.application != null) {
            // We don't hard fail if the SDK has been already initialized.
            EmbraceLogger.logWarning("Embrace SDK has already been initialized");
            return;
        }
        this.application = Preconditions.checkNotNull(application, "application must not be null");
        Context context = this.application.getApplicationContext();

        // Bring up all services and inject dependencies
        try {
            /* ------------------------------------------------------------------------------------
             *  Device instrumentation (power, memory, CPU, network, preferences, CPU)
             *  */
            this.clock = new SystemClock();
            this.memoryCleanerService = new EmbraceMemoryCleanerService();
            this.powerService = new EmbracePowerService(context, memoryCleanerService);
            this.memoryService = EmbraceMemoryService.ofContext(context, memoryCleanerService);
            this.orientationService = new EmbraceOrientationService(memoryCleanerService);
            this.activityService = new EmbraceActivityService(application, memoryService, orientationService);
            this.exceptionsService = new EmbraceExceptionService(activityService);
            this.preferencesService = new EmbracePreferencesService(context, activityService);
            this.networkConnectivityService = new EmbraceNetworkConnectivityService(context, memoryCleanerService);
            this.cpuService = new EmbraceCpuService();
            this.buildInfo = BuildInfo.fromResources(context);
            this.localConfig = LocalConfig.fromResources(context);
            this.metadataService = EmbraceMetadataService.ofContext(
                    context,
                    buildInfo,
                    localConfig,
                    appFramework,
                    preferencesService,
                    activityService);
            ConnectionClassService connectionClassService = new EmbraceConnectionClassService(memoryCleanerService);

            /* ------------------------------------------------------------------------------------
             *  API services
             *  */
            this.cacheService = new EmbraceCacheService(context);
            ApiClient apiClient = new ApiClient(
                    localConfig,
                    metadataService,
                    cacheService,
                    enableIntegrationTesting);
            this.configService = new EmbraceConfigService(
                    apiClient,
                    activityService,
                    cacheService,
                    metadataService);
            this.anrService = new EmbraceAnrService(
                    clock,
                    memoryCleanerService,
                    configService,
                    cacheService);
            this.exceptionsService.setConfigService(this.configService);
            this.signalQualityService = new EmbraceSignalQualityService(
                    context,
                    configService,
                    activityService,
                    connectionClassService,
                    memoryCleanerService);
            this.breadcrumbService = new EmbraceBreadcrumbService(
                    clock,
                    configService,
                    localConfig,
                    activityService,
                    memoryCleanerService
            );
            if (configService.isSdkDisabled()) {
                stop();
                return;
            }
            this.userService = new EmbraceUserService(
                    preferencesService);
            this.screenshotService = new EmbraceScreenshotService(
                    activityService,
                    configService,
                    apiClient);
            this.networkLoggingService = new EmbraceNetworkLoggingService(
                    clock,
                    connectionClassService,
                    configService,
                    localConfig,
                    memoryCleanerService);
            this.performanceInfoService = new EmbracePerformanceInfoService(
                    anrService,
                    networkConnectivityService,
                    networkLoggingService,
                    powerService,
                    cpuService,
                    memoryService,
                    signalQualityService,
                    connectionClassService,
                    metadataService);
            this.eventService = new EmbraceEventService(
                    apiClient,
                    configService,
                    localConfig,
                    metadataService,
                    performanceInfoService,
                    userService,
                    screenshotService,
                    activityService,
                    memoryCleanerService);
            this.remoteLogger = new EmbraceRemoteLogger(
                    metadataService,
                    screenshotService,
                    apiClient,
                    userService,
                    configService,
                    memoryCleanerService);
            this.sessionService = new EmbraceSessionService(
                    preferencesService,
                    performanceInfoService,
                    metadataService,
                    breadcrumbService,
                    powerService,
                    activityService,
                    apiClient,
                    eventService,
                    remoteLogger,
                    userService,
                    configService,
                    cacheService,
                    exceptionsService,
                    localConfig,
                    memoryCleanerService,
                    orientationService);
            this.crashService = new EmbraceCrashService(
                    localConfig,
                    sessionService,
                    metadataService,
                    apiClient,
                    userService,
                    eventService);
        } catch (Exception ex) {
            EmbraceLogger.logError("Exception occurred while initializing the Embrace SDK. Instrumentation may be disabled.", ex, true);
            return;
        }

        EmbraceLogger.logInfo(String.format("Embrace SDK started. API key: %s Version: %s", this.localConfig.getAppId(), BuildConfig.VERSION_NAME));

        // Intercept Android network calls
        StreamHandlerFactoryInstaller.registerFactory(localConfig.getConfigurations().getNetworking().getCaptureRequestContentLength());
        started.set(true);
    }

    /**
     * Whether or not the SDK has been started.
     *
     * @return true if the SDK is started, false otherwise
     */
    public boolean isStarted() {
        return started.get();
    }

    /**
     * Shuts down the Embrace SDK.
     */
    void stop() {
        if (started.compareAndSet(true, false)) {
            EmbraceLogger.logInfo("Shutting down Embrace SDK.");
            try {
                // Keep the config service active to give the application the opportunity to
                // update the disabled flag, in case the SDK has been re-enabled.

                if (this.anrService != null) {
                    this.anrService.close();
                }

                if (this.powerService != null) {
                    this.powerService.close();
                }

                if (this.memoryService != null) {
                    this.memoryService.close();
                }

                if (this.cpuService != null) {
                    this.cpuService.close();
                }

                if (this.anrService != null) {
                    this.anrService.close();
                }

                if (this.activityService != null) {
                    this.activityService.close();
                }

                if (this.sessionService != null) {
                    this.sessionService.close();
                }

                if (this.eventService != null) {
                    this.eventService.close();
                }

                if (this.networkConnectivityService != null) {
                    this.networkConnectivityService.close();
                }

                if (this.signalQualityService != null) {
                    this.signalQualityService.close();
                }

                this.memoryCleanerService = null;
                this.powerService = null;
                this.memoryService = null;
                this.breadcrumbService = null;
                this.activityService = null;
                this.preferencesService = null;
                this.networkConnectivityService = null;
                this.cpuService = null;
                this.anrService = null;
                this.buildInfo = null;
                this.localConfig = null;
                this.metadataService = null;
                this.performanceInfoService = null;
                this.cacheService = null;
                this.userService = null;
                this.screenshotService = null;
                this.eventService = null;
                this.remoteLogger = null;
                this.sessionService = null;
                this.crashService = null;
                this.signalQualityService = null;
                this.networkLoggingService = null;

                this.application = null;
            } catch (Exception ex) {
                EmbraceLogger.logDebug("Error while shutting down Embrace SDK", ex);
            }
        }
    }

    /**
     * This method sets a logging level, but this logging level is never used.
     *
     * @param severity the severity
     * @deprecated as the log level is never used. Use {@link EmbraceLogger}.
     */
    @Deprecated
    public void setLogLevel(EmbraceLogger.Severity severity) {
        // This method does not do anything and is purely retained for backwards-compatibility
    }

    /**
     * Sets the user ID. This would typically be some form of unique identifier such as a UUID or
     * database key for the user.
     *
     * @param userId the unique identifier for the user
     */
    public void setUserIdentifier(String userId) {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                EmbraceLogger.logWarning("User updates are disabled, ignoring identifier update.");
                return;
            }
            userService.setUserIdentifier(userId);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot set user identifier");
        }
    }

    /**
     * Clears the currently set user ID. For example, if the user logs out.
     */
    public void clearUserIdentifier() {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                EmbraceLogger.logWarning("User updates are disabled, ignoring identifier update.");
                return;
            }
            userService.clearUserIdentifier();
        }
    }

    /**
     * Sets the current user's email address.
     *
     * @param email the email address of the current user
     */
    public void setUserEmail(String email) {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                EmbraceLogger.logWarning("User updates are disabled, ignoring email update.");
                return;
            }
            userService.setUserEmail(email);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot clear user identifier");
        }
    }

    /**
     * Clears the currently set user's email address.
     */
    public void clearUserEmail() {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                EmbraceLogger.logWarning("User updates are disabled, ignoring email update.");
                return;
            }
            userService.clearUserEmail();
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot clear user email");
        }
    }

    /**
     * Sets this user as a paying user. This adds a persona to the user's identity.
     */
    public void setUserAsPayer() {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                EmbraceLogger.logWarning("User updates are disabled, ignoring payer user update.");
                return;
            }
            userService.setUserAsPayer();
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot set user as payer");
        }
    }

    /**
     * Clears this user as a paying user. This would typically be called if a user is no longer
     * paying for the service and has reverted back to a basic user.
     */
    public void clearUserAsPayer() {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                EmbraceLogger.logWarning("User updates are disabled, ignoring payer user update.");
                return;
            }
            userService.clearUserAsPayer();
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot clear user as payer");
        }
    }

    /**
     * Sets a custom user persona. A persona is a trait associated with a given user.
     *
     * @param persona the persona to set
     */
    public void setUserPersona(String persona) {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                EmbraceLogger.logWarning(ERROR_USER_UPDATES_DISABLED);
                return;
            }
            userService.setUserPersona(persona);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot set user persona");
        }
    }

    /**
     * Clears the custom user persona, if it is set.
     *
     * @param persona the persona to clear
     */
    public void clearUserPersona(String persona) {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                EmbraceLogger.logWarning(ERROR_USER_UPDATES_DISABLED);
                return;
            }
            userService.clearUserPersona(persona);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot clear user persona");
        }
    }

    /**
     * Clears all custom user personas from the user.
     */
    public void clearAllUserPersonas() {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                EmbraceLogger.logWarning(ERROR_USER_UPDATES_DISABLED);
                return;
            }
            userService.clearAllUserPersonas();
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot clear user personas");
        }
    }

    public boolean addSessionProperty(String key, String value, boolean permanent) {
        if (isStarted()) {
            return sessionService.addProperty(key, value, permanent);
        }
        EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot add session property");
        return false;
    }

    public boolean removeSessionProperty(String key) {
        if (isStarted()) {
            return sessionService.removeProperty(key);
        }

        EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot remove session property");
        return false;
    }

    @Nullable
    public Map<String, String> getSessionProperties() {
        if (isStarted()) {
            return sessionService.getProperties();
        }

        EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot gets session properties");
        return null;
    }

    /**
     * Sets the username of the currently logged in user.
     *
     * @param username the username to set
     */
    public void setUsername(String username) {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                EmbraceLogger.logWarning("User updates are disabled, ignoring username update.");
                return;
            }
            userService.setUsername(username);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot set username");
        }
    }

    /**
     * Clears the username of the currently logged in user, for example if the user has logged out.
     */
    public void clearUsername() {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                EmbraceLogger.logWarning("User updates are disabled, ignoring username update.");
                return;
            }
            userService.clearUsername();
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot clear username");
        }
    }

    /**
     * Starts an event or 'moment'. Events are used for encapsulating particular activities within
     * the app, such as a user adding an item to their shopping cart.
     * <p>
     * The length of time an event takes to execute is recorded, and a screenshot can be taken if
     * an event is 'late'.
     *
     * @param name a name identifying the event
     */
    public void startEvent(String name) {
        if (isStarted()) {
            eventService.startEvent(name);
        } else {
            EmbraceLogger.logWarning(ERROR_NOT_INITIALIZED);
        }
    }

    /**
     * Starts an event or 'moment'. Events are used for encapsulating particular activities within
     * the app, such as a user adding an item to their shopping cart.
     * <p>
     * The length of time an event takes to execute is recorded, and a screenshot can be taken if
     * an event is 'late'.
     *
     * @param name       a name identifying the event
     * @param identifier an identifier distinguishing between multiple events with the same name
     */
    public void startEvent(String name, String identifier) {
        if (isStarted()) {
            eventService.startEvent(name, identifier);
        }
    }

    /**
     * Starts an event or 'moment'. Events are used for encapsulating particular activities within
     * the app, such as a user adding an item to their shopping cart.
     * <p>
     * The length of time an event takes to execute is recorded, and a screenshot can be taken if
     * an event is 'late'.
     *
     * @param name            a name identifying the event
     * @param identifier      an identifier distinguishing between multiple events with the same name
     * @param allowScreenshot true if a screenshot should be taken for a late event, false otherwise
     */
    public void startEvent(String name, String identifier, boolean allowScreenshot) {
        if (isStarted()) {
            eventService.startEvent(name, identifier, allowScreenshot);
        } else {
            EmbraceLogger.logWarning(ERROR_NOT_INITIALIZED);
        }
    }

    /**
     * Starts an event or 'moment'. Events are used for encapsulating particular activities within
     * the app, such as a user adding an item to their shopping cart.
     * <p>
     * The length of time an event takes to execute is recorded, and a screenshot can be taken if
     * an event is 'late'.
     *
     * @param name       a name identifying the event
     * @param identifier an identifier distinguishing between multiple events with the same name
     * @param properties custom key-value pairs to provide with the event
     */
    public void startEvent(String name, String identifier, Map<String, Object> properties) {
        if (isStarted()) {
            eventService.startEvent(name, identifier, normalizeProperties(properties));
        } else {
            EmbraceLogger.logWarning(ERROR_NOT_INITIALIZED);
        }
    }

    /**
     * Starts an event or 'moment'. Events are used for encapsulating particular activities within
     * the app, such as a user adding an item to their shopping cart.
     * <p>
     * The length of time an event takes to execute is recorded, and a screenshot can be taken if
     * an event is 'late'.
     *
     * @param name            a name identifying the event
     * @param identifier      an identifier distinguishing between multiple events with the same name
     * @param allowScreenshot true if a screenshot should be taken for a late event, false otherwise
     * @param properties      custom key-value pairs to provide with the event
     */
    public void startEvent(String name, String identifier, boolean allowScreenshot, Map<String, Object> properties) {
        if (isStarted()) {
            eventService.startEvent(name, identifier, allowScreenshot, normalizeProperties(properties));
        } else {
            EmbraceLogger.logWarning(ERROR_NOT_INITIALIZED);
        }
    }

    /**
     * Signals the end of an event with the specified name.
     * <p>
     * The duration of the event is computed, and a screenshot taken (if enabled) if the event was
     * late.
     *
     * @param name the name of the event to end
     */
    public void endEvent(String name) {
        if (isStarted()) {
            eventService.endEvent(name);
        } else {
            EmbraceLogger.logWarning(ERROR_NOT_INITIALIZED);
        }
    }

    /**
     * Signals the end of an event with the specified name.
     * <p>
     * The duration of the event is computed, and a screenshot taken (if enabled) if the event was
     * late.
     *
     * @param name       the name of the event to end
     * @param identifier the identifier of the event to end, distinguishing between events with the same name
     */
    public void endEvent(String name, String identifier) {
        if (isStarted()) {
            eventService.endEvent(name, identifier);
        } else {
            EmbraceLogger.logWarning(ERROR_NOT_INITIALIZED);
        }
    }

    /**
     * Signals the end of an event with the specified name.
     * <p>
     * The duration of the event is computed, and a screenshot taken (if enabled) if the event was
     * late.
     *
     * @param name       the name of the event to end
     * @param properties custom key-value pairs to provide with the event
     */
    public void endEvent(String name, Map<String, Object> properties) {
        if (isStarted()) {
            eventService.endEvent(name, normalizeProperties(properties));
        } else {
            EmbraceLogger.logWarning(ERROR_NOT_INITIALIZED);
        }
    }

    /**
     * Signals the end of an event with the specified name.
     * <p>
     * The duration of the event is computed, and a screenshot taken (if enabled) if the event was
     * late.
     *
     * @param name       the name of the event to end
     * @param identifier the identifier of the event to end, distinguishing between events with the same name
     * @param properties custom key-value pairs to provide with the event
     */
    public void endEvent(String name, String identifier, Map<String, Object> properties) {
        if (isStarted()) {
            eventService.endEvent(name, identifier, normalizeProperties(properties));
        } else {
            EmbraceLogger.logWarning(ERROR_NOT_INITIALIZED);
        }
    }

    /**
     * Signals that the app has completed startup.
     */
    public void endAppStartup() {
        endEvent(STARTUP_EVENT_NAME);
    }

    /**
     * Signals that the app has completed startup.
     *
     * @param properties properties to include as part of the startup moment
     */
    public void endAppStartup(Map<String, Object> properties) {
        endEvent(STARTUP_EVENT_NAME, null, properties);
    }

    /**
     * Retrieve the HTTP request header to extract trace ID from.
     *
     * @return the Trace ID header.
     */
    public String getTraceIdHeader() {
        return isStarted() ? this.localConfig.getConfigurations().getNetworking().getTraceIdHeader() : LocalConfig.SdkConfigs.Networking.CONFIG_TRACE_ID_HEADER_DEFAULT_VALUE;
    }

    /**
     * Manually log a network request. In most cases the Embrace SDK.
     *
     * @param request An EmbraceNetworkRequest with at least the following set: url, method, start time,
     *                end time, and either status code or error
     */
    public void logNetworkRequest(EmbraceNetworkRequest request) {

        if (isStarted()) {

            if (request == null) {
                return;
            }

            if (!request.canSend()) {
                return;
            }

            if (request.getError() != null) {
                networkLoggingService.logNetworkError(
                        request.getUrl(),
                        request.getHttpMethod(),
                        request.getStartTime(),
                        request.getEndTime(),
                        request.getError().getClass().getCanonicalName(),
                        request.getError().getLocalizedMessage(),
                        request.getTraceId());
            } else {
                networkLoggingService.logNetworkCall(
                        request.getUrl(),
                        request.getHttpMethod(),
                        request.getResponseCode(),
                        request.getStartTime(),
                        request.getEndTime(),
                        request.getBytesIn(),
                        request.getBytesOut(),
                        request.getTraceId());
            }
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot log network request");
        }
    }

    /**
     * Logs the fact that a network call occurred. These are recorded and sent to Embrace as part
     * of a particular session.
     *
     * @param url           the URL of the network call
     * @param httpMethod    the HTTP method of the network call
     * @param statusCode    the status code returned by the server
     * @param startTime     the time that the network call started
     * @param endTime       the time that the network call was completed
     * @param bytesSent     the number of bytes sent as part of the network call
     * @param bytesReceived the number of bytes returned by the server in response to the network call
     */
    public void logNetworkCall(
            String url,
            HttpMethod httpMethod,
            int statusCode,
            long startTime,
            long endTime,
            long bytesSent,
            long bytesReceived) {
        logNetworkCall(url, httpMethod, statusCode, startTime, endTime, bytesSent, bytesReceived, null);
    }

    /**
     * Logs the fact that a network call occurred. These are recorded and sent to Embrace as part
     * of a particular session.
     *
     * @param url           the URL of the network call
     * @param httpMethod    the HTTP method of the network call
     * @param statusCode    the status code returned by the server
     * @param startTime     the time that the network call started
     * @param endTime       the time that the network call was completed
     * @param bytesSent     the number of bytes sent as part of the network call
     * @param bytesReceived the number of bytes returned by the server in response to the network call
     * @param traceId       the optional trace id that can be used to trace a particular request
     */
    public void logNetworkCall(
            String url,
            HttpMethod httpMethod,
            int statusCode,
            long startTime,
            long endTime,
            long bytesSent,
            long bytesReceived,
            String traceId) {

        logNetworkCall(url, httpMethod, statusCode, startTime, endTime, bytesSent, bytesReceived, traceId, null);
    }

    /**
     * Logs the fact that a network call occurred. These are recorded and sent to Embrace as part
     * of a particular session.
     *
     * @param url                the URL of the network call
     * @param httpMethod         the HTTP method of the network call
     * @param statusCode         the status code returned by the server
     * @param startTime          the time that the network call started
     * @param endTime            the time that the network call was completed
     * @param bytesSent          the number of bytes sent as part of the network call
     * @param bytesReceived      the number of bytes returned by the server in response to the network call
     * @param traceId            the optional trace id that can be used to trace a particular request
     * @param networkCaptureData the additional data captured if network body capture is enabled for the URL
     */
    public void logNetworkCall(
            String url,
            HttpMethod httpMethod,
            int statusCode,
            long startTime,
            long endTime,
            long bytesSent,
            long bytesReceived,
            String traceId,
            NetworkCaptureData networkCaptureData) {

        if (isStarted()) {
            if (configService.isUrlDisabled(url) || localConfig.isUrlDisabled(url)) {
                EmbraceLogger.logWarning("Recording of network calls disabled for url: " + url);
                return;
            }
            networkLoggingService.logNetworkCall(
                    url,
                    httpMethod.name(),
                    statusCode,
                    startTime,
                    endTime,
                    bytesSent,
                    bytesReceived,
                    traceId,
                    networkCaptureData);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot log network call");
        }
    }

    /**
     * Logs the fact that an exception was thrown when attempting to make a network call.
     * <p>
     * These are client-side exceptions and not server-side exceptions, such as a DNS error or
     * failing to connect to the remote server.
     *
     * @param url          the URL of the network call
     * @param httpMethod   the HTTP method of the network call
     * @param startTime    the time that the network call started
     * @param endTime      the time that the network call was completed
     * @param errorType    the type of the exception
     * @param errorMessage the message returned by the exception
     */
    public void logNetworkClientError(
            String url,
            HttpMethod httpMethod,
            long startTime,
            long endTime,
            String errorType,
            String errorMessage) {
        logNetworkClientError(url, httpMethod, startTime, endTime, errorType, errorMessage, null);
    }

    /**
     * Logs the fact that an exception was thrown when attempting to make a network call.
     * <p>
     * These are client-side exceptions and not server-side exceptions, such as a DNS error or
     * failing to connect to the remote server.
     *
     * @param url          the URL of the network call
     * @param httpMethod   the HTTP method of the network call
     * @param startTime    the time that the network call started
     * @param endTime      the time that the network call was completed
     * @param errorType    the type of the exception
     * @param errorMessage the message returned by the exception
     * @param traceId      the optional trace id that can be used to trace a particular request
     */
    public void logNetworkClientError(
            String url,
            HttpMethod httpMethod,
            long startTime,
            long endTime,
            String errorType,
            String errorMessage,
            String traceId) {
        logNetworkClientError(url, httpMethod, startTime, endTime, errorType, errorMessage, traceId, null);
    }

    /**
     * Logs the fact that an exception was thrown when attempting to make a network call.
     * <p>
     * These are client-side exceptions and not server-side exceptions, such as a DNS error or
     * failing to connect to the remote server.
     *
     * @param url                the URL of the network call
     * @param httpMethod         the HTTP method of the network call
     * @param startTime          the time that the network call started
     * @param endTime            the time that the network call was completed
     * @param errorType          the type of the exception
     * @param errorMessage       the message returned by the exception
     * @param traceId            the optional trace id that can be used to trace a particular request
     * @param networkCaptureData the additional data captured if network body capture is enabled for the URL
     */
    public void logNetworkClientError(
            String url,
            HttpMethod httpMethod,
            long startTime,
            long endTime,
            String errorType,
            String errorMessage,
            String traceId,
            NetworkCaptureData networkCaptureData) {

        if (isStarted()) {
            if (configService.isUrlDisabled(url) || localConfig.isUrlDisabled(url)) {
                EmbraceLogger.logWarning("Recording of network calls disabled for url: " + url);
                return;
            }
            networkLoggingService.logNetworkError(
                    url,
                    httpMethod.name(),
                    startTime,
                    endTime,
                    errorType,
                    errorMessage,
                    traceId,
                    networkCaptureData);
        } else {
            EmbraceLogger.logDebug("Embrace SDK is not initialized yet, cannot log network error");
        }
    }

    /**
     * Remotely logs a message at INFO level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message the message to remotely log
     */
    public void logInfo(String message) {
        logInfo(message, null);
    }

    /**
     * Remotely logs a message at INFO level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message    the message to remotely log
     * @param properties custom key-value pairs to include with the log message
     */
    public void logInfo(String message, Map<String, Object> properties) {
        logMessage(EmbraceEvent.Type.INFO_LOG, message, properties, false, null, null, false);
    }


    /**
     * Remotely logs a message at WARN level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message the message to remotely log
     */
    public void logWarning(String message) {
        logWarning(message, null, false, null);
    }

    /**
     * Remotely logs a message at WARN level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message    the message to remotely log
     * @param properties custom key-value pairs to include with the log message
     */
    public void logWarning(String message, Map<String, Object> properties) {
        logWarning(message, properties, false, null);
    }

    /**
     * Remotely logs a message at WARN level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message         the message to remotely log
     * @param properties      custom key-value pairs to include with the log message
     * @param allowScreenshot true if a screenshot should be taken for this message, false otherwise
     */
    public void logWarning(String message, Map<String, Object> properties, Boolean allowScreenshot) {
        logWarning(message, properties, allowScreenshot, null);
    }

    /**
     * Remotely logs a message at WARN level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message              the message to remotely log
     * @param properties           custom key-value pairs to include with the log message
     * @param allowScreenshot      true if a screenshot should be taken for this message, false otherwise
     * @param javascriptStackTrace javascript stack trace coming from the the RN side
     */
    public void logWarning(String message, Map<String, Object> properties, Boolean allowScreenshot, String javascriptStackTrace) {
        logMessage(EmbraceEvent.Type.WARNING_LOG, message, properties, allowScreenshot, null, javascriptStackTrace, false);
    }


    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     *
     * @param message the message to remotely log
     */
    public void logError(String message) {
        logError(message, null, true, null);
    }

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     *
     * @param message    the message to remotely log
     * @param properties custom key-value pairs to include with the log message
     */
    public void logError(String message, Map<String, Object> properties) {
        logError(message, properties, true, null);
    }

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     *
     * @param message         the message to remotely log
     * @param properties      custom key-value pairs to include with the log message
     * @param allowScreenshot true if a screenshot should be taken for this message, false otherwise
     */
    public void logError(String message, Map<String, Object> properties, boolean allowScreenshot) {
        logError(message, properties, allowScreenshot, null);
    }

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     *
     * @param message              the message to remotely log
     * @param properties           custom key-value pairs to include with the log message
     * @param allowScreenshot      true if a screenshot should be taken for this message, false otherwise
     * @param javascriptStackTrace javascript stack trace coming from the the RN side
     */
    public void logError(String message, Map<String, Object> properties, boolean allowScreenshot, String javascriptStackTrace) {
        logMessage(EmbraceEvent.Type.ERROR_LOG, message, properties, allowScreenshot, null, javascriptStackTrace, false);
    }

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     *
     * @param message              the message to remotely log
     * @param properties           custom key-value pairs to include with the log message
     * @param allowScreenshot      true if a screenshot should be taken for this message, false otherwise
     * @param javascriptStackTrace javascript stack trace coming from the the RN side
     * @param isException          mark the log as an exception
     */

    public void logError(String message, Map<String, Object> properties, boolean allowScreenshot, String javascriptStackTrace, Boolean isException) {
        logMessage(EmbraceEvent.Type.ERROR_LOG, message, properties, allowScreenshot, null, javascriptStackTrace, isException);
    }

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     * <p>
     * The stacktrace from the throwable will be used instead of the stack trace from where this method is called.
     *
     * @param e the throwable to remotely log
     */
    public void logError(Throwable e) {
        logError(e, null, false);
    }

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     * <p>
     * The stacktrace from the throwable will be used instead of the stack trace from where this method is called.
     *
     * @param e          the throwable to remotely log
     * @param properties custom key-value pairs to include with the log message
     */
    public void logError(Throwable e, Map<String, Object> properties) {
        logError(e, properties, false);
    }

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     * <p>
     * The stacktrace from the throwable will be used instead of the stack trace from where this method is called.
     *
     * @param e               the throwable to remotely log
     * @param properties      custom key-value pairs to include with the log message
     * @param allowScreenshot true if a screenshot should be taken for this message, false otherwise
     */
    public void logError(Throwable e, Map<String, Object> properties, boolean allowScreenshot) {
        logError(e, e.getLocalizedMessage(), properties, allowScreenshot);
    }

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     * <p>
     * The stacktrace from the throwable will be used instead of the stack trace from where this method is called.
     *
     * @param e               the throwable to remotely log
     * @param message         message to log (overrides getLocalizedMessage() from e)
     * @param properties      custom key-value pairs to include with the log message
     * @param allowScreenshot true if a screenshot should be taken for this message, false otherwise
     */
    public void logError(Throwable e, String message, Map<String, Object> properties, boolean allowScreenshot) {
        logMessage(EmbraceEvent.Type.ERROR_LOG, message, properties, allowScreenshot, e.getStackTrace(), null, true);
    }

    void logMessage(
            EmbraceEvent.Type type,
            String message,
            Map<String, Object> properties,
            boolean allowScreenshot,
            StackTraceElement[] stackTraceElements,
            String javascriptStackTrace,
            Boolean isException) {
        if (isStarted()) {
            try {
                this.remoteLogger.log(message, type, allowScreenshot, isException, normalizeProperties(properties), stackTraceElements, javascriptStackTrace);
            } catch (Exception ex) {
                EmbraceLogger.logDebug("Failed to log message using Embrace SDK.", ex);
            }
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot log message.");
        }
    }

    /**
     * Logs a breadcrumb.
     * <p>
     * Breadcrumbs track a user's journey through the application and will be shown on the timeline.
     *
     * @param message the name of the breadcrumb to log
     */
    public void logBreadcrumb(String message) {
        if (isStarted()) {
            this.breadcrumbService.logCustom(message, System.currentTimeMillis());
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot log breadcrumb.");
        }
    }

    /**
     * Logs a javascript unhandled exception.
     *
     * @param name       name of the exception.
     * @param message    exception message.
     * @param type       error type.
     * @param stacktrace exception stacktrace.
     */
    public void logUnhandledJsException(String name, String message, String type, String stacktrace) {
        if (isStarted()) {
            JsException exception = new JsException(name, message, type, stacktrace);
            this.crashService.logUnhandledJsException(exception);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot log JS exception.");
        }
    }

    /**
     * Sets the react native version number.
     *
     * @param version react native version number.
     */
    public void setReactNativeVersionNumber(String version) {
        if (isStarted()) {
            if (version == null) {
                EmbraceLogger.logError("ReactNative version must not be null");
                return;
            }

            if (version.isEmpty()) {
                EmbraceLogger.logError("ReactNative version must have non-zero length");
                return;
            }

            this.preferencesService.setReactNativeVersionNumber(version);
        } else {
            EmbraceLogger.logError("Embrace SDK is not initialized yet, cannot set React Native version number.");
        }
    }

    /**
     * Sets javascript patch number.
     *
     * @param number javascript patch number.
     */
    public void setJavaScriptPatchNumber(String number) {
        if (isStarted()) {
            if (number == null) {
                EmbraceLogger.logError("JavaScript patch number must not be null");
                return;
            }

            if (number.isEmpty()) {
                EmbraceLogger.logError("JavaScript patch number must have non-zero length");
                return;
            }

            this.preferencesService.setJavaScriptPatchNumber(number);
        } else {
            EmbraceLogger.logError("Embrace SDK is not initialized yet, cannot set JavaScript patch number.");
        }
    }


    /**
     * Sets the path of the javascript bundle.
     *
     * @param url path of the javascript bundle.
     */
    public void setJavaScriptBundleURL(String url) {
        if (isStarted()) {
            if (url == null) {
                EmbraceLogger.logError("JavaScript bundle URL must not be null");
                return;
            }

            if (url.isEmpty()) {
                EmbraceLogger.logError("JavaScript bundle URL must have non-zero length");
                return;
            }

            if (this.preferencesService.getJavaScriptBundleURL().isPresent()) {

                String currentUrl = this.preferencesService.getJavaScriptBundleURL().get();

                if (url.equals(currentUrl)) {
                    return;
                }
            }

            this.preferencesService.setJavaScriptBundleURL(url);
        } else {
            EmbraceLogger.logError("Embrace SDK is not initialized yet, cannot set JavaScript bundle URL.");
        }
    }

    /**
     * Registers a {@link ConnectionQualityListener}, notifying the listener each time that there is
     * a change in the connection quality.
     *
     * @param listener the listener to register
     */
    public void addConnectionQualityListener(ConnectionQualityListener listener) {
        if (isStarted()) {
            try {
                signalQualityService.addListener(listener);
            } catch (Exception ex) {
                EmbraceLogger.logDebug("Failed to add connection quality listener", ex);
            }
        } else {
            EmbraceLogger.logDebug("Embrace SDK is not initialized yet, cannot add listener.");

        }
    }

    /**
     * Removes a registered {@link ConnectionQualityListener}, suspending connection quality
     * notifications.
     *
     * @param listener the listener to remove
     */
    public void removeConnectionQualityListener(ConnectionQualityListener listener) {
        if (isStarted()) {
            try {
                signalQualityService.removeListener(listener);
            } catch (Exception ex) {
                EmbraceLogger.logDebug("Failed to remove connection quality listener", ex);
            }
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot remove listener.");
        }
    }

    /**
     * Ends the current session and starts a new one.
     */
    public synchronized void endSession() {
        endSession(false);
    }

    /**
     * Ends the current session and starts a new one.
     * <p>
     * Cleans all the user info on the device.
     */
    public synchronized void endSession(boolean clearUserInfo) {
        if (isStarted()) {
            if (localConfig.getConfigurations().getSessionConfig().getMaxSessionSeconds().isPresent()) {
                EmbraceLogger.logWarning("Can't close the session, automatic session close enabled.");
                return;
            }

            if (localConfig.getConfigurations().getSessionConfig().getAsyncEnd() ||
                    configService.getConfig().endSessionInBackgroundThread()) {
                EmbraceLogger.logWarning("Can't close the session, session ending in background thread enabled.");
                return;
            }

            if (clearUserInfo) {
                userService.clearAllUserInfo();
            }

            sessionService.triggerStatelessSessionEnd(Session.SessionLifeEventType.MANUAL);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot end session.");
        }
    }

    /**
     * Get the user identifier assigned to the device by Embrace
     *
     * @return the device identifier created by Embrace
     */
    public String getDeviceId() {
        return preferencesService.getDeviceIdentifier().orNull();
    }

    /**
     * Causes a crash with an exception. Use this for test purposes only
     */
    public void throwException() {
        throw new RuntimeException("EmbraceException", new Throwable("Embrace test exception"));
    }

    public boolean shouldCaptureNetworkBody(String url, String method) {
        if (isStarted()) {

            if (!hasNetworkCaptureRules()) {
                return false;
            }

            if (url == null) {
                EmbraceLogger.logDebug("Cannot check for rules with a null url.");
                return false;
            }

            if (method == null) {
                EmbraceLogger.logDebug("Cannot check for rules with a null http method.");
                return false;
            }

            return networkLoggingService.shouldCaptureNetworkBody(url, method);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot check for capture rules.");
            return false;
        }
    }

    private boolean hasNetworkCaptureRules() {
        if (isStarted()) {
            return !this.configService.getConfig().getCaptureRules().isEmpty();
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot check for capture rules.");
            return false;
        }
    }

    /**
     * Log the start of a fragment.
     * <p>
     * A matching call to endFragment must be made.
     *
     * @param name the name of the fragment to log
     */
    public boolean startFragment(String name) {
        if (isStarted()) {
            return this.breadcrumbService.startFragment(name);
        }
        return false;
    }

    /**
     * Log the end of a fragment.
     * <p>
     * A matching call to startFragment must be made before this is called.
     *
     * @param name the name of the fragment to log
     */
    public boolean endFragment(String name) {
        if (isStarted()) {
            return this.breadcrumbService.endFragment(name);
        }
        return false;
    }

    /**
     * Logs the fact that a particular view was entered.
     * <p>
     * If the previously logged view has the same name, a duplicate view breadcrumb will not be
     * logged.
     *
     * @param screen the name of the view to log
     */
    void logView(String screen) {
        if (isStarted()) {
            this.breadcrumbService.logView(screen, System.currentTimeMillis());
        }
    }

    /**
     * Logs that a particular WebView URL was loaded.
     *
     * @param url the url to log
     */
    void logWebView(String url) {
        if (isStarted()) {
            this.breadcrumbService.logWebView(url, System.currentTimeMillis());
        }
    }

    /**
     * Logs the fact that a particular view was entered.
     * <p>
     * If the previously logged view has the same name, a duplicate view breadcrumb will be
     * logged, and not treated as a duplicate.
     *
     * @param screen the name of the view to log
     */
    void forceLogView(String screen) {
        if (isStarted()) {
            this.breadcrumbService.forceLogView(screen, System.currentTimeMillis());
        }
    }

    /**
     * Logs a tap on a screen element.
     *
     * @param point       the coordinates of the screen tap
     * @param elementName the name of the element which was tapped
     * @param type        the type of tap that occurred
     */
    void logTap(Pair<Float, Float> point, String elementName, TapBreadcrumb.TapBreadcrumbType type) {
        if (isStarted()) {
            this.breadcrumbService.logTap(point, elementName, System.currentTimeMillis(), type);
        }
    }

    EventService getEventService() {
        return eventService;
    }

    EmbraceRemoteLogger getRemoteLogger() {
        return remoteLogger;
    }

    EmbraceExceptionService getExceptionsService() {
        return exceptionsService;
    }

    MetadataService getMetadataService() {
        return metadataService;
    }

    LocalConfig getLocalConfig() {
        return localConfig;
    }

    EmbraceSessionService getSessionService() {
        return sessionService;
    }

    private Map<String, Object> normalizeProperties(Map<String, Object> properties) {
        Map<String, Object> normalizedProperties = new HashMap<>();
        if (properties != null) {
            try {
                normalizedProperties = PropertyUtils.sanitizeProperties(properties);
            } catch (Exception e) {
                EmbraceLogger.logError("Exception occurred while normalizing the properties.", e, true);
            }
        }
        return normalizedProperties;
    }

    public enum AppFramework {
        NATIVE(1),
        REACT_NATIVE(2);

        private int value;

        AppFramework(int value) {
            this.value = value;
        }

        public int getValue() {
            return value;
        }
    }
}