package io.embrace.android.embracesdk;

import com.google.gson.annotations.SerializedName;

import java.util.List;
import java.util.Map;

/**
 * Represents a particular user's session within the app.
 */
class Session {
    /**
     * Enum to discriminate the different ways a session can start / end
     */
    enum SessionLifeEventType {
        @SerializedName("s")
        STATE,
        @SerializedName("m")
        MANUAL,
        @SerializedName("t")
        TIMED
    }

    /**
     * A unique ID which identifies the session.
     */
    @SerializedName("id")
    private final String sessionId;

    /**
     * The time that the session started.
     */
    @SerializedName("st")
    private final Long startTime;

    /**
     * The time that the session ended.
     */
    @SerializedName("et")
    private final Long endTime;

    /**
     * The ordinal of the session, starting from 1.
     */
    @SerializedName("sn")
    private final Integer number;

    @SerializedName("ht")
    private final Long lastHeartbeatTime;

    @SerializedName("tt")
    private final Long terminationTime;

    @SerializedName("ls")
    private final String lastState;

    @SerializedName("ba")
    private final Double startingBatteryLevel;

    @SerializedName("ce")
    private final Boolean endedCleanly;

    @SerializedName("tr")
    private final Boolean receivedTermination;

    @SerializedName("cs")
    private final Boolean coldStart;

    @SerializedName("ss")
    private final List<String> eventIds;

    @SerializedName("t")
    private final String sessionType;

    @SerializedName("il")
    private final List<String> infoLogIds;

    @SerializedName("wl")
    private final List<String> warningLogIds;

    @SerializedName("el")
    private final List<String> errorLogIds;

    @SerializedName("e")
    private final ExceptionError exceptionError;

    @SerializedName("cr")
    private final Boolean crashed;

    @SerializedName("exc")
    private final String stacktrace;

    @SerializedName("em")
    private final SessionLifeEventType endType;

    @SerializedName("sm")
    private final SessionLifeEventType startType;

    @SerializedName("oc")
    private final List<Orientation> orientations;

    @SerializedName("sp")
    private final Map<String, String> properties;

    private transient final UserInfo user;

    public String getSessionId() {
        return sessionId;
    }

    public UserInfo getUser() {
        return user;
    }

    public Long getStartTime() {
        return startTime;
    }

    public Long getEndTime() {
        return endTime;
    }

    public Integer getNumber() {
        return number;
    }

    public Long getLastHeartbeatTime() {
        return lastHeartbeatTime;
    }

    public Long getTerminationTime() {
        return terminationTime;
    }

    public String getLastState() {
        return lastState;
    }

    public Double getStartingBatteryLevel() {
        return startingBatteryLevel;
    }

    public Boolean isEndedCleanly() {
        return endedCleanly;
    }

    public Boolean isReceivedTermination() {
        return receivedTermination;
    }

    public Boolean isColdStart() {
        return coldStart;
    }

    public List<String> getEventIds() {
        return eventIds;
    }

    public String getSessionType() {
        return sessionType;
    }

    public List<String> getInfoLogIds() {
        return infoLogIds;
    }

    public List<String> getWarningLogIds() {
        return warningLogIds;
    }

    public List<String> getErrorLogIds() {
        return errorLogIds;
    }

    public ExceptionError getExceptionError() {
        return exceptionError;
    }

    public String getStacktrace() {
        return stacktrace;
    }

    public SessionLifeEventType getEndType() {
        return endType;
    }

    public SessionLifeEventType getStartType() {
        return startType;
    }

    public List<Orientation> getOrientations() {
        return orientations;
    }

    public Map<String, String> getProperties() {
        return properties;
    }

    private Session(Builder builder) {
        sessionId = builder.sessionId;
        user = builder.user;
        startTime = builder.startTime;
        endTime = builder.endTime;
        number = builder.number;
        lastHeartbeatTime = builder.lastHeartbeatTime;
        terminationTime = builder.terminationTime;
        lastState = builder.lastState;
        startingBatteryLevel = builder.startingBatteryLevel;
        endedCleanly = builder.endedCleanly;
        receivedTermination = builder.receivedTermination;
        coldStart = builder.coldStart;
        eventIds = builder.eventIds;
        sessionType = builder.sessionType;
        infoLogIds = builder.infoLogIds;
        warningLogIds = builder.warningLogIds;
        errorLogIds = builder.errorLogIds;
        exceptionError = builder.exceptionError;
        crashed = builder.crashed;
        stacktrace = builder.stacktrace;
        startType = builder.startType;
        endType = builder.endType;
        orientations = builder.orientations;
        properties = builder.properties;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static Builder newBuilder(Session copy) {
        Builder builder = new Builder();
        builder.sessionId = copy.getSessionId();
        builder.user = copy.getUser();
        builder.startTime = copy.getStartTime();
        builder.endTime = copy.getEndTime();
        builder.number = copy.getNumber();
        builder.lastHeartbeatTime = copy.getLastHeartbeatTime();
        builder.terminationTime = copy.getTerminationTime();
        builder.lastState = copy.getLastState();
        builder.startingBatteryLevel = copy.getStartingBatteryLevel();
        builder.endedCleanly = copy.isEndedCleanly();
        builder.receivedTermination = copy.isReceivedTermination();
        builder.coldStart = copy.isColdStart();
        builder.eventIds = copy.getEventIds();
        builder.sessionType = copy.getSessionType();
        builder.infoLogIds = copy.getInfoLogIds();
        builder.warningLogIds = copy.getWarningLogIds();
        builder.errorLogIds = copy.getErrorLogIds();
        builder.exceptionError = copy.getExceptionError();
        builder.stacktrace = copy.getStacktrace();
        builder.startType = copy.getStartType();
        builder.endType = copy.getEndType();
        builder.orientations = copy.getOrientations();
        builder.properties = copy.getProperties();

        return builder;
    }


    public static final class Builder {

        private String sessionId;
        private UserInfo user;
        private Long startTime;
        private Long endTime;
        private Integer number;
        private Long lastHeartbeatTime;
        private Long terminationTime;
        private String lastState;
        private Double startingBatteryLevel;
        private Boolean endedCleanly;
        private Boolean receivedTermination;
        private Boolean coldStart;
        private List<String> eventIds;
        private String sessionType;
        private List<String> infoLogIds;
        private List<String> warningLogIds;
        private List<String> errorLogIds;
        private ExceptionError exceptionError;
        private Boolean crashed;
        private String stacktrace;
        private SessionLifeEventType endType;
        private SessionLifeEventType startType;
        private List<Orientation> orientations;
        private Map<String, String> properties;

        private Builder() {
        }

        public Builder withSessionId(String sessionId) {
            this.sessionId = sessionId;
            return this;
        }

        public Builder withUserInfo(UserInfo userInfo) {
            this.user = userInfo;
            return this;
        }

        public Builder withStartTime(Long startTime) {
            this.startTime = startTime;
            return this;
        }

        public Builder withEndTime(Long endTime) {
            this.endTime = endTime;
            return this;
        }

        public Builder withNumber(Integer number) {
            this.number = number;
            return this;
        }

        public Builder withLastHeartbeatTime(Long lastHeartbeatTime) {
            this.lastHeartbeatTime = lastHeartbeatTime;
            return this;
        }

        public Builder withTerminationTime(Long terminationTime) {
            this.terminationTime = terminationTime;
            return this;
        }

        public Builder withLastState(String lastState) {
            this.lastState = lastState;
            return this;
        }

        public Builder withStartingBatteryLevel(double startingBatteryLevel) {
            this.startingBatteryLevel = startingBatteryLevel;
            return this;
        }

        public Builder withEndedCleanly(boolean endedCleanly) {
            this.endedCleanly = endedCleanly;
            return this;
        }

        public Builder withReceivedTermination(boolean receivedTermination) {
            this.receivedTermination = receivedTermination;
            return this;
        }

        public Builder withColdStart(boolean coldStart) {
            this.coldStart = coldStart;
            return this;
        }

        public Builder withEventIds(List<String> eventIds) {
            this.eventIds = eventIds;
            return this;
        }

        public Builder withSessionType(String sessionType) {
            this.sessionType = sessionType;
            return this;
        }

        public Builder withInfoLogIds(List<String> infoLogIds) {
            this.infoLogIds = infoLogIds;
            return this;
        }

        public Builder withWarningLogIds(List<String> warningLogIds) {
            this.warningLogIds = warningLogIds;
            return this;
        }

        public Builder withErrorLogIds(List<String> errorLogIds) {
            this.errorLogIds = errorLogIds;
            return this;
        }

        public Builder withExceptionErrors(ExceptionError exceptionError) {
            this.exceptionError = exceptionError;
            return this;
        }

        public Builder withCrashed(boolean crashed) {
            this.crashed = crashed;
            return this;
        }

        public Builder withExceptionStacktrace(String stacktrace) {
            this.stacktrace = stacktrace;
            return this;
        }

        public Builder withEndType(SessionLifeEventType endType) {
            this.endType = endType;
            return this;
        }

        public Builder withStartType(SessionLifeEventType startType) {
            this.startType = startType;
            return this;
        }

        public Builder withProperties(EmbraceSessionProperties properties) {
            this.properties = properties.get();
            return this;
        }

        public Builder withOrientations(List<Orientation> orientations) {
            this.orientations = orientations;
            return this;
        }

        public Session build() {
            return new Session(this);
        }
    }
}
