package io.embrace.android.embracesdk;

import com.fernandocejas.arrow.optional.Optional;

import java.util.Set;

interface PreferencesService {

    long DAY_IN_MS = 60 * 60 * 24 * 1000L;

    /**
     * @return the last registered Host App version name
     */
    Optional<String> getAppVersion();

    /**
     * Sets the App Version
     *
     * @param appVersion host app version
     */
    void setAppVersion(String appVersion);

    /**
     * @return the last registered OS Version
     */
    Optional<String> getOsVersion();

    /**
     * Sets the OS Version
     *
     * @param osVersion the OS Version
     */
    void setOsVersion(String osVersion);

    /**
     * @return the app install date
     */
    Optional<Long> getInstallDate();

    /**
     * Sets the install date of the app.
     *
     * @param installDate install date in millis timestamp.
     */
    void setInstallDate(long installDate);

    /**
     * @return the embrace device unique identifier.
     */
    Optional<String> getDeviceIdentifier();

    /**
     * Sets the unique identifier for this device.
     *
     * @param deviceIdentifier Embrace unique device identifier.
     */
    void setDeviceIdentifier(String deviceIdentifier);

    /**
     * @return the last SDK startup status registered.
     */
    Optional<String> getSDKStartupStatus();

    /**
     * @return if the sdk is disabled
     */
    boolean getSDKDisabled();

    /**
     * @param disabled the SDK disabled status
     */
    void setSDKDisabled(boolean disabled);

    /**
     * @return if the user is payer
     */
    boolean getUserPayer();

    /**
     * @param payer status of payer for the user
     */
    void setUserPayer(boolean payer);

    /**
     * @return user identifier
     */
    Optional<String> getUserIdentifier();

    /**
     * @param identifier user unique identifier
     */
    void setUserIdentifier(String identifier);

    /**
     * @return user email
     */
    Optional<String> getUserEmailAddress();

    /**
     * @param emailAddress user email address
     */
    void setUserEmailAddress(String emailAddress);

    /**
     * @return user's personas
     */
    Optional<Set<String>> getUserPersonas();

    /**
     * @param personas personas for the user
     */
    void setUserPersonas(Set<String> personas);

    /**
     * @return custom personas
     * <p>
     * No longer used, will be removed in a future version.
     * <p>
     * Method is still present to ensure that during any upgrades to SDK3, any custom
     * personas are merged with the user personas list.
     */
    @Deprecated
    Optional<Set<String>> getCustomPersonas();

    /**
     * @return username for the user
     */
    Optional<String> getUsername();

    /**
     * @param username user's username
     */
    void setUsername(String username);

    /**
     * @return the last time config was fetched from the server
     */
    Optional<Long> getLastConfigFetchDate();

    /**
     * @param fetchDate time of the last time config was fetched from the server
     */
    void setLastConfigFetchDate(long fetchDate);

    /**
     * @return if the user message needs to retry send
     */
    boolean userMessageNeedsRetry();

    /**
     * @param needsRetry sets if the user message needs to retry send
     */
    void setUserMessageNeedsRetry(boolean needsRetry);

    /**
     * Whether or not the app was installed within the last 24 hours.
     *
     * @return true if it is the user's first day, false otherwise
     */
    default boolean isUsersFirstDay() {
        return getInstallDate().isPresent() &&
                System.currentTimeMillis() - getInstallDate().get() <= DAY_IN_MS;
    }

    /**
     * @param sessionNumber sets incremented by one
     */
    void setSessionNumber(int sessionNumber);

    /**
     * @return last session number.
     */
    Optional<Integer> getSessionNumber();

    /**
     * @param url for javaScript bundle.
     */
    void setJavaScriptBundleURL(String url);

    /**
     * @return last javaScript bundle string url.
     */
    Optional<String> getJavaScriptBundleURL();

    /**
     * @param number for javaScript patch.
     */
    void setJavaScriptPatchNumber(String number);

    /**
     * @return last javaScript patch string number.
     */
    Optional<String> getJavaScriptPatchNumber();

    /**
     * @param version for react native.
     */
    void setReactNativeVersionNumber(String version);

    /**
     * @return last react native version.
     */
    Optional<String> getReactNativeVersionNumber();
}
