package io.embrace.android.embracesdk;

import android.util.Base64;

import com.google.gson.annotations.SerializedName;

import java.util.UUID;

/**
 * Describes a HTTP network call.
 * <p>
 * The request could result in either a HTTP response, or a client-side error.
 */
final class NetworkCall {
    /** A random ID which can be used to uniquely identify the request. */
    @SerializedName("rid")
    private final String requestId;

    /** The HTTP method the network request corresponds to. */
    @SerializedName("x")
    private final String httpMethod;

    /** The duration of the network request. */
    @SerializedName("dur")
    private final long duration;

    /** The number of bytes received during the network request. */
    @SerializedName("bi")
    private final long bytesReceived;

    /** The number of bytes sent during the network request. */
    @SerializedName("bo")
    private final long bytesSent;

    /** The URL being requested. */
    @SerializedName("url")
    private final String url;

    /** The start time of the request. */
    @SerializedName("st")
    private final long startTime;

    /** The end time of the request. */
    @SerializedName("et")
    private final long endTime;

    /** The HTTP status code returned by the request, if the request was completed. */
    @SerializedName("rc")
    private final Integer statusCode;

    /** Whether an exception was thrown whilst attempting to make the network call. */
    @SerializedName("er")
    private final boolean didClientError;

    /** If an exception was thrown, the name of the class which caused the exception. */
    @SerializedName("ed")
    private final String errorType;

    /** If an exception was thrown, the exception message. */
    @SerializedName("de")
    private final String errorMessage;

    public String getRequestId() {
        return requestId;
    }

    public String getHttpMethod() {
        return httpMethod;
    }

    public long getDuration() {
        return duration;
    }

    public long getBytesReceived() {
        return bytesReceived;
    }

    public long getBytesSent() {
        return bytesSent;
    }

    public String getUrl() {
        return url;
    }

    public long getStartTime() {
        return startTime;
    }

    public long getEndTime() {
        return endTime;
    }

    public Integer getStatusCode() {
        return statusCode;
    }

    public boolean isDidClientError() {
        return didClientError;
    }

    public String getErrorType() {
        return errorType;
    }

    public String getErrorMessage() {
        return errorMessage;
    }

    private NetworkCall(Builder builder) {
        requestId = builder.requestId;
        httpMethod = builder.httpMethod;
        duration = builder.duration;
        bytesReceived = builder.bytesReceived;
        bytesSent = builder.bytesSent;
        url = builder.url;
        startTime = builder.startTime;
        endTime = builder.endTime;
        statusCode = builder.statusCode;
        didClientError = builder.didClientError;
        errorType = builder.errorType;
        errorMessage = builder.errorMessage;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static final class Builder {
        private String requestId = createRequestId();
        private String httpMethod;
        private long duration;
        private long bytesReceived;
        private long bytesSent;
        private String url;
        private long startTime;
        private long endTime;
        private Integer statusCode;
        private boolean didClientError;
        private String errorType;
        private String errorMessage;

        private Builder() {
        }

        public Builder withRequestId(String requestId) {
            this.requestId = requestId;
            return this;
        }

        public Builder withHttpMethod(String httpMethod) {
            this.httpMethod = httpMethod;
            return this;
        }

        public Builder withDuration(long duration) {
            this.duration = duration;
            return this;
        }

        public Builder withBytesReceived(long bytesReceived) {
            this.bytesReceived = bytesReceived;
            return this;
        }

        public Builder withBytesSent(long bytesSent) {
            this.bytesSent = bytesSent;
            return this;
        }

        public Builder withUrl(String url) {
            this.url = url;
            return this;
        }

        public Builder withStartTime(long startTime) {
            this.startTime = startTime;
            return this;
        }

        public Builder withEndTime(long endTime) {
            this.endTime = endTime;
            return this;
        }

        public Builder withStatusCode(Integer statusCode) {
            this.statusCode = statusCode;
            return this;
        }

        public Builder withDidClientError(boolean didClientError) {
            this.didClientError = didClientError;
            return this;
        }

        public Builder withErrorType(String errorType) {
            this.errorType = errorType;
            return this;
        }

        public Builder withErrorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
            return this;
        }

        public NetworkCall build() {
            return new NetworkCall(this);
        }
    }

    private static String createRequestId() {
        return String.format("%.6s", Base64.encodeToString(UUID.randomUUID().toString().getBytes(), Base64.NO_WRAP)
                .replace('+', '0')
                .replace(' ', '0'));
    }

    public boolean isValidNetworkCall(){
        if (!didClientError && statusCode == null) {
            EmbraceLogger.logDebug("Failed to log network call: Status code is required.");
        }
        if (statusCode != null && (statusCode < 100 || statusCode > 599)) {
            EmbraceLogger.logDebug("Failed to log network call: Invalid status code (valid ranges: 100 ~ 599 inclusive).");
            return false;
        } else if (startTime < 0) {
            EmbraceLogger.logDebug("Failed to log network call: Start time is a negative value.");
            return false;
        } else if (endTime < 0) {
            EmbraceLogger.logDebug("Failed to log network call: End time is a negative value.");
            return false;
        }
        return true;
    }
}
