package io.embrace.android.embracesdk;

import android.content.Context;
import android.content.SharedPreferences;
import android.preference.PreferenceManager;
import android.text.TextUtils;

import com.fernandocejas.arrow.checks.Preconditions;
import com.fernandocejas.arrow.optional.Optional;

import java.util.HashSet;
import java.util.Set;

final class EmbracePreferencesService implements PreferencesService, ActivityListener {

    static final String SDK_STARTUP_IN_PROGRESS = "startup_entered";
    static final String SDK_STARTUP_COMPLETED = "startup_completed";
    private static final String SDK_STARTUP_STATUS_KEY = "io.embrace.sdkstartup";

    private static final String DEVICE_IDENTIFIER_KEY = "io.embrace.deviceid";
    private static final String PREVIOUS_APP_VERSION_KEY = "io.embrace.lastappversion";
    private static final String PREVIOUS_OS_VERSION_KEY = "io.embrace.lastosversion";
    private static final String INSTALL_DATE_KEY = "io.embrace.installtimestamp";
    private static final String USER_IDENTIFIER_KEY = "io.embrace.userid";
    private static final String USER_EMAIL_ADDRESS_KEY = "io.embrace.useremail";
    private static final String USER_USERNAME_KEY = "io.embrace.username";
    private static final String USER_IS_PAYER_KEY = "io.embrace.userispayer";
    private static final String USER_PERSONAS_KEY = "io.embrace.userpersonas";
    private static final String CUSTOM_PERSONAS_KEY = "io.embrace.custompersonas";
    private static final String LAST_USER_MESSAGE_FAILED_KEY = "io.embrace.userupdatefailed";
    private static final String LAST_SESSION_NUMBER_KEY = "io.embrace.sessionnumber";
    private static final String JAVA_SCRIPT_BUNDLE_URL_KEY = "io.embrace.jsbundle.url";
    private static final String JAVA_SCRIPT_PATCH_NUMBER_KEY = "io.embrace.javascript.patch";
    private static final String REACT_NATIVE_VERSION_KEY = "io.embrace.reactnative.version";

    private static final String SDK_DISABLED_KEY = "io.embrace.disabled";
    private static final String SDK_CONFIG_FETCHED_TIMESTAMP = "io.embrace.sdkfetchedtimestamp";

    private final SharedPreferences preferences;

    @Override
    public void applicationStartupComplete() {
        setStringPreference(preferences, SDK_STARTUP_STATUS_KEY, SDK_STARTUP_COMPLETED);
    }

    EmbracePreferencesService(Context context, ActivityService activityService) {
        Preconditions.checkNotNull(context, "context must not be null");
        this.preferences = PreferenceManager.getDefaultSharedPreferences(context);
        Preconditions.checkNotNull(activityService).addListener(this);
        setStringPreference(preferences, SDK_STARTUP_STATUS_KEY, SDK_STARTUP_IN_PROGRESS);

    }

    private static Optional<String> getStringPreference(SharedPreferences preferences, String key) {
        String value = null;
        if (preferences != null) {
            value = preferences.getString(key, null);
        }
        return Optional.fromNullable(value);
    }

    private static void setStringPreference(SharedPreferences preferences, String key, String value) {
        if (preferences != null && !TextUtils.isEmpty(key)) {
            SharedPreferences.Editor editor = preferences.edit();
            editor.putString(key, value);
            editor.apply();
        }
    }

    private static Optional<Long> getLongPreference(SharedPreferences preferences, String key) {
        long defaultValue = -1;
        long value = defaultValue;
        if (preferences != null) {
            value = preferences.getLong(key, defaultValue);
        }
        if (value == -1) {
            return Optional.absent();
        } else {
            return Optional.fromNullable(value);
        }
    }

    private static void setLongPreference(SharedPreferences preferences, String key, long value) {

        if (preferences != null) {
            SharedPreferences.Editor editor = preferences.edit();
            editor.putLong(key, value);
            editor.apply();
        }
    }

    private static Optional<Float> getFloatPreference(SharedPreferences preferences, String key) {
        Float defaultValue = -1f;
        float value = defaultValue;
        if (preferences != null) {
            value = preferences.getFloat(key, defaultValue);
        }
        if (value == -1f) {
            return Optional.absent();
        } else {
            return Optional.fromNullable(value);
        }
    }

    private static void setFloatPreference(SharedPreferences preferences, String key, float value) {
        if (preferences != null) {
            SharedPreferences.Editor editor = preferences.edit();
            editor.putFloat(key, value);
            editor.apply();
        }
    }

    private static Optional<Integer> getIntegerPreference(SharedPreferences preferences, String key) {
        int defaultValue = -1;
        int value = defaultValue;
        if (preferences != null) {
            value = preferences.getInt(key, defaultValue);
        }
        if (value == -1) {
            return Optional.absent();
        } else {
            return Optional.fromNullable(value);
        }
    }

    private static void setIntegerPreference(SharedPreferences preferences, String key, int value) {
        if (preferences != null) {
            SharedPreferences.Editor editor = preferences.edit();
            editor.putInt(key, value);
            editor.apply();
        }
    }

    private static boolean getBooleanPreference(SharedPreferences preferences, String key, boolean defaultValue) {
        boolean value = defaultValue;
        if (preferences != null) {
            value = preferences.getBoolean(key, defaultValue);
        }
        return value;
    }

    private static void setBooleanPreference(SharedPreferences preferences, String key, boolean value) {
        if (preferences != null) {
            SharedPreferences.Editor editor = preferences.edit();
            editor.putBoolean(key, value);
            editor.apply();
        }
    }

    private static void setArrayPreference(SharedPreferences preferences, String key, Set<String> value) {
        if (preferences != null) {
            SharedPreferences.Editor editor = preferences.edit();
            editor.putStringSet(key, value);
            editor.apply();
        }
    }

    private static Optional<Set<String>> getArrayPreference(SharedPreferences preferences, String key) {
        Set<String> defaultValue = null;
        Set<String> value = new HashSet<>();
        if (preferences != null) {
            value = preferences.getStringSet(key, defaultValue);
        }
        return Optional.fromNullable(value);
    }

    @Override
    public Optional<String> getAppVersion() {
        return getStringPreference(preferences, PREVIOUS_APP_VERSION_KEY);
    }

    @Override
    public void setAppVersion(String appVersion) {
        setStringPreference(preferences, PREVIOUS_APP_VERSION_KEY, appVersion);
    }

    @Override
    public Optional<String> getOsVersion() {
        return getStringPreference(preferences, PREVIOUS_OS_VERSION_KEY);
    }

    @Override
    public void setOsVersion(String osVersion) {
        setStringPreference(preferences, PREVIOUS_OS_VERSION_KEY, osVersion);
    }

    @Override
    public Optional<Long> getInstallDate() {
        return getLongPreference(preferences, INSTALL_DATE_KEY);
    }

    @Override
    public void setInstallDate(long installDate) {
        setLongPreference(preferences, INSTALL_DATE_KEY, installDate);
    }

    @Override
    public Optional<String> getDeviceIdentifier() {
        return getStringPreference(preferences, DEVICE_IDENTIFIER_KEY);
    }

    @Override
    public void setDeviceIdentifier(String deviceIdentifier) {
        setStringPreference(preferences, DEVICE_IDENTIFIER_KEY, deviceIdentifier);
    }

    @Override
    public Optional<String> getSDKStartupStatus() {
        return getStringPreference(preferences, SDK_STARTUP_STATUS_KEY);
    }

    @Override
    public boolean getSDKDisabled() {
        return getBooleanPreference(preferences, SDK_DISABLED_KEY, false);
    }

    @Override
    public void setSDKDisabled(boolean disabled) {
        setBooleanPreference(preferences, SDK_DISABLED_KEY, disabled);
    }

    @Override
    public boolean getUserPayer() {
        return getBooleanPreference(preferences, USER_IS_PAYER_KEY, false);
    }

    @Override
    public void setUserPayer(boolean payer) {
        setBooleanPreference(preferences, USER_IS_PAYER_KEY, payer);
    }

    @Override
    public Optional<String> getUserIdentifier() {
        return getStringPreference(preferences, USER_IDENTIFIER_KEY);
    }

    @Override
    public void setUserIdentifier(String identifier) {
        setStringPreference(preferences, USER_IDENTIFIER_KEY, identifier);
    }

    @Override
    public Optional<String> getUserEmailAddress() {
        return getStringPreference(preferences, USER_EMAIL_ADDRESS_KEY);
    }

    @Override
    public void setUserEmailAddress(String emailAddress) {
        setStringPreference(preferences, USER_EMAIL_ADDRESS_KEY, emailAddress);
    }

    @Override
    public Optional<Set<String>> getUserPersonas() {
        return getArrayPreference(preferences, USER_PERSONAS_KEY);
    }

    @Override
    public void setUserPersonas(Set<String> personas) {
        setArrayPreference(preferences, USER_PERSONAS_KEY, personas);
    }

    @Override
    public Optional<Set<String>> getCustomPersonas() {
        return getArrayPreference(preferences, CUSTOM_PERSONAS_KEY);
    }

    @Override
    public Optional<String> getUsername() {
        return getStringPreference(preferences, USER_USERNAME_KEY);
    }

    @Override
    public void setUsername(String username) {
        setStringPreference(preferences, USER_USERNAME_KEY, username);
    }

    @Override
    public Optional<Long> getLastConfigFetchDate() {
        return getLongPreference(preferences, SDK_CONFIG_FETCHED_TIMESTAMP);
    }

    @Override
    public void setLastConfigFetchDate(long fetchDate) {
        setLongPreference(preferences, SDK_CONFIG_FETCHED_TIMESTAMP, fetchDate);
    }

    @Override
    public boolean userMessageNeedsRetry() {
        return getBooleanPreference(preferences, LAST_USER_MESSAGE_FAILED_KEY, false);
    }

    @Override
    public void setUserMessageNeedsRetry(boolean needsRetry) {
        setBooleanPreference(preferences, LAST_USER_MESSAGE_FAILED_KEY, needsRetry);
    }

    @Override
    public void setSessionNumber(int sessionNumber) {
        setIntegerPreference(preferences, LAST_SESSION_NUMBER_KEY, sessionNumber);
    }

    @Override
    public Optional<Integer> getSessionNumber() {
        return getIntegerPreference(preferences, LAST_SESSION_NUMBER_KEY);
    }

    @Override
    public void setJavaScriptBundleURL(String url) {
        setStringPreference(preferences, JAVA_SCRIPT_BUNDLE_URL_KEY, url);
    }

    @Override
    public Optional<String> getJavaScriptBundleURL() {
        return getStringPreference(preferences, JAVA_SCRIPT_BUNDLE_URL_KEY);
    }

    @Override
    public void setJavaScriptPatchNumber(String number) {
        setStringPreference(preferences, JAVA_SCRIPT_PATCH_NUMBER_KEY, number);
    }

    @Override
    public Optional<String> getJavaScriptPatchNumber() {
        return getStringPreference(preferences, JAVA_SCRIPT_PATCH_NUMBER_KEY);
    }

    @Override
    public void setReactNativeVersionNumber(String version) {
        setStringPreference(preferences, REACT_NATIVE_VERSION_KEY, version);
    }

    @Override
    public Optional<String> getReactNativeVersionNumber() {
        return getStringPreference(preferences, REACT_NATIVE_VERSION_KEY);
    }
}
