package io.embrace.android.embracesdk;

import android.app.ActivityManager;
import android.content.Context;

import com.fernandocejas.arrow.checks.Preconditions;

import java.util.ArrayList;
import java.util.List;
import java.util.NavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.TimeUnit;

/**
 * Polls for the device's available and used memory.
 * <p>
 * Stores memory warnings when the {@link ActivityService} detects a memory trim event.
 */
class EmbraceMemoryService implements MemoryService, MemoryCleanerListener {
    private final Runtime runtime = Runtime.getRuntime();

    private final ScheduledWorker memoryWorker;

    private final NavigableMap<Long, MemorySample> memorySamples;

    private final NavigableMap<Long, MemoryWarning> memoryWarnings;

    private static final int BYTES_IN_MB = 1024 * 1024;

    public EmbraceMemoryService(ActivityManager activityManager, MemoryCleanerService memoryCleanerService) {
        Preconditions.checkNotNull(activityManager, "activityManager must not be null");
        this.memoryWorker = ScheduledWorker.ofSingleThread("Memory Service");
        this.memoryWorker.scheduleAtFixedRate(() -> queryMemory(activityManager), 0, 2, TimeUnit.SECONDS);
        this.memorySamples = new ConcurrentSkipListMap<>();
        this.memoryWarnings = new ConcurrentSkipListMap<>();
        Preconditions.checkNotNull(memoryCleanerService).addListener(this);
    }

    public static EmbraceMemoryService ofContext(Context context, MemoryCleanerService memoryCleanerService) {
        return new EmbraceMemoryService((ActivityManager) context.getSystemService(Context.ACTIVITY_SERVICE),
                memoryCleanerService);
    }

    private void queryMemory(ActivityManager activityManager) {
        try {
            ActivityManager.MemoryInfo memoryInfo = new ActivityManager.MemoryInfo();
            activityManager.getMemoryInfo(memoryInfo);
            long heapUsed = (runtime.totalMemory() - runtime.freeMemory()) / BYTES_IN_MB;
            long systemAvailable = memoryInfo.availMem / BYTES_IN_MB;
            long timestamp = System.currentTimeMillis();
            memorySamples.put(timestamp, new MemorySample(timestamp, heapUsed, systemAvailable));
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to query for memory usage", ex);
        }
    }

    @Override
    public List<MemorySample> getMemorySamples(long startTime, long endTime) {
        return new ArrayList<>(this.memorySamples.subMap(startTime, endTime).values());
    }

    @Override
    public List<MemoryWarning> getMemoryWarnings(long startTime, long endTime) {
        return new ArrayList<>(this.memoryWarnings.subMap(startTime, endTime).values());
    }

    @Override
    public void close() {
        EmbraceLogger.logDebug("Stopping EmbraceMemoryService");
        memoryWorker.close();
    }

    @Override
    public void onMemoryWarning() {
        long timestamp = System.currentTimeMillis();
        this.memoryWarnings.put(timestamp, new MemoryWarning(timestamp));
    }

    @Override
    public void cleanCollections() {
        this.memorySamples.clear();
        this.memoryWarnings.clear();
    }
}
