package io.embrace.android.embracesdk;

import android.content.Context;
import android.telephony.SignalStrength;
import android.telephony.TelephonyManager;

import java.lang.reflect.Method;

/**
 * Calculates signal level for LTE/EVDO.
 */
class SignalStrengthUtils {

    static final int SIGNAL_STRENGTH_NONE_OR_UNKNOWN = 0;

    static final int SIGNAL_STRENGTH_POOR = 1;
    static final int SIGNAL_STRENGTH_MODERATE = 2;
    static final int SIGNAL_STRENGTH_GOOD = 3;
    static final int SIGNAL_STRENGTH_GREAT = 4;


    private final TelephonyManager mTelephonyManager;

    SignalStrengthUtils(Context context) {
        mTelephonyManager = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
    }

    /**
     * @return a value between 0 {@link #SIGNAL_STRENGTH_NONE_OR_UNKNOWN} and 4 {@link #SIGNAL_STRENGTH_GREAT}.
     */
    int getLevel(SignalStrength signalStrength) {
        try {
            int lteLevel = getLteLevel(signalStrength);
            if (lteLevel != SIGNAL_STRENGTH_NONE_OR_UNKNOWN) {
                return lteLevel;
            } else {
                int evdoLevel = getEvdoLevel(signalStrength);
                if (evdoLevel != SIGNAL_STRENGTH_NONE_OR_UNKNOWN) {
                    return evdoLevel;
                } else {
                    return SIGNAL_STRENGTH_POOR;
                }
            }
        } catch (Throwable ex) {
            return SIGNAL_STRENGTH_NONE_OR_UNKNOWN;
        }
    }

    /**
     * Get Evdo as level 0..4
     */
    private int getEvdoLevel(SignalStrength signalStrength) {
        int evdoDbm = signalStrength.getEvdoDbm();
        int evdoSnr = signalStrength.getEvdoSnr();
        int levelEvdoDbm;
        int levelEvdoSnr;

        if (evdoDbm >= -65) levelEvdoDbm = SIGNAL_STRENGTH_GREAT;
        else if (evdoDbm >= -75) levelEvdoDbm = SIGNAL_STRENGTH_GOOD;
        else if (evdoDbm >= -90) levelEvdoDbm = SIGNAL_STRENGTH_MODERATE;
        else if (evdoDbm >= -105) levelEvdoDbm = SIGNAL_STRENGTH_POOR;
        else
            levelEvdoDbm = SIGNAL_STRENGTH_NONE_OR_UNKNOWN;

        if (evdoSnr >= 7) levelEvdoSnr = SIGNAL_STRENGTH_GREAT;
        else if (evdoSnr >= 5) levelEvdoSnr = SIGNAL_STRENGTH_GOOD;
        else if (evdoSnr >= 3) levelEvdoSnr = SIGNAL_STRENGTH_MODERATE;
        else if (evdoSnr >= 1) levelEvdoSnr = SIGNAL_STRENGTH_POOR;
        else
            levelEvdoSnr = SIGNAL_STRENGTH_NONE_OR_UNKNOWN;

        int level = levelEvdoDbm < levelEvdoSnr ? levelEvdoDbm : levelEvdoSnr;
        return level;
    }

    /**
     * Get LTE as level 0..4
     */
    private int getLteLevel(SignalStrength signalStrength) {
        if (mTelephonyManager.getNetworkType() != TelephonyManager.NETWORK_TYPE_LTE) {
            return SIGNAL_STRENGTH_NONE_OR_UNKNOWN;
        }
        try {
            Method methodGetLteLevel = SignalStrength.class.getMethod("getLteLevel");
            return (Integer) methodGetLteLevel.invoke(signalStrength);
        } catch (Throwable t) {
            return SIGNAL_STRENGTH_NONE_OR_UNKNOWN;
        }
    }

}
