package io.embrace.android.embracesdk;

import android.app.Activity;
import android.text.TextUtils;
import android.util.Pair;

import com.fernandocejas.arrow.checks.Preconditions;
import com.fernandocejas.arrow.optional.Optional;

import java.util.Deque;
import java.util.List;
import java.util.concurrent.LinkedBlockingDeque;

import io.embrace.android.embracesdk.TapBreadcrumb.TapBreadcrumbType;
import java9.util.stream.Collectors;
import java9.util.stream.StreamSupport;

/**
 * Handles the logging of breadcrumbs.
 * <p>
 * Breadcrumbs record a user's journey through the app and are split into:
 * <ul>
 * <li>View breadcrumbs: Each time the user changes view in the app</li>
 * <li>Tap breadcrumbs: Each time the user taps a UI element in the app</li>
 * <li>Custom breadcrumbs: User-defined interactions within the app</li>
 * </ul>
 * Breadcrumbs are limited at query-time by default to 100 per session, but this can be overridden
 * in server-side configuration. They are stored in an unbounded queue.
 */
class EmbraceBreadcrumbService implements BreadcrumbService, ActivityListener, MemoryCleanerListener {

    /**
     * The default limit, which can be overriden by {@link Config}.
     */
    private static final Integer DEFAULT_BREADCRUMB_LIMIT = 100;

    /**
     * The config service, for retrieving the breadcrumb limit.
     */
    private final ConfigService configService;

    /*
     * Local config for controlling whether tap coordinates are omitted
     */
    private final LocalConfig localConfig;

    /**
     * A deque of breadcrumbs.
     */
    private final LinkedBlockingDeque<ViewBreadcrumb> viewBreadcrumbs = new LinkedBlockingDeque<>();

    private final LinkedBlockingDeque<TapBreadcrumb> tapBreadcrumbs = new LinkedBlockingDeque<>();

    private final LinkedBlockingDeque<CustomBreadcrumb> customBreadcrumbs = new LinkedBlockingDeque<>();

    private final LinkedBlockingDeque<WebViewBreadcrumb> webViewBreadcrumbs = new LinkedBlockingDeque<>();

    EmbraceBreadcrumbService(ConfigService configService,
                             LocalConfig localConfig,
                             ActivityService activityService,
                             MemoryCleanerService memoryCleanerService) {

        this.configService = Preconditions.checkNotNull(
                configService,
                "configService must not be null");
        this.localConfig = Preconditions.checkNotNull(
                localConfig,
                "localConfig must not be null");
        Preconditions.checkNotNull(activityService, "activityService must not be null");
        activityService.addListener(this);
        Preconditions.checkNotNull(memoryCleanerService).addListener(this);
    }

    @Override
    public void logView(String screen, long timestamp) {
        addToViewLogsQueue(screen, timestamp, false);

    }

    @Override
    public void forceLogView(String screen, long timestamp) {
        addToViewLogsQueue(screen, timestamp, true);
    }

    @Override
    public void logTap(Pair<Float, Float> point, String element, long timestamp, TapBreadcrumbType type) {
        try {
            if (!localConfig.getConfigurations().getTaps().getCaptureCoordinates()) {
                point = new Pair<>(0.0F, 0.0F);
            }
            this.tapBreadcrumbs.push(new TapBreadcrumb(point, element, timestamp, type));
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to log tap breadcrumb for element " + element);
        }
    }

    @Override
    public void logCustom(String message, long timestamp) {
        if (TextUtils.isEmpty(message)) {
            EmbraceLogger.logWarning("Breadcrumb message must not be blank");
            return;
        }
        try {
            this.customBreadcrumbs.push(new CustomBreadcrumb(message, timestamp));
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to log custom breadcrumb with message " + message, ex);
        }
    }

    @Override
    public void logWebView(String url, long startTime) {
        try {
            this.webViewBreadcrumbs.push(new WebViewBreadcrumb(url, startTime));
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to log WebView breadcrumb for url " + url);
        }
    }

    @Override
    public List<ViewBreadcrumb> getViewBreadcrumbsForSession(long startTime, long endTime) {
        int limit = configService.getConfig().getViewBreadcrumbLimit().or(DEFAULT_BREADCRUMB_LIMIT);

        return filterBreadcrumbsForTimeWindow(
                this.viewBreadcrumbs,
                startTime,
                endTime,
                limit);
    }

    @Override
    public List<TapBreadcrumb> getTapBreadcrumbsForSession(long startTime, long endTime) {
        int limit = configService.getConfig().getTapBreadcrumbLimit().or(DEFAULT_BREADCRUMB_LIMIT);

        return filterBreadcrumbsForTimeWindow(
                this.tapBreadcrumbs,
                startTime,
                endTime,
                limit);
    }

    @Override
    public List<CustomBreadcrumb> getCustomBreadcrumbsForSession(long startTime, long endTime) {
        int limit = configService.getConfig().getCustomBreadcrumbLimit().or(DEFAULT_BREADCRUMB_LIMIT);

        return filterBreadcrumbsForTimeWindow(
                this.customBreadcrumbs,
                startTime,
                endTime,
                limit);
    }

    @Override
    public List<WebViewBreadcrumb> getWebViewBreadcrumbsForSession(long startTime, long endTime) {
        int limit = configService.getConfig().getWebViewBreadcrumbLimit().or(DEFAULT_BREADCRUMB_LIMIT);

        return filterBreadcrumbsForTimeWindow(
                this.webViewBreadcrumbs,
                startTime,
                endTime,
                limit);
    }

    @Override
    public Breadcrumbs getBreadcrumbs(long start, long end) {
        return Breadcrumbs.newBuilder()
                .withCustomBreadcrumbs(getCustomBreadcrumbsForSession(start, end))
                .withTapBreadcrumbs(getTapBreadcrumbsForSession(start, end))
                .withViewBreadcrumbs(getViewBreadcrumbsForSession(start, end))
                .withWebViewBreadcrumbs(getWebViewBreadcrumbsForSession(start, end))
                .build();
    }

    @Override
    public Optional<String> getLastViewBreadcrumbScreenName() {
        if (this.viewBreadcrumbs.isEmpty()) {
            return Optional.absent();
        } else {
            return this.viewBreadcrumbs.peek().getScreen();
        }
    }

    @Override
    public void onView(Activity activity) {
        logView(activity.getClass().getName(), System.currentTimeMillis());
    }

    @Override
    public void cleanCollections() {
        this.viewBreadcrumbs.clear();
        this.tapBreadcrumbs.clear();
        this.customBreadcrumbs.clear();
        this.webViewBreadcrumbs.clear();
    }

    /**
     * Adds the view breadcrumb to the queue.
     *
     * @param screen    name of the screen.
     * @param timestamp time of occurrence of the tap event.
     * @param force     will run no duplication checks on the previous view breadcrumb registry.
     */
    private synchronized void addToViewLogsQueue(String screen, long timestamp, boolean force) {
        try {
            ViewBreadcrumb lastViewBreadcrumb = this.viewBreadcrumbs.peek();
            if (force || lastViewBreadcrumb == null
                    || !lastViewBreadcrumb.getScreen().or("").equalsIgnoreCase(String.valueOf(screen))) {
                this.viewBreadcrumbs.push(new ViewBreadcrumb(screen, timestamp));

            }
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to add view breadcrumb for " + screen, ex);
        }
    }

    /**
     * Returns the latest breadcrumbs within the specified interval, up to the maximum queue size or
     * configured limit in the app configuration.
     *
     * @param breadcrumbs breadcrumbs list to filter.
     * @param startTime   beginning of the time window.
     * @param endTime     end of the time window.
     * @return filtered breadcrumbs from the provided FixedSizeDeque.
     */
    private <T extends Breadcrumb> List<T> filterBreadcrumbsForTimeWindow(
            Deque<T> breadcrumbs,
            long startTime,
            long endTime,
            int limit) {

        return StreamSupport.stream(breadcrumbs)
                .filter(breadcrumb -> breadcrumb.getTimestamp() >= startTime)
                .filter(breadcrumb -> endTime <= 0L || breadcrumb.getTimestamp() <= endTime)
                .limit(limit)
                .collect(Collectors.toList());
    }
}

