package io.embrace.android.embracesdk;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;

import com.fernandocejas.arrow.checks.Preconditions;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;

class EmbraceNetworkConnectivityService extends BroadcastReceiver implements NetworkConnectivityService, MemoryCleanerListener {
    private final IntentFilter intentFilter = new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION);
    private final NavigableMap<Long, NetworkStatus> networkReachable;
    private final ConnectivityManager connectivityManager;
    private final Context context;

    public EmbraceNetworkConnectivityService(Context context, MemoryCleanerService memoryCleanerService) {
        this.context = Preconditions.checkNotNull(context, "context must not be null");
        this.connectivityManager =
                (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        this.networkReachable = new TreeMap<>();
        Preconditions.checkNotNull(memoryCleanerService).addListener(this);
        try {
            context.registerReceiver(this, intentFilter);
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to register EmbraceNetworkConnectivityService broadcast receiver. Connectivity status will be unavailable.", ex);
        }
    }

    @Override
    public void onReceive(Context context, Intent intent) {
        try {
            NetworkInfo networkInfo = connectivityManager.getActiveNetworkInfo();
            if (networkInfo != null && networkInfo.isConnected()) {
                // Network is reachable
                if (networkInfo.getType() == ConnectivityManager.TYPE_WIFI) {
                    saveStatus(NetworkStatus.WIFI);
                } else if (networkInfo.getType() == ConnectivityManager.TYPE_MOBILE) {
                    saveStatus(NetworkStatus.WAN);
                }
            } else {
                // Network is not reachable
                saveStatus(NetworkStatus.NOT_REACHABLE);
            }
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to handle CONNECTIVITY_ACTION broadcast", ex);
        }
    }

    @Override
    public List<Interval> getNetworkInterfaceIntervals(long startTime, long endTime) {
        synchronized (this) {
            List<Interval> results = new ArrayList<>();
            for (Map.Entry<Long, NetworkStatus> entry : networkReachable.subMap(startTime, endTime).entrySet()) {
                NetworkStatus value = entry.getValue();
                if (!value.equals(NetworkStatus.NOT_REACHABLE)) {
                    long currentTime = entry.getKey();
                    Long next = networkReachable.higherKey(currentTime);
                    results.add(new Interval(currentTime, next != null ? next : currentTime, value.getName()));
                }
            }
            return results;
        }
    }


    private void saveStatus(NetworkStatus networkStatus) {
        synchronized (this) {
            if (networkReachable.isEmpty() || !networkReachable.lastEntry().getValue().equals(networkStatus)) {
                networkReachable.put(System.currentTimeMillis(), networkStatus);
            }
        }
    }

    @Override
    public void close() {
        context.unregisterReceiver(this);
    }

    @Override
    public void cleanCollections() {
        this.networkReachable.clear();
    }

    private enum NetworkStatus {
        NOT_REACHABLE("none"),
        WIFI("wifi"),
        WAN("wan");

        private String name;

        NetworkStatus(String name) {
            this.name = name;
        }

        String getName() {
            return this.name;
        }
    }

}
