package io.embrace.android.embracesdk;

import android.content.Context;
import android.content.res.Resources;

import com.fernandocejas.arrow.optional.Optional;
import com.google.gson.Gson;
import com.google.gson.annotations.SerializedName;

import java.util.Collections;
import java.util.List;

final class LocalConfig {

    /**
     * Build info app id name.
     */
    private static final String BUILD_INFO_APP_ID = "emb_app_id";

    /**
     * Build info sdk config id name.
     */
    private static final String BUILD_INFO_SDK_CONFIG = "emb_sdk_config";

    /**
     * The Embrace app ID. This is used to identify the app within the database.
     */
    private final String appId;

    /**
     * The Embrace sdk configurations. This is used to setup configurations.
     */
    private final SdkConfigs configs;

    private LocalConfig(String appId, SdkConfigs configs) {
        this.appId = appId;
        this.configs = Optional.fromNullable(configs).orNull();
    }

    String getAppId() {
        return appId;
    }

    public SdkConfigs getConfigurations() {
        return configs;
    }

    /**
     * Loads the build information from resources provided by the config file packaged within the application by Gradle at
     * build-time.
     *
     * @return the local configuration
     */

    static LocalConfig fromResources(Context context) {
        try {

            String appId = context.getResources().getString(getResourcesIdentifier(context, BUILD_INFO_APP_ID, "string"));

            String sdkConfigJson = null;
            int sdkConfigJsonId = getResourcesIdentifier(context, BUILD_INFO_SDK_CONFIG, "string");
            if (sdkConfigJsonId != 0) {
                sdkConfigJson = context.getResources().getString(sdkConfigJsonId);
            }

            return buildConfig(appId, sdkConfigJson);
        } catch (Exception ex) {
            throw new RuntimeException("Failed to load local config from resources.", ex);
        }
    }

    static LocalConfig buildConfig(String appId, String sdkConfigs) {
        if (appId == null || appId.isEmpty()) {
            throw new IllegalArgumentException("Embrace AppId cannot be null or empty.");
        }

        SdkConfigs configs;
        if (sdkConfigs != null && !sdkConfigs.isEmpty()) {
            configs = new Gson().fromJson(sdkConfigs, SdkConfigs.class);
        } else {
            configs = new SdkConfigs();
        }

        return new LocalConfig(appId, configs);
    }

    /**
     * Given a config property name and a config property type, retrieves the embrace config resource id.
     *
     * @param context
     * @param configProperty
     * @param type
     * @return
     */
    private static int getResourcesIdentifier(Context context, String configProperty, String type) {
        Resources resources = context.getResources();
        return resources.getIdentifier(configProperty, type, context.getPackageName());
    }

    static class SdkConfigs {

        /**
         * Base URL settings
         */
        @SerializedName("base_urls")
        private final BaseUrls baseUrls;

        /**
         * Crash handler settings
         */
        @SerializedName("crash_handler")
        private final CrashHandler crashHandler;

        /**
         * Startup moment settings
         */
        @SerializedName("startup_moment")
        private final StartupMoment startupMoment;

        /**
         * Networking moment settings
         */
        @SerializedName("networking")
        private final Networking networking;

        /**
         * Session config settings
         */
        @SerializedName("session")
        private final SessionConfig sessionConfig;

        /**
         * Taps
         */
        @SerializedName("taps")
        private final Taps taps;

        SdkConfigs() {
            this(null, null, null, null, null, null);
        }

        SdkConfigs(BaseUrls baseUrls, CrashHandler crashHandler, StartupMoment startupMoment, Networking networking, SessionConfig sessionConfig, Taps taps) {

            this.baseUrls = Optional.fromNullable(baseUrls).or(new BaseUrls());
            this.crashHandler = Optional.fromNullable(crashHandler).or(new CrashHandler());
            this.startupMoment = Optional.fromNullable(startupMoment).or(new StartupMoment());
            this.networking = Optional.fromNullable(networking).or(new Networking());
            this.sessionConfig = Optional.fromNullable(sessionConfig).or(new SessionConfig());
            this.taps = Optional.fromNullable(taps).or(new Taps());
        }

        BaseUrls getBaseUrls() {
            return baseUrls;
        }

        CrashHandler getCrashHandler() {
            return crashHandler;
        }

        StartupMoment getStartupMoment() {
            return startupMoment;
        }

        Networking getNetworking() {
            return networking;
        }

        SessionConfig getSessionConfig() {
            return sessionConfig;
        }

        Taps getTaps() {
            return taps;
        }

        /**
         * Represents the base URLs element specified in the Embrace config file.
         */
        static class BaseUrls {
            static final String CONFIG_DEFAULT = "https://config.emb-api.com";
            static final String DATA_DEFAULT = "https://data.emb-api.com";
            static final String DATA_DEV_DEFAULT = "https://data-dev.emb-api.com";
            static final String IMAGES_DEFAULT = "https://images.emb-api.com";

            /**
             * Data base URL.
             */
            @SerializedName("data")
            private String data;

            /**
             * Data dev base URL.
             */
            @SerializedName("data_dev")
            private String dataDev;

            /**
             * Config base URL.
             */
            @SerializedName("config")
            private String config;

            /**
             * Images base URL.
             */
            @SerializedName("images")
            private String images;

            BaseUrls() {
                this(null, null, null, null);
            }

            BaseUrls(String config, String data, String dataDev, String images) {
                this.config = Optional.fromNullable(config).or(CONFIG_DEFAULT);
                this.data = Optional.fromNullable(data).or(DATA_DEFAULT);
                this.dataDev = Optional.fromNullable(dataDev).or(DATA_DEV_DEFAULT);
                this.images = Optional.fromNullable(images).or(IMAGES_DEFAULT);
            }

            String getConfig() {
                return config;
            }

            String getData() {
                return data;
            }

            String getDataDev() {
                return dataDev;
            }

            String getImages() {
                return images;
            }
        }

        /**
         * Represents the crash handler element specified in the Embrace config file.
         */
        static class CrashHandler {

            static final Boolean ENABLED_DEFAULT = true;

            /**
             * Control whether the Embrace SDK automatically attaches to the uncaught exception handler.
             */
            @SerializedName("enabled")
            private Boolean enabled;

            CrashHandler() {
                this(null);
            }

            CrashHandler(Boolean enabled) {
                this.enabled = Optional.fromNullable(enabled).or(ENABLED_DEFAULT);
            }

            Boolean getEnabled() {
                return enabled;
            }
        }

        /**
         * Represents the startup moment configuration element specified in the Embrace config file.
         */
        static class StartupMoment {

            static final Boolean AUTOMATICALLY_END_DEFAULT = true;
            static final Boolean TAKE_SCREENSHOT_DEFAULT = true;

            /**
             * Control whether the startup moment is automatically ended.
             */
            @SerializedName("automatically_end")
            private Boolean automaticallyEnd;

            /**
             * Control whether startup moment screenshots are taken.
             */
            @SerializedName("take_screenshot")
            private Boolean takeScreenshot;


            StartupMoment() {
                this(null, null);
            }

            StartupMoment(Boolean automaticallyEnd, Boolean takeScreenshot) {
                this.automaticallyEnd = Optional.fromNullable(automaticallyEnd).or(AUTOMATICALLY_END_DEFAULT);
                this.takeScreenshot = Optional.fromNullable(takeScreenshot).or(TAKE_SCREENSHOT_DEFAULT);
            }

            Boolean getAutomaticallyEnd() {
                return automaticallyEnd;
            }

            Boolean getTakeScreenshot() {
                return takeScreenshot;
            }
        }

        /**
         * Represents the networking configuration element specified in the Embrace config file.
         */
        static class Networking {

            /**
             * Sets the default name of the HTTP request header to extract trace ID from.
             */
            static final String CONFIG_TRACE_ID_HEADER_DEFAULT_VALUE = "x-emb-trace-id";

            /**
             * Capture request content length by default.
             */
            static final Boolean CAPTURE_REQUEST_CONTENT_LENGTH = true;

            /**
             * The Trace ID Header that can be used to trace a particular request.
             */
            @SerializedName("trace_id_header")
            private String traceIdHeader;

            /**
             * The default capture limit for the specified domains.
             */
            @SerializedName("default_capture_limit")
            private final Integer defaultCaptureLimit;

            /**
             * List of domains to be limited for tracking.
             */
            @SerializedName("domains")
            private final List<Domain> domains;

            /**
             * Control whether request size for native Android requests is captured.
             */
            @SerializedName("capture_request_content_length")
            private Boolean captureRequestContentLength;

            Networking() {
                this(null, null, null, null);
            }

            Networking(String traceIdHeader, Integer defaultCaptureLimit, List<Domain> domains, Boolean captureRequestContentLength) {
                this.traceIdHeader = Optional.fromNullable(traceIdHeader).or(CONFIG_TRACE_ID_HEADER_DEFAULT_VALUE);
                this.defaultCaptureLimit = defaultCaptureLimit;
                this.domains = domains;
                this.captureRequestContentLength = Optional.fromNullable(captureRequestContentLength).or(CAPTURE_REQUEST_CONTENT_LENGTH);
            }

            String getTraceIdHeader() {
                return traceIdHeader;
            }

            Boolean getCaptureRequestContentLength() {
                return captureRequestContentLength;
            }

            public Optional<Integer> getDefaultCaptureLimit() {
                return Optional.fromNullable(defaultCaptureLimit);
            }

            public List<Domain> getDomains() {
                return domains != null ? domains : Collections.emptyList();
            }

            /**
             * Represents each domain element specified in the Embrace config file.
             */
            class Domain {

                /**
                 * Url for the domain.
                 */
                @SerializedName("domain_name")
                private final String domain;

                /**
                 * Limit for the number of requests to be tracked.
                 */
                @SerializedName("domain_limit")
                private final Integer limit;

                Domain(String domain, Integer limit) {
                    this.domain = domain;
                    this.limit = limit;
                }

                public String getDomain() {
                    return domain;
                }

                public Integer getLimit() {
                    return limit;
                }
            }
        }

        /**
         * Represents the session configuration element specified in the Embrace config file.
         */
        static class SessionConfig {

            /**
             * Default minimum allowed end session time.
             */
            private static final int DEFAULT_MINIMUM_SESSION_SECONDS = 60;

            /**
             * Specify a maximum time before a session is allowed to exist before it is ended.
             */
            @SerializedName("max_session_seconds")
            private Integer maxSessionSeconds;

            SessionConfig() {
                this(null);
            }

            SessionConfig(Integer maxSessionSeconds) {
                this.maxSessionSeconds = maxSessionSeconds;
            }

            Optional<Integer> getMaxSessionSeconds() {
                if (maxSessionSeconds != null) {
                    if (maxSessionSeconds >= DEFAULT_MINIMUM_SESSION_SECONDS) {
                        return Optional.of(maxSessionSeconds);
                    } else {
                        EmbraceLogger.logWarning("Automatic end session disabled. Config max_session_seconds should be more than 60 seconds.");
                        return Optional.absent();
                    }
                } else {
                    return Optional.absent();
                }
            }
        }

        static class Taps {

            static final Boolean CAPTURE_COORDINATES_DEFAULT = true;

            /**
             * Control whether tap coordindates are captured.
             */
            @SerializedName("capture_coordinates")
            private Boolean captureCoordinates;

            Taps() {
                this(null);
            }

            Taps(Boolean captureCoordinates) {
                this.captureCoordinates = Optional.fromNullable(captureCoordinates).or(CAPTURE_COORDINATES_DEFAULT);
            }

            Boolean getCaptureCoordinates() {
                return captureCoordinates;
            }
        }
    }
}
