package io.embrace.android.embracesdk;

import android.content.Context;

import com.fernandocejas.arrow.optional.Optional;
import com.google.gson.Gson;
import com.google.gson.annotations.SerializedName;
import com.google.gson.stream.JsonReader;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Collections;
import java.util.List;

final class LocalConfig {
    /**
     * Build info json file name.
     */
    private static final String FILE_LOCAL_CONFIG = "embrace-config.json";

    /**
     * The Embrace app ID. This is used to identify the app within the database.
     */
    @SerializedName("app_id")
    private final String appId;

    /**
     * Base URL settings
     */
    @SerializedName("base_urls")
    private final LocalConfig.BaseUrls baseUrls;

    /**
     * Crash handler settings
     */
    @SerializedName("crash_handler")
    private final LocalConfig.CrashHandler crashHandler;

    /**
     * Startup moment settings
     */
    @SerializedName("startup_moment")
    private final LocalConfig.StartupMoment startupMoment;

    /**
     * Networking moment settings
     */
    @SerializedName("networking")
    private final LocalConfig.Networking networking;

    /**
     * Session config settings
     */
    @SerializedName("session")
    private final LocalConfig.SessionConfig sessionConfig;

    /*
     * Taps
     */
    @SerializedName("taps")
    private final LocalConfig.Taps taps;

    LocalConfig() {
        this(null, null, null, null, null, null, null);
    }

    private LocalConfig(String appId, BaseUrls baseUrls, CrashHandler crashHandler, StartupMoment startupMoment, Networking networking, SessionConfig sessionConfig, Taps taps) {
        this.appId = Optional.fromNullable(appId).orNull();

        this.baseUrls = Optional.fromNullable(baseUrls).or(new BaseUrls());
        this.crashHandler = Optional.fromNullable(crashHandler).or(new CrashHandler());
        this.startupMoment = Optional.fromNullable(startupMoment).or(new StartupMoment());
        this.networking = Optional.fromNullable(networking).or(new Networking());
        this.sessionConfig = Optional.fromNullable(sessionConfig).or(new SessionConfig());
        this.taps = Optional.fromNullable(taps).or(new Taps());
    }

    BaseUrls getBaseUrls() {
        return baseUrls;
    }

    CrashHandler getCrashHandler() {
        return crashHandler;
    }

    StartupMoment getStartupMoment() {
        return startupMoment;
    }

    Networking getNetworking() {
        return networking;
    }

    SessionConfig getSessionConfig() {
        return sessionConfig;
    }

    String getAppId() {
        return appId;
    }

    Taps getTaps() {
        return taps;
    }

    /**
     * Loads the build information from a json file packaged within the application by Gradle at
     * build-time.
     *
     * @return the build information
     */

    static LocalConfig fromFile(Context context) {
        try (InputStream inputStream = context.getApplicationContext().getAssets().open(FILE_LOCAL_CONFIG)
        ) {
            return LocalConfig.fromInputStream(inputStream);
        } catch (IOException e) {
            EmbraceLogger.logDebug("Embrace config file " + FILE_LOCAL_CONFIG + " does not exist");
        }
        return new LocalConfig();
    }

    static LocalConfig fromInputStream(InputStream inputStream) {
        try (
                InputStreamReader inputStreamReader = new InputStreamReader(inputStream);
                JsonReader reader = new JsonReader(inputStreamReader)
        ) {
            // fromJson will return null when reading from empty files
            Optional<LocalConfig> config = Optional.fromNullable(
                    new Gson().fromJson(reader, LocalConfig.class)
            );
            if (config.isPresent()) {
                EmbraceLogger.logInfo("Read configuration from " + FILE_LOCAL_CONFIG);
            }
            return config.or(new LocalConfig());
        } catch (Exception ex) {
            EmbraceLogger.logWarning("Failed to read Embrace config file " + FILE_LOCAL_CONFIG, ex);
        }
        return new LocalConfig();
    }

    /**
     * Represents the base URLs element specified in the Embrace config file.
     */
    static class BaseUrls {
        static final String CONFIG_DEFAULT = "https://config.emb-api.com";
        static final String DATA_DEFAULT = "https://data.emb-api.com";
        static final String DATA_DEV_DEFAULT = "https://data-dev.emb-api.com";
        static final String IMAGES_DEFAULT = "https://images.emb-api.com";

        /**
         * Data base URL.
         */
        @SerializedName("data")
        private String data;

        /**
         * Data dev base URL.
         */
        @SerializedName("data_dev")
        private String dataDev;

        /**
         * Config base URL.
         */
        @SerializedName("config")
        private String config;

        /**
         * Images base URL.
         */
        @SerializedName("images")
        private String images;

        BaseUrls() {
            this(null, null, null, null);
        }

        BaseUrls(String config, String data, String dataDev, String images) {
            this.config = Optional.fromNullable(config).or(CONFIG_DEFAULT);
            this.data = Optional.fromNullable(data).or(DATA_DEFAULT);
            this.dataDev = Optional.fromNullable(dataDev).or(DATA_DEV_DEFAULT);
            this.images = Optional.fromNullable(images).or(IMAGES_DEFAULT);
        }

        String getConfig() {
            return config;
        }

        String getData() {
            return data;
        }

        String getDataDev() {
            return dataDev;
        }

        String getImages() {
            return images;
        }
    }

    /**
     * Represents the crash handler element specified in the Embrace config file.
     */
    static class CrashHandler {

        static final Boolean ENABLED_DEFAULT = true;

        /**
         * Control whether the Embrace SDK automatically attaches to the uncaught exception handler.
         */
        @SerializedName("enabled")
        private Boolean enabled;

        CrashHandler() {
            this(null);
        }

        CrashHandler(Boolean enabled) {
            this.enabled = Optional.fromNullable(enabled).or(ENABLED_DEFAULT);
        }

        Boolean getEnabled() {
            return enabled;
        }
    }

    /**
     * Represents the startup moment configuration element specified in the Embrace config file.
     */
    static class StartupMoment {

        static final Boolean AUTOMATICALLY_END_DEFAULT = true;
        static final Boolean TAKE_SCREENSHOT_DEFAULT = true;

        /**
         * Control whether the startup moment is automatically ended.
         */
        @SerializedName("automatically_end")
        private Boolean automaticallyEnd;

        /**
         * Control whether startup moment screenshots are taken.
         */
        @SerializedName("take_screenshot")
        private Boolean takeScreenshot;


        StartupMoment() {
            this(null, null);
        }

        StartupMoment(Boolean automaticallyEnd, Boolean takeScreenshot) {
            this.automaticallyEnd = Optional.fromNullable(automaticallyEnd).or(AUTOMATICALLY_END_DEFAULT);
            this.takeScreenshot = Optional.fromNullable(takeScreenshot).or(TAKE_SCREENSHOT_DEFAULT);
        }

        Boolean getAutomaticallyEnd() {
            return automaticallyEnd;
        }

        Boolean getTakeScreenshot() {
            return takeScreenshot;
        }
    }

    /**
     * Represents the networking configuration element specified in the Embrace config file.
     */
    static class Networking {

        /**
         * Sets the default name of the HTTP request header to extract trace ID from.
         */
        static final String CONFIG_TRACE_ID_HEADER_DEFAULT_VALUE = "x-emb-trace-id";

        /**
         * The Trace ID Header that can be used to trace a particular request.
         */
        @SerializedName("trace_id_header")
        private String traceIdHeader;

        /**
         * The default capture limit for the specified domains.
         */
        @SerializedName("default_capture_limit")
        private final Integer defaultCaptureLimit;

        /**
         * List of domains to be limited for tracking.
         */
        @SerializedName("domains")
        private final List<Domain> domains;

        Networking() {
            this(null, null, null);
        }

        Networking(String traceIdHeader, Integer defaultCaptureLimit, List<Domain> domains) {
            this.traceIdHeader = Optional.fromNullable(traceIdHeader).or(CONFIG_TRACE_ID_HEADER_DEFAULT_VALUE);
            this.defaultCaptureLimit = defaultCaptureLimit;
            this.domains = domains;
        }

        String getTraceIdHeader() {
            return traceIdHeader;
        }

        public Optional<Integer> getDefaultCaptureLimit() {
            return Optional.fromNullable(defaultCaptureLimit);
        }

        public List<Domain> getDomains() {
            return domains != null ? domains : Collections.emptyList();
        }

        /**
         * Represents each domain element specified in the Embrace config file.
         */
        class Domain {

            /**
             * Url for the domain.
             */
            @SerializedName("domain_name")
            private final String domain;

            /**
             * Limit for the number of requests to be tracked.
             */
            @SerializedName("domain_limit")
            private final Integer limit;

            Domain(String domain, Integer limit) {
                this.domain = domain;
                this.limit = limit;
            }

            public String getDomain() {
                return domain;
            }

            public Integer getLimit() {
                return limit;
            }
        }
    }

    /**
     * Represents the session configuration element specified in the Embrace config file.
     */
    class SessionConfig {

        /**
         * Specify a maximum time before a session is allowed to exist before it is ended.
         */
        @SerializedName("max_session_seconds")
        private Integer maxSessionSeconds;

        SessionConfig() {
            this(null);
        }

        SessionConfig(Integer maxSessionSeconds) {
            this.maxSessionSeconds = maxSessionSeconds;
        }

        Optional<Integer> getMaxSessionSeconds() {
            return Optional.fromNullable(maxSessionSeconds);
        }
    }

    static class Taps {

        static final Boolean CAPTURE_COORDINATES_DEFAULT = true;

        /**
         * Control whether tap coordindates are captured.
         */
        @SerializedName("capture_coordinates")
        private Boolean captureCoordinates;

        Taps() {
            this(null);
        }

        Taps(Boolean captureCoordinates) {
            this.captureCoordinates = Optional.fromNullable(captureCoordinates).or(CAPTURE_COORDINATES_DEFAULT);
        }

        Boolean getCaptureCoordinates() {
            return captureCoordinates;
        }
    }
}
