package io.embrace.android.embracesdk;

import android.util.Base64;

import com.google.gson.Gson;
import com.google.gson.annotations.SerializedName;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Describes a Crash (uncaught {@link Throwable} by a unique ID, and a list of exceptions (causes).
 */
final class Crash {
    @SerializedName("id")
    private final String id;

    @SerializedName("ex")
    private final List<ExceptionInfo> exceptions;

    @SerializedName("rep_js")
    private final List<String> jsExceptions;

    private Crash(String id, List<ExceptionInfo> exceptions, List<String> jsExceptions) {
        this.id = id;
        this.exceptions = exceptions;
        this.jsExceptions = jsExceptions;
    }

    /**
     * Creates a crash from a {@link Throwable}. Extracts each cause and converts it to
     * {@link ExceptionInfo}. Optionally includes a {@link JsException}.
     *
     * @param throwable   the throwable to parse
     * @param jsException an optional JS exception that is associated with the crash
     * @return a crash
     */
    static Crash ofThrowable(Throwable throwable, JsException jsException) {
        List<String> jsExceptions = null;
        if (jsException != null) {
            try {
                byte[] jsonException = new Gson().toJson(jsException, jsException.getClass()).getBytes();
                String encodedString = Base64.encodeToString(jsonException, Base64.NO_WRAP);
                jsExceptions = Collections.singletonList(encodedString);
            } catch (Exception ex) {
                EmbraceLogger.logError("Failed to parse javascript exception", ex, true);
            }
        }

        return new Crash(Uuid.getEmbUuid(), exceptionInfo(throwable), jsExceptions);
    }

    private static List<ExceptionInfo> exceptionInfo(Throwable ex) {
        List<ExceptionInfo> result = new ArrayList<>();
        Throwable throwable = ex;
        while (throwable != null && !throwable.equals(throwable.getCause())) {
            ExceptionInfo exceptionInfo = ExceptionInfo.ofThrowable(throwable);
            result.add(0, exceptionInfo);
            throwable = throwable.getCause();
        }
        return result;
    }

    String getCause() {
        return exceptions != null && !exceptions.isEmpty() ? this.exceptions.get(0).getName() : null;
    }
}
