package io.embrace.android.embracesdk;

import com.google.gson.annotations.SerializedName;

/**
 * Version and environment information relating to the app which has embedded the Embrace SDK.
 */
final class AppInfo {

    /**
     * The version of the app which has embedded the Embrace SDK.
     */
    @SerializedName("v")
    private final String appVersion;

    /**
     * The framework used by the app.
     */
    @SerializedName("f")
    private final Integer appFramework;

    /**
     * A unique ID for the build which is generated at build time. This is written to a JSON file in
     * the build directory and read by {@link BuildInfo}.
     */
    @SerializedName("bi")
    private final String buildId;

    /**
     * The build type name. This is written to a JSON file in the build directory and read by
     * {@link BuildInfo}.
     */
    @SerializedName("bt")
    private final String buildType;

    /**
     * The flavor name. This is written to a JSON file in the build directory and read by
     * {@link BuildInfo}.
     */
    @SerializedName("fl")
    private final String buildFlavor;

    /**
     * The name of the environment, i.e. dev or prod, determined by whether this is a debug build.
     */
    @SerializedName("e")
    private final String environment;

    /**
     * Whether the app was updated since the previous launch.
     */
    @SerializedName("vu")
    private final boolean appUpdated;

    /**
     * Whether the app was updated since the previous launch.
     */
    @SerializedName("vul")
    private final boolean appUpdatedThisLaunch;

    /**
     * The app bundle version.
     */
    @SerializedName("bv")
    private final String bundleVersion;

    /**
     * Whether the OS was updated since the last launch.
     */
    @SerializedName("ou")
    private final boolean osUpdated;

    /**
     * Whether the OS was updated since the last launch.
     */
    @SerializedName("oul")
    private final boolean osUpdatedThisLaunch;

    /**
     * The version number of the Embrace SDK.
     */
    @SerializedName("sdk")
    private final String sdkVersion;

    /**
     * The simple version number of the Embrace SDK.
     */
    @SerializedName("sdc")
    private final String sdkSimpleVersion;

    /**
     * The react native bundle hashed.
     */
    @SerializedName("rn")
    private final String reactNativeBundlePath;

    /**
     * The java script patch number.
     */
    @SerializedName("jsp")
    private final String javaScriptPatchNumber;

    /**
     * The react native version number.
     */
    @SerializedName("rnv")
    private final String reactNativeVersion;

    private AppInfo(Builder builder) {
        appFramework = builder.appFramework;
        appVersion = builder.appVersion;
        buildId = builder.buildId;
        buildType = builder.buildType;
        buildFlavor = builder.buildFlavor;
        environment = builder.environment;
        appUpdated = builder.appUpdated;
        appUpdatedThisLaunch = builder.appUpdatedThisLaunch;
        bundleVersion = builder.bundleVersion;
        osUpdated = builder.osUpdated;
        osUpdatedThisLaunch = builder.osUpdatedThisLaunch;
        sdkVersion = builder.sdkVersion;
        sdkSimpleVersion = builder.sdkSimpleVersion;
        reactNativeBundlePath = builder.reactNativeBundlePath;
        javaScriptPatchNumber = builder.javaScriptPatch;
        reactNativeVersion = builder.reactNativeVersion;
    }

    static Builder newBuilder() {
        return new Builder();
    }


    static final class Builder {
        private Integer appFramework;
        private String appVersion;
        private String buildId;
        private String buildType;
        private String buildFlavor;
        private String environment;
        private boolean appUpdated;
        private boolean appUpdatedThisLaunch;
        private String bundleVersion;
        private boolean osUpdated;
        private boolean osUpdatedThisLaunch;
        private String sdkVersion;
        private String sdkSimpleVersion;
        private String reactNativeBundlePath;
        private String javaScriptPatch;
        private String reactNativeVersion;

        private Builder() {
        }

        Builder withAppFramework(Embrace.AppFramework appFramework) {
            this.appFramework = appFramework.getValue();
            return this;
        }

        Builder withAppVersion(String appVersion) {
            this.appVersion = appVersion;
            return this;
        }

        Builder withBuildId(String buildId) {
            this.buildId = buildId;
            return this;
        }

        Builder withBuildType(String buildType) {
            this.buildType = buildType;
            return this;
        }

        Builder withBuildFlavor(String buildFlavor) {
            this.buildFlavor = buildFlavor;
            return this;
        }

        Builder withEnvironment(String environment) {
            this.environment = environment;
            return this;
        }

        Builder withAppUpdated(boolean appUpdated) {
            this.appUpdated = appUpdated;
            return this;
        }

        Builder withAppUpdatedThisLaunch(boolean appUpdatedThisLaunch) {
            this.appUpdatedThisLaunch = appUpdatedThisLaunch;
            return this;
        }

        Builder withBundleVersion(String bundleVersion) {
            this.bundleVersion = bundleVersion;
            return this;
        }

        Builder withOsUpdated(boolean osUpdated) {
            this.osUpdated = osUpdated;
            return this;
        }

        Builder withOsUpdatedThisLaunch(boolean osUpdatedThisLaunch) {
            this.osUpdatedThisLaunch = osUpdatedThisLaunch;
            return this;
        }

        Builder withSdkVersion(String sdkVersion) {
            this.sdkVersion = sdkVersion;
            return this;
        }

        Builder withSdkSimpleVersion(String sdkSimpleVersion) {
            this.sdkSimpleVersion = sdkSimpleVersion;
            return this;
        }

        Builder withReactNativeBundlePath(String reactNativeBundlePath) {
            this.reactNativeBundlePath = reactNativeBundlePath;
            return this;
        }

        Builder withJavaScriptPatchNumber(String number) {
            this.javaScriptPatch = number;
            return this;
        }

        Builder withReactNativeVersion(String version) {
            this.reactNativeVersion = version;
            return this;
        }

        AppInfo build() {
            return new AppInfo(this);
        }
    }
}
