package io.embrace.android.embracesdk;

import java.util.AbstractMap.SimpleEntry;
import java.util.Map;
import java.util.Map.Entry;

import java9.util.stream.Collectors;
import java9.util.stream.StreamSupport;

/**
 * Utility to for sanitizing user-supplied properties.
 */
final class PropertyUtils {

    static final int MAX_PROPERTY_SIZE = 10;

    /**
     * This method will normalize the map by applying the following rules:
     * <p>
     * - Null key registries will be discarded.
     * - Null value registries will be renamed to null as a String.
     * - Cap the properties map to a maximum of {@link PropertyUtils#MAX_PROPERTY_SIZE} properties.
     *
     * @param properties properties to be normalized.
     * @return a normalized Map of the provided properties.
     */
    static Map<String, Object> sanitizeProperties(Map<String, Object> properties) {

        if (properties.size() > MAX_PROPERTY_SIZE) {
            EmbraceLogger.logWarning("The maximum number of properties is " + MAX_PROPERTY_SIZE + ", the rest will be ignored.");
        }

        return StreamSupport.stream(properties.entrySet())
                .filter(entry -> entry.getKey() != null)
                .limit(MAX_PROPERTY_SIZE)
                .map(PropertyUtils::mapNullValue)
                .collect(Collectors.toMap(Entry::getKey, Entry::getValue));
    }

    private static Entry<String, Object> mapNullValue(Entry<String, Object> entry) {

        return new SimpleEntry<>(entry.getKey(), entry.getValue() == null ? "null" : entry.getValue());
    }

    private PropertyUtils() {
        // Restricted constructor
    }
}
