package io.embrace.android.embracesdk;

import android.content.Context;

import com.fernandocejas.arrow.optional.Optional;
import com.google.gson.Gson;
import com.google.gson.annotations.SerializedName;
import com.google.gson.stream.JsonReader;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

final class LocalConfig {
    /**
     * Build Info json file name to be retrieved from the assets.
     */
    private static final String FILE_LOCAL_CONFIG = "embrace-config.json";

    /**
     * Base URL settings
     */
    @SerializedName("base_urls")
    private final LocalConfig.BaseUrls baseUrls;

    /**
     * Crash handler settings
     */
    @SerializedName("crash_handler")
    private final LocalConfig.CrashHandler crashHandler;

    /**
     * Startup moment settings
     */
    @SerializedName("startup_moment")
    private final LocalConfig.StartupMoment startupMoment;

    /**
     * Networking moment settings
     */
    @SerializedName("networking")
    private final LocalConfig.Networking networking;

    LocalConfig() {
        this(null, null, null, null);
    }

    private LocalConfig(BaseUrls baseUrls, CrashHandler crashHandler, StartupMoment startupMoment, Networking networking) {
        this.baseUrls = Optional.fromNullable(baseUrls).or(new BaseUrls());
        this.crashHandler = Optional.fromNullable(crashHandler).or(new CrashHandler());
        this.startupMoment = Optional.fromNullable(startupMoment).or(new StartupMoment());
        this.networking = Optional.fromNullable(networking).or(new Networking());
    }

    BaseUrls getBaseUrls() {
        return baseUrls;
    }

    CrashHandler getCrashHandler() {
        return crashHandler;
    }

    StartupMoment getStartupMoment() {
        return startupMoment;
    }

    Networking getNetworking() {
        return networking;
    }

    /**
     * Loads the build information from a JSON file packaged within the application by Gradle at
     * build-time.
     *
     * @return the build information
     */

    static LocalConfig fromFile(Context context) {
        try (InputStream inputStream = context.getApplicationContext().getAssets().open(FILE_LOCAL_CONFIG)
        ) {
            return LocalConfig.fromInputStream(inputStream);
        } catch (IOException e) {
            EmbraceLogger.logDebug("Embrace config file " + FILE_LOCAL_CONFIG + " does not exist");
        }
        return new LocalConfig();
    }

    static LocalConfig fromInputStream(InputStream inputStream) {
        try (
                InputStreamReader inputStreamReader = new InputStreamReader(inputStream);
                JsonReader reader = new JsonReader(inputStreamReader)
        ) {
            // fromJson will return null when reading from empty files
            Optional<LocalConfig> config = Optional.fromNullable(
                    new Gson().fromJson(reader, LocalConfig.class)
            );
            if (config.isPresent()) {
                EmbraceLogger.logInfo("Read configuration from " + FILE_LOCAL_CONFIG);
            }
            return config.or(new LocalConfig());
        } catch (Exception ex) {
            EmbraceLogger.logWarning("Failed to read Embrace config file " + FILE_LOCAL_CONFIG, ex);
        }
        return new LocalConfig();
    }

    /**
     * Represents the base URLs element specified in the Embrace config file.
     */
    static class BaseUrls {
        static final String CONFIG_DEFAULT = "https://config.emb-api.com";
        static final String DATA_DEFAULT = "https://data.emb-api.com";
        static final String DATA_DEV_DEFAULT = "https://data-dev.emb-api.com";
        static final String IMAGES_DEFAULT = "https://images.emb-api.com";

        /**
         * Data base URL.
         */
        @SerializedName("data")
        private String data;

        /**
         * Data dev base URL.
         */
        @SerializedName("data_dev")
        private String dataDev;

        /**
         * Config base URL.
         */
        @SerializedName("config")
        private String config;

        /**
         * Images base URL.
         */
        @SerializedName("images")
        private String images;

        BaseUrls() {
            this(null, null, null, null);
        }

        BaseUrls(String config, String data, String dataDev, String images) {
            this.config = Optional.fromNullable(config).or(CONFIG_DEFAULT);
            this.data = Optional.fromNullable(data).or(DATA_DEFAULT);
            this.dataDev = Optional.fromNullable(dataDev).or(DATA_DEV_DEFAULT);
            this.images = Optional.fromNullable(images).or(IMAGES_DEFAULT);
        }

        String getConfig() {
            return config;
        }

        String getData() {
            return data;
        }

        String getDataDev() {
            return dataDev;
        }

        String getImages() {
            return images;
        }
    }

    /**
     * Represents the crash handler element specified in the Embrace config file.
     */
    static class CrashHandler {

        static final Boolean ENABLED_DEFAULT = true;

        /**
         * Control whether the Embrace SDK automatically attaches to the uncaught exception handler.
         */
        @SerializedName("enabled")
        private Boolean enabled;

        CrashHandler() {
            this(null);
        }

        CrashHandler(Boolean enabled) {
            this.enabled = Optional.fromNullable(enabled).or(ENABLED_DEFAULT);
        }

        Boolean getEnabled() {
            return enabled;
        }
    }

    /**
     * Represents the startup moment configuration element specified in the Embrace config file.
     */
    static class StartupMoment {

        static final Boolean AUTOMATICALLY_END_DEFAULT = true;

        /**
         * Control whether the startup moment is automatically ended.
         */
        @SerializedName("automatically_end")
        private Boolean automaticallyEnd;

        StartupMoment() {
            this(null);
        }

        StartupMoment(Boolean automaticallyEnd) {
            this.automaticallyEnd = Optional.fromNullable(automaticallyEnd).or(AUTOMATICALLY_END_DEFAULT);
        }

        Boolean getAutomaticallyEnd() {
            return automaticallyEnd;
        }
    }

    /**
     * Represents the networking configuration element specified in the Embrace config file.
     */
    static class Networking {

        /**
         * Sets name of the HTTP request header to extract trace ID from.
         */
        private static final String CONFIG_TRACE_ID_HEADER_DEFAULT_VALUE = "x-emb-trace-id";

        /**
         * The Trace ID Header that can be used to trace a particular request.
         */
        @SerializedName("trace_id_header")
        private String traceIdHeader;

        Networking() {
            this(null);
        }

        Networking(String traceIdHeader) {
            this.traceIdHeader = Optional.fromNullable(traceIdHeader).or(CONFIG_TRACE_ID_HEADER_DEFAULT_VALUE);
        }

        String getTraceIdHeader() {
            return traceIdHeader;
        }
    }
}
