package io.embrace.android.embracesdk;

import com.fernandocejas.arrow.checks.Preconditions;
import com.fernandocejas.arrow.optional.Optional;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import java9.util.J8Arrays;
import java9.util.stream.Collectors;

/**
 * Logs network calls according to defined limits per domain.
 * <p>
 * Limits can be defined either in server-side configuration or within the gradle configuration.
 * A limit of 0 disables logging for the domain. All network calls are captured up to the limit,
 * and the number of calls is also captured if the limit is exceeded.
 */
class EmbraceNetworkLoggingService implements NetworkLoggingService, MemoryCleanerListener {

    public static final String DNS_PATTERN = "([a-zA-Z0-9]([a-zA-Z0-9\\-]{0,63}[a-zA-Z0-9])?)(\\.[a-zA-Z]{1,63})(\\.[a-zA-Z]{1,2})?$";
    public static final String IPV4_PATTERN = "^((25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.){3}(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$";
    public static final String IPV6_PATTERN = "(([a-fA-F0-9]{1,4}|):){1,7}([a-fA-F0-9]{1,4}|:)";

    private static final int DEFAULT_NETWORK_CALL_LIMIT = 1000;

    private final ConnectionClassService connectionClassService;

    private final ConfigService configService;

    /**
     * Network calls per domain.
     */
    private final NavigableMap<Long, NetworkCallV2> networkCalls;

    private final Map<String, DomainSettings> domainSettings;

    private final BuildInfo buildInfo;

    public EmbraceNetworkLoggingService(
            ConnectionClassService connectionClassService,
            ConfigService configService,
            BuildInfo buildInfo,
            MemoryCleanerService memoryCleanerService) {

        this.connectionClassService = Preconditions.checkNotNull(
                connectionClassService,
                "connectionClassService must not be null");
        this.configService = Preconditions.checkNotNull(
                configService,
                "configService must not be null");
        this.buildInfo = Preconditions.checkNotNull(
                buildInfo,
                "buildInfo must not be null");
        this.networkCalls = new ConcurrentSkipListMap<>();
        this.domainSettings = new ConcurrentHashMap<>();
        Preconditions.checkNotNull(memoryCleanerService).addListener(this);
    }


    @Override
    public List<NetworkCallV2> getNetworkCalls(long startTime, long endTime) {
        return new ArrayList<>(this.networkCalls.subMap(startTime, endTime).values());
    }

    @Override
    public NetworkSessionV2 getNetworkSession(long startTime, long lastKnownTime) {
        List<NetworkCallV2> calls = getNetworkCalls(startTime, lastKnownTime);

        // Limit defined by the application at build-time
        Optional<Integer> gradleLimit = Optional.absent();
        Optional<BuildInfo.Network> networkConfig = buildInfo.getNetworkConfig();
        if (networkConfig.isPresent()) {
            gradleLimit = networkConfig.get().getDefaultCaptureLimit();
        }
        int defaultLimit = configService.getConfig()
                .getDefaultNetworkCallLimit()
                .or(gradleLimit)
                .or(DEFAULT_NETWORK_CALL_LIMIT);

        Map<String, NetworkSessionV2.DomainCount> callsPerDomain = new HashMap<>();
        List<NetworkCallV2> networkCalls = new ArrayList<>();

        int ipAddressCount = 0;
        for (NetworkCallV2 call : calls) {
            // Get the domain, if it can be successfully parsed
            Optional<String> domain = getDomain(call.getUrl());
            if (!domain.isPresent() || isIpAddress(domain.get())) {
                if (ipAddressCount < defaultLimit) {
                    ipAddressCount++;
                }
            } else {
                DomainSettings settings = this.domainSettings.get(domain.get());
                if (settings == null) {
                    networkCalls.add(call);
                } else {
                    String suffix = settings.getSuffix();
                    int limit = settings.getLimit();
                    NetworkSessionV2.DomainCount count = callsPerDomain.get(suffix);
                    if (count == null) {
                        count = new NetworkSessionV2.DomainCount(0, limit);
                    }
                    // Exclude if the network call exceeds the limit
                    if (count.getRequestCount() < limit) {
                        networkCalls.add(call);
                    }
                    // Track the number of calls for each domain (or configured suffix)
                    callsPerDomain.put(
                            suffix,
                            new NetworkSessionV2.DomainCount(count.getRequestCount() + 1, limit));
                }
            }
        }
        Map<String, NetworkSessionV2.DomainCount> overLimit = new HashMap<>();
        for (Map.Entry<String, NetworkSessionV2.DomainCount> entry : callsPerDomain.entrySet()) {
            NetworkSessionV2.DomainCount value = entry.getValue();
            if (value.getRequestCount() > value.getCaptureLimit()) {
                overLimit.put(entry.getKey(), value);
            }
        }
        return new NetworkSessionV2(networkCalls, overLimit);
    }

    @Override
    public void logNetworkCall(
            String url,
            String httpMethod,
            Integer statusCode,
            long startTime,
            long endTime,
            Long bytesSent,
            Long bytesReceived,
            String traceId) {

        long duration = Math.max(endTime - startTime, 0);

        traceId = EmbraceNetworkUtils.getValidTraceId(traceId);

        NetworkCallV2 networkCall = NetworkCallV2.newBuilder()
                .withUrl(stripUrl(url))
                .withHttpMethod(httpMethod)
                .withStartTime(startTime)
                .withDuration(duration)
                .withEndTime(endTime)
                .withBytesSent(bytesSent)
                .withBytesReceived(bytesReceived)
                .withResponseCode(statusCode)
                .withTraceId(traceId)
                .build();
        networkCalls.put(startTime, networkCall);
        connectionClassService.logBandwidth(bytesReceived, duration);
        storeSettings(url);
    }

    @Override
    public void logNetworkError(
            String url,
            String httpMethod,
            long startTime,
            long endTime,
            String errorType,
            String errorMessage,
            String traceId) {

        traceId = EmbraceNetworkUtils.getValidTraceId(traceId);

        NetworkCallV2 networkCall = NetworkCallV2.newBuilder()
                .withUrl(stripUrl(url))
                .withHttpMethod(httpMethod)
                .withStartTime(startTime)
                .withEndTime(endTime)
                .withDuration(Math.max(endTime - startTime, 0))
                .withErrorType(errorType)
                .withTraceId(traceId)
                .build();

        networkCalls.put(startTime, networkCall);
        storeSettings(url);
    }

    private void storeSettings(String url) {
        try {
            Map<String, Integer> remoteLimits = configService.getConfig().getNetworkCallLimitsPerDomain();
            Map<String, Integer> mergedLimits = new HashMap<>();
            if (buildInfo.getNetworkConfig().isPresent()) {
                for (BuildInfo.Network.Domain domain : buildInfo.getNetworkConfig().get().getDomains()) {
                    mergedLimits.put(domain.getDomain(), domain.getLimit());
                }
            }
            mergedLimits.putAll(remoteLimits);

            Optional<String> domain = getDomain(url);
            if (!domain.isPresent()) {
                return;
            }
            String domainString = domain.get();
            if (domainSettings.containsKey(domainString)) {
                return;
            }
            for (Map.Entry<String, Integer> entry : mergedLimits.entrySet()) {
                if (domainString.endsWith(entry.getKey())) {
                    domainSettings.put(domainString, new DomainSettings(entry.getValue(), entry.getKey()));
                    return;
                }
            }
            int defaultLimit = configService.getConfig()
                    .getDefaultNetworkCallLimit()
                    .or(DEFAULT_NETWORK_CALL_LIMIT);
            domainSettings.put(domainString, new DomainSettings(defaultLimit, trimDomain(domainString)));
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to determine limits for URL: " + url, ex);
        }
    }


    /**
     * Strips off the query string and hash fragment from a URL.
     *
     * @param url the URL to parse
     * @return the URL with the hash fragment and query string parameters removed
     */
    private static String stripUrl(String url) {
        if (url == null) {
            return null;
        }

        int pathPos = url.lastIndexOf('/');
        String suffix = pathPos < 0 ? url : url.substring(pathPos);

        int queryPos = suffix.indexOf('?');
        int fragmentPos = suffix.indexOf('#');
        int terminalPos = Math.min(queryPos < 0 ? Integer.MAX_VALUE : queryPos,
                fragmentPos < 0 ? Integer.MAX_VALUE : fragmentPos);

        return url.substring(0, (pathPos < 0 ? 0 : pathPos) + Math.min(suffix.length(), terminalPos));
    }

    @Override
    public void cleanCollections() {
        this.networkCalls.clear();
        this.domainSettings.clear();
    }

    class DomainSettings {
        private final int limit;

        private final String suffix;

        DomainSettings(int limit, String suffix) {
            this.limit = limit;
            this.suffix = suffix;
        }

        public int getLimit() {
            return limit;
        }

        public String getSuffix() {
            return suffix;
        }
    }

    /**
     * Removes the first component of a domain if it contains more than one period. For example:
     * example.com -> example.com
     * www.example.com -> www.example.com
     * api.prod.example.com -> prod.example.com
     *
     * @param domain the domain to trim
     * @return the trimmed domain
     */
    static String trimDomain(String domain) {
        String[] parts = domain.split(".");
        if (parts.length > 2) {
            // Contains more than one period, remove the first part
            return J8Arrays.stream(parts)
                    .skip(1)
                    .collect(Collectors.joining("."));
        }
        return domain;
    }

    /**
     * Gets the host of a URL.
     *
     * @param url the URL
     * @return the hostname or IP address
     */
    static Optional<String> getDomain(String url) {

        // This is necessary for the "new URL(url)" logic.
        if (!url.startsWith("http")) url = String.format("http://%s", url);

        Pattern pattern = Pattern.compile(String.format("%s|%s|%s",
                DNS_PATTERN, IPV4_PATTERN, IPV6_PATTERN));
        Matcher matcher;
        try {
            matcher = pattern.matcher(new URL(url).getHost());
        } catch (MalformedURLException ignored) {
            matcher = pattern.matcher(url);
        }

        if (matcher.find()) {
            return Optional.fromNullable(matcher.group(0));
        } else {
            return Optional.absent();
        }
    }

    /**
     * Tests whether a hostname is an IP address
     *
     * @param domain the hostname to test
     * @return true if the domain is an IP address, false otherwise
     */
    static boolean isIpAddress(String domain) {
        return Pattern.compile(String.format("%s|%s", IPV4_PATTERN, IPV6_PATTERN)).matcher(domain).find();
    }
}