package io.embrace.android.embracesdk.network;

import android.net.Uri;
import android.webkit.URLUtil;

import java.util.EnumSet;
import java.util.Locale;

import io.embrace.android.embracesdk.EmbraceLogger;
import io.embrace.android.embracesdk.network.http.HttpMethod;

/**
 * This class is used to create manually-recorded network requests.
 */
public class EmbraceNetworkRequest {

    private static final int TRACE_ID_MAXIMUM_ALLOWED_LENGTH = 64;

    private static EnumSet<HttpMethod> allowedMethods = EnumSet.of(
            HttpMethod.GET,
            HttpMethod.PUT,
            HttpMethod.POST,
            HttpMethod.DELETE,
            HttpMethod.PATCH);

    /**
     * The request's URL. Must start with http:// or https://
     */
    private String url;
    /**
     * The request's method. Must be one of the following: GET, PUT, POST, DELETE, PATCH.
     */
    private HttpMethod httpMethod;
    /**
     * The time the request started.
     */
    private Long startTime;
    /**
     * The time the request ended. Must be greater than the startTime.
     */
    private Long endTime;
    /**
     * The number of bytes received.
     */
    private Long bytesIn;
    /**
     * The number of bytes received.
     */
    private Long bytesOut;
    /**
     * The response status of the request. Must be in the range 100 to 599.
     */
    private Integer responseCode;
    /**
     * Error that describes a non-HTTP error, e.g. a connection error.
     */
    private Throwable error;
    /**
     * Optional trace ID that can be used to trace a particular request. Max length is 64 characters.
     */
    private String traceId;

    private EmbraceNetworkRequest(Builder builder) {
        this.url = builder.urlString;
        this.httpMethod = builder.httpMethod;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.bytesIn = builder.bytesIn;
        this.bytesOut = builder.bytesOut;
        this.responseCode = builder.responseCode;
        this.error = builder.error;
        this.traceId = builder.traceId;
    }

    private static boolean validateMethod(HttpMethod method) {

        if (method == null) {
            EmbraceLogger.logError("Method cannot be null");
            return false;
        }

        if (!allowedMethods.contains(method)) {
            EmbraceLogger.logError(String.format("Not a valid method: %s", method.name()));
            return false;
        }

        return true;
    }

    public boolean canSend() {
        if (this.url == null) {
            EmbraceLogger.logError("Request must contain URL");
            return false;
        }
        if (this.httpMethod == null) {
            EmbraceLogger.logError("Request must contain method");
            return false;
        }
        if (this.startTime == null) {
            EmbraceLogger.logError("Request must contain startTime");
            return false;
        }
        if (this.endTime == null) {
            EmbraceLogger.logError("Request must contain endTime");
            return false;
        }
        if ((this.responseCode == null || this.responseCode == -1) && this.error == null) {
            EmbraceLogger.logError("Request must either have responseCode or error set");
            return false;
        }

        return true;
    }

    public String getUrl() {
        return url;
    }

    public String getHttpMethod() {
        return httpMethod != null ? httpMethod.name().toUpperCase() : null;
    }

    public Long getStartTime() {
        return startTime;
    }

    public Long getEndTime() {
        return endTime;
    }

    public Long getBytesIn() {
        return bytesIn == null ? 0 : bytesIn;
    }

    public Long getBytesOut() {
        return bytesOut == null ? 0 : bytesOut;
    }

    public Integer getResponseCode() {
        return responseCode;
    }

    public Throwable getError() {
        return error;
    }

    public String getTraceId() {
        return traceId;
    }

    public String description() {
        return String.format("<%s: %s URL = %s Method = %s Start = %s>",
                this,
                this,
                this.url,
                this.httpMethod,
                this.startTime);
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static final class Builder {
        private String urlString;
        private HttpMethod httpMethod;
        private Long startTime;
        private Long endTime;
        private Long bytesIn;
        private Long bytesOut;
        private Integer responseCode;
        private Throwable error;
        private String traceId;

        private Builder() {

        }

        public void withUrl(String urlString) {
            this.urlString = urlString;
        }

        public void withHttpMethod(HttpMethod httpMethod) {
            this.httpMethod = httpMethod;
        }

        public void withStartTime(Long startTime) {
            this.startTime = startTime;
        }

        public void withEndTime(Long date) {

            if (date == null) {
                EmbraceLogger.logError("End time cannot be null");
                return;
            }

            if (date <= this.startTime) {
                EmbraceLogger.logError("End time cannot be before the start time");
                return;
            }

            this.endTime = date;
        }

        @Deprecated
        public void withByteIn(Long bytesIn) {

            withBytesIn(bytesIn);
        }

        public void withBytesIn(Long bytesIn) {

            if (bytesIn == null) {
                EmbraceLogger.logError("BytesIn cannot be null");
                return;
            }

            if (bytesIn < 0) {
                EmbraceLogger.logError("BytesIn must be a positive long");
                return;
            }

            this.bytesIn = bytesIn;
        }

        @Deprecated
        public void withByteOut(Long bytesOut) {

            withBytesOut(bytesOut);
        }

        public void withBytesOut(Long bytesOut) {

            if (bytesOut == null) {
                EmbraceLogger.logError("BytesOut cannot be null");
                return;
            }

            if (bytesOut < 0) {
                EmbraceLogger.logError("BytesOut must be a positive long");
                return;
            }

            this.bytesOut = bytesOut;
        }

        public void withResponseCode(Integer responseCode) {

            if (responseCode < 100 || responseCode > 599) {
                EmbraceLogger.logError(String.format(Locale.getDefault(), "Invalid responseCode: %d", responseCode));
                return;
            }

            this.responseCode = responseCode;
        }

        public void withError(Throwable error) {

            if (error == null) {
                EmbraceLogger.logError("Ignoring null error");
                return;
            }

            this.error = error;
        }

        public void withTraceId(String traceId) {

            if (traceId == null) {
                EmbraceLogger.logError("Ignoring null traceId");
                return;
            }

            if (traceId.length() > TRACE_ID_MAXIMUM_ALLOWED_LENGTH) {
                EmbraceLogger.logError(String.format("Truncating traceId to %s characters", traceId));
                this.traceId = traceId.substring(0, TRACE_ID_MAXIMUM_ALLOWED_LENGTH);
            } else {
                this.traceId = traceId;
            }
        }

        public EmbraceNetworkRequest build() {

            Uri url;
            try {
                url = Uri.parse(urlString);
            } catch (NullPointerException ex) {
                EmbraceLogger.logError(String.format("Invalid URL: %s", urlString));
                return null;
            }

            if (url == null || url.getScheme() == null || url.getHost() == null) {
                EmbraceLogger.logError(String.format("Invalid URL: %s", urlString));
                return null;
            }

            if (!URLUtil.isHttpsUrl(urlString) && !URLUtil.isHttpUrl(urlString)) {
                EmbraceLogger.logError(String.format("Only http and https schemes are supported: %s", urlString));
                return null;
            }

            if (!validateMethod(httpMethod)) {
                return null;
            }

            if (startTime == null) {
                EmbraceLogger.logError("Start time cannot be null");
                return null;
            }

            return new EmbraceNetworkRequest(this);
        }
    }
}
